<?php

namespace BitApps\BTCBI_PRO\Triggers\FluentCart;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Flow\Flow;

final class FluentCartController
{
    public static function info()
    {
        return [
            'name'              => 'FluentCart',
            'type'              => 'custom_form_submission',
            'is_active'         => self::isPluginInstalled(),
            'documentation_url' => 'https://bit-integrations.com/wp-docs/trigger/fluentcart-integrations/',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'fluent-cart/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public function getAllTasks()
    {
        if (!self::isPluginInstalled()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'FluentCart'));
        }

        wp_send_json_success(StaticData::tasks());
    }

    public static function productPurchased($data)
    {
        if (empty($data['cart']) || empty($data['order'])) {
            return;
        }

        $data['cart'] = $data['cart']->toArray();
        $data['order'] = $data['order']->toArray();

        return self::flowExecute('fluent_cart/checkout/prepare_other_data', $data);
    }

    public static function productUpdated($data)
    {
        if (empty($data['product'])) {
            return;
        }

        $data['product'] = $data['product']->toArray();

        return self::flowExecute('fluent_cart/product_updated', $data);
    }

    public static function cartUpdated($data)
    {
        if (empty($data['cart'])) {
            return;
        }

        $data['cart'] = $data['cart']->toArray();

        return self::flowExecute('fluent_cart/checkout/cart_amount_updated', $data);
    }

    public static function orderRefunded($data)
    {
        return self::executeOrderPaymentTrigger('fluent_cart/order_refunded', $data);
    }

    public static function orderPartiallyRefunded($data)
    {
        return self::executeOrderPaymentTrigger('fluent_cart/order_partially_refunded', $data);
    }

    public static function orderFullyRefunded($data)
    {
        return self::executeOrderPaymentTrigger('fluent_cart/order_fully_refunded', $data);
    }

    public static function orderCancelled($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/order_status_changed_to_canceled', $data);
    }

    public static function orderStatusChanged($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/order_status_changed', $data);
    }

    public static function orderStatusUpdatedToProcessing($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/order_status_changed_to_processing', $data);
    }

    public static function orderStatusUpdatedToCompleted($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/order_status_changed_to_completed', $data);
    }

    public static function orderStatusUpdatedToOnHold($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/order_status_changed_to_on-hold', $data);
    }

    public static function orderShippingStatusChanged($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/shipping_status_changed', $data);
    }

    public static function orderShipped($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/shipping_status_changed_to_shipped', $data);
    }

    public static function orderUnshipped($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/shipping_status_changed_to_unshipped', $data);
    }

    public static function orderUnshippable($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/shipping_status_changed_to_unshippable', $data);
    }

    public static function orderDelivered($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/shipping_status_changed_to_delivered', $data);
    }

    public static function orderMarkedAsPaid($data)
    {
        return self::executeOrderStatusTrigger('fluent_cart/order_paid_done', $data);
    }

    /**
     * Execute order status related triggers.
     *
     * @param string $entityId
     * @param array  $data
     */
    private static function executeOrderStatusTrigger($entityId, $data)
    {
        if (empty($data['order'])) {
            return;
        }

        $data['order'] = $data['order']->toArray();

        return self::flowExecute($entityId, $data);
    }

    /**
     * Execute order payment related triggers.
     *
     * @param string $entityId
     * @param array  $data
     */
    private static function executeOrderPaymentTrigger($entityId, $data)
    {
        if (empty($data['order']) || empty($data['customer']) || empty($data['transaction'])) {
            return;
        }

        $data['order'] = $data['order']->toArray();
        $data['customer'] = $data['customer']->toArray();
        $data['transaction'] = $data['transaction']->toArray();

        return self::flowExecute($entityId, $data);
    }

    private static function flowExecute($entityId, $data)
    {
        $formData = Helper::prepareFetchFormatFields($data);

        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        $flows = Flow::exists('FluentCart', $entityId);

        if (!$flows) {
            return;
        }

        $data = array_column($formData, 'value', 'name');

        Flow::execute('FluentCart', $entityId, $data, $flows);

        return ['type' => 'success'];
    }

    private static function isPluginInstalled()
    {
        return \defined('FLUENTCART_PLUGIN_PATH');
    }
}
