<?php

namespace BitApps\BTCBI_PRO\Triggers\FormGent;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Flow\Flow;

final class FormGentController
{
    public static function info()
    {
        return [
            'name'              => 'FormGent',
            'type'              => 'custom_form_submission',
            'is_active'         => self::isPluginInstalled(),
            'documentation_url' => 'https://bit-integrations.com/wp-docs/trigger/formgent-integrations/',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'formgent/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public function getAllTasks()
    {
        if (!self::isPluginInstalled()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'FormGent'));
        }

        wp_send_json_success(StaticData::forms());
    }

    public static function handleFormSubmission($responseId, $form, $request)
    {
        $formData = $request->get_param('form_data') ?? [];

        $formData['form_id'] = $form->ID ?? null;
        $formData['response_id'] = $responseId;

        if (empty($formData['form_id'])) {
            return;
        }

        $formData = Helper::prepareFetchFormatFields($formData);

        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData('btcbi_formgent_after_create_form_response_test', array_values($formData), 'form_id.value', $form->ID);

        $flows = Flow::exists('FormGent', 'formgent_after_create_form_response');

        if (!$flows) {
            return;
        }

        $data = array_column($formData, 'value', 'name');

        foreach ($flows as $flow) {
            $flowDetails = Helper::parseFlowDetails($flow->flow_details);

            if (
                !isset($flowDetails->primaryKey)
                || !Helper::isPrimaryKeysMatch($formData, $flowDetails->primaryKey)
            ) {
                continue;
            }

            Flow::execute('FormGent', 'formgent_after_create_form_response', $data, [$flow]);
        }

        return ['type' => 'success'];
    }

    public static function handleFormCreated($FormDTO, $request)
    {
        $data = self::prepareFormData($FormDTO, $request);

        return self::flowExecute('formgent_after_create_form', $data);
    }

    public static function handleFormTitleUpdated($request)
    {
        return self::flowExecute(
            'formgent_after_update_form_title',
            $request->get_params() ?? []
        );
    }

    public static function handleFormStatusUpdated($request)
    {
        return self::flowExecute(
            'formgent_after_update_form_status',
            $request->get_params() ?? []
        );
    }

    public static function handleFormDuplicated($FormDTO, $request)
    {
        $data = self::prepareFormData($FormDTO, $request);

        return self::flowExecute('formgent_after_duplicate_form', $data);
    }

    public static function handleFormDeleted($FormId, $form)
    {
        return self::flowExecute('formgent_after_delete_form', (array) $form);
    }

    public static function handleFormResponsesDeleted($responseIds, $request)
    {
        $data = $request->get_params() ?? [];

        $data['response_ids'] = wp_json_encode($responseIds);

        return self::flowExecute('formgent_after_delete_responses', $data);
    }

    public static function handleFormResponsesStarred($request)
    {
        return self::flowExecute(
            'formgent_after_update_response_starred',
            $request->get_params() ?? []
        );
    }

    public static function handleFormResponsesRead($request)
    {
        return self::flowExecute(
            'formgent_after_update_response_read',
            $request->get_params() ?? []
        );
    }

    public static function handleQuizResult($request, $result)
    {
        $data = array_merge($request->get_params() ?? [], $result);

        return self::flowExecute('formgent_after_quiz_result', $data);
    }

    private static function prepareFormData($FormDTO, $request)
    {
        $data = $request->get_params() ?? [];

        $data['form_id'] = $FormDTO->get_id() ?? null;
        $data['form_title'] = $FormDTO->get_title() ?? null;
        $data['form_status'] = $FormDTO->get_status() ?? null;
        $data['form_type'] = $FormDTO->get_type() ?? null;
        $data['form_content'] = (string) $FormDTO->get_content() ?? null;
        $data['form_fields'] = $FormDTO->get_fields() ?? null;
        $data['is_save_incomplete_data'] = $FormDTO->is_save_incomplete_data() ?? null;

        return Helper::prepareFetchFormatFields($data);
    }

    private static function flowExecute($entityId, $data)
    {
        $formData = Helper::prepareFetchFormatFields($data);

        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        $flows = Flow::exists('FormGent', $entityId);

        if (!$flows) {
            return;
        }

        $data = array_column($formData, 'value', 'name');

        Flow::execute('FormGent', $entityId, $data, $flows);

        return ['type' => 'success'];
    }

    private static function isPluginInstalled()
    {
        return class_exists('FormGent');
    }
}
