<?php

namespace BitApps\BTCBI_PRO\Triggers\GiveWp;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Flow\Flow;
use Give_Payment;
use Give_Subscription;

final class GiveWpController
{
    private const DEPRECATED_USER_DONATION = 1;

    private const DEPRECATED_SUBSCRIPTION_DONATION = 2;

    private const DEPRECATED_RECURRING_DONATION = 3;

    public static function info()
    {
        return [
            'name'              => 'GiveWP',
            'type'              => 'custom_form_submission',
            'is_active'         => self::pluginActive(),
            'documentation_url' => 'https://bit-integrations.com/wp-docs/trigger/givewp-integrations-as-a-trigger/',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'givewp/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public static function pluginActive()
    {
        return class_exists('Give');
    }

    public function getAll()
    {
        if (!self::pluginActive()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'GiveWp'));
        }

        wp_send_json_success([
            [
                'form_name'           => __('Donation Form Submission', 'bit-integrations-pro'),
                'triggered_entity_id' => 'btcbi_give_donation_form_submission',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Recurring Donation Cancelled By Form', 'bit-integrations-pro'),
                'triggered_entity_id' => 'btcbi_give_recurring_donation_cancelled',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('User Continues Recurring Donation', 'bit-integrations-pro'),
                'triggered_entity_id' => 'btcbi_give_user_continues_recurring_donation',
                'skipPrimaryKey'      => true
            ]
        ]);
    }

    public static function handleUserDonation($paymentId, $status, $oldStatus)
    {
        if ('publish' !== $status || !self::pluginActive()) {
            return;
        }

        if ($flows = Flow::exists('GiveWp', self::DEPRECATED_USER_DONATION)) {
            GiveWpFallback::handleUserDonation($paymentId, $flows);
        }

        if (!\function_exists('give_get_donor_donation_comment')) {
            return;
        }

        $payment = new Give_Payment($paymentId);

        $data = [
            'first_name'        => $payment->first_name ?? '',
            'last_name'         => $payment->last_name ?? '',
            'email'             => $payment->email ?? '',
            'currency'          => $payment->currency ?? '',
            'donated_amount'    => $payment->subtotal ?? '',
            'donation_amount'   => $payment->subtotal ?? '',
            'form_id'           => $payment->form_id ?? '',
            'form_title'        => $payment->form_title ?? '',
            'name_title_prefix' => $payment->title_prefix ?? '',
            'date'              => $payment->date ?? '',
        ];

        if (\is_array($payment->address)) {
            $data['address1'] = $payment->address['line1'] ?? '';
            $data['address2'] = $payment->address['line2'] ?? '';
            $data['city'] = $payment->address['city'] ?? '';
            $data['state'] = $payment->address['state'] ?? '';
            $data['zip'] = $payment->address['zip'] ?? '';
            $data['country'] = $payment->address['country'] ?? '';
        }

        $meta = $payment->get_meta() ?? [];

        if (\is_array($meta) && isset($meta['user_info'])) {
            $data = array_merge($data, $meta['user_info']);
            unset($meta['user_info']);
        }

        $data['payment_meta'] = $meta;

        $comment = (array) give_get_donor_donation_comment($paymentId, $payment->donor_id);

        $data['donation_comment'] = $comment['comment_content'] ?? '';

        self::executeFlow('btcbi_give_donation_form_submission', $data);
    }

    public static function handleSubscriptionDonationCancel($subscriptionId, $subscription)
    {
        if ($flows = Flow::exists('GiveWp', self::DEPRECATED_SUBSCRIPTION_DONATION)) {
            return GiveWpFallback::handleSubscriptionDonationCancel($subscriptionId, $subscription, $flows);
        }

        $data = (array) $subscription;

        if (isset($subscription->donor->user_id)) {
            $data = array_merge($data, GiveWpHelper::getUserInfo($subscription->donor->user_id));
        }

        self::executeFlow('btcbi_give_recurring_donation_cancelled', $data);
    }

    public static function handleRecurringDonation($status, $rowId, $data, $where)
    {
        if ($flows = Flow::exists('GiveWp', self::DEPRECATED_RECURRING_DONATION)) {
            return GiveWpFallback::handleRecurringDonation($rowId, $data, $flows);
        }

        if (!class_exists('Give_Subscription')) {
            return;
        }

        $subscription = new Give_Subscription($rowId);

        $data = (array) $subscription;

        $data['total_payment'] = $subscription->get_total_payments();

        if (isset($subscription->donor->user_id)) {
            $data = array_merge($data, GiveWpHelper::getUserInfo($subscription->donor->user_id));
        }

        self::executeFlow('btcbi_give_user_continues_recurring_donation', $data);
    }

    public static function all_donation_form()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'posts';
        $donationForm = $wpdb->get_results("SELECT ID, post_title FROM {$table_name} WHERE post_type = 'give_forms' AND post_status = 'publish'");

        wp_send_json_success($donationForm);
    }

    private static function executeFlow(string $entityId, array $data): void
    {
        $formData = Helper::prepareFetchFormatFields($data);

        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        $flows = Flow::exists('GiveWp', $entityId);

        if (empty($flows)) {
            return;
        }

        $data = array_column($formData, 'value', 'name');

        Flow::execute('GiveWp', $entityId, $data, $flows);
    }
}
