<?php

namespace BitApps\BTCBI_PRO\Triggers\MailMint;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Flow\Flow;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Models\ContactModel;

/**
 * MailMint Controller for handling triggers and integrations
 */
final class MailMintController
{
    private const PLUGIN_NAME = 'Mail Mint';

    private const PLUGIN_CONSTANT = 'MAILMINT';

    private const ENTITY_FORM_SUBMIT = 'mailmint_after_form_submit';

    private const ENTITY_CONTACT_CREATED = 'mint_after_contact_creation';

    private const ENTITY_CONTACT_DELETED = 'mailmint_after_delete_contact';

    private const ENTITY_TAG_APPLIED = 'mailmint_tag_applied';

    private const ENTITY_LIST_APPLIED = 'mailmint_list_applied';

    private const ENTITY_CAMPAIGN_CREATED = 'mailmint_campaign_created';

    private const ENTITY_CAMPAIGN_EMAIL_SENT = 'mailmint_after_campaign_start';

    private const SUCCESS_RESPONSE = ['type' => 'success'];

    /**
     * Get plugin information and configuration
     *
     * @return array Plugin configuration array
     */
    public static function info(): array
    {
        return [
            'name'              => self::PLUGIN_NAME,
            'type'              => 'custom_form_submission',
            'is_active'         => self::isPluginInstalled(),
            'documentation_url' => 'https://bit-integrations.com/wp-docs/trigger/mail-mint/',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'mail_mint/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    /**
     * Get all available tasks/triggers for MailMint
     *
     * @return void Sends JSON response
     */
    public function getAllTasks(): void
    {
        if (!self::isPluginInstalled()) {
            wp_send_json_error(
                \sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), self::PLUGIN_NAME)
            );
        }

        $tasks = [
            [
                'form_name'           => __('Form Submission', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_FORM_SUBMIT,
                'skipPrimaryKey'      => false
            ],
            [
                'form_name'           => __('Contact Created', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_CONTACT_CREATED,
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Contact Deleted', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_CONTACT_DELETED,
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Tags Added To Contact', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_TAG_APPLIED,
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Lists Added To Contact', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_LIST_APPLIED,
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Campaign Created', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_CAMPAIGN_CREATED,
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Campaign Email Sent', 'bit-integrations-pro'),
                'triggered_entity_id' => self::ENTITY_CAMPAIGN_EMAIL_SENT,
                'skipPrimaryKey'      => true
            ],
        ];

        wp_send_json_success($tasks);
    }

    /**
     * Handle form submission trigger
     *
     * @param int $formId    The form ID
     * @param int $contactId The contact ID
     *
     * @return array|null Success response or null on failure
     */
    public static function handleFormSubmission($formId, $contactId): ?array
    {
        if (!self::isValidFormSubmission($formId, $contactId)) {
            return null;
        }

        $formData = Helper::prepareFetchFormatFields(
            array_merge(
                ['form_id' => $formId],
                ContactModel::get($contactId)
            )
        );

        $entityId = self::ENTITY_FORM_SUBMIT;

        Helper::setTestData(
            "btcbi_{$entityId}_test",
            array_values($formData),
            'form_id.value',
            $formId
        );

        $flows = Flow::exists('MailMint', $entityId);
        if (!$flows) {
            return null;
        }

        $data = array_column($formData, 'value', 'name');

        foreach ($flows as $flow) {
            $flowDetails = Helper::parseFlowDetails($flow->flow_details);

            if (!self::isValidFlowDetails($flowDetails, $formData)) {
                continue;
            }

            Flow::execute('MailMint', $entityId, $data, [$flow]);
        }

        return self::SUCCESS_RESPONSE;
    }

    /**
     * Handle contact created trigger
     *
     * @param int $contactId The contact ID
     *
     * @return array|null Success response or null on failure
     */
    public static function handleContactCreated($contactId): ?array
    {
        if (!self::isValidContactId($contactId)) {
            return null;
        }

        $contactData = ContactModel::get($contactId);

        if (empty($contactData)) {
            return null;
        }

        return static::flowExecute(self::ENTITY_CONTACT_CREATED, $contactData);
    }

    /**
     * Handle contact deleted trigger
     *
     * @param int $contactId The contact ID
     *
     * @return array|null Success response or null on failure
     */
    public static function handleContactDeleted($contactId): ?array
    {
        if (!class_exists('Mint\MRM\DataBase\Models\ContactModel')) {
            return null;
        }

        $contactData = ContactModel::get($contactId);

        if (empty($contactData)) {
            return null;
        }

        return static::flowExecute(self::ENTITY_CONTACT_DELETED, $contactData);
    }

    /**
     * Handle tags added to contact trigger
     *
     * @param array $tags      Array of tags
     * @param int   $contactId The contact ID
     *
     * @return array|null Success response or null on failure
     */
    public static function handleTagsAddedToContact(array $tags, $contactId): ?array
    {
        if (!self::isValidTagsOrListsData($tags, $contactId)) {
            return null;
        }

        $contactData = ContactModel::get($contactId);

        if (empty($contactData)) {
            return null;
        }

        $flows = Flow::exists('MailMint', self::ENTITY_TAG_APPLIED);

        foreach ($tags as $tag) {
            if (isset($tag['created_at']) || empty($tag['id'])) {
                continue;
            }

            $data = [
                'contact' => $contactData,
                'tag'     => ContactGroupModel::get($tag['id'])
            ];

            static::flowExecute(self::ENTITY_TAG_APPLIED, $data, $flows);
        }

        return self::SUCCESS_RESPONSE;
    }

    /**
     * Handle lists added to contact trigger
     *
     * @param array $lists     Array of lists
     * @param int   $contactId The contact ID
     *
     * @return array|null Success response or null on failure
     */
    public static function handleListsAddedToContact(array $lists, $contactId): ?array
    {
        if (!self::isValidTagsOrListsData($lists, $contactId)) {
            return null;
        }

        $contactData = ContactModel::get($contactId);

        if (empty($contactData)) {
            return null;
        }

        $flows = Flow::exists('MailMint', self::ENTITY_LIST_APPLIED);

        foreach ($lists as $list) {
            if (isset($list['created_at']) || empty($list['id'])) {
                continue;
            }

            $data = [
                'contact' => $contactData,
                'list'    => ContactGroupModel::get($list['id'])
            ];

            static::flowExecute(self::ENTITY_LIST_APPLIED, $data, $flows);
        }

        return self::SUCCESS_RESPONSE;
    }

    /**
     * Handle campaign created trigger
     *
     * @param int   $campaignId The campaign ID
     * @param array $data       Campaign data
     *
     * @return array|null Success response or null on failure
     */
    public static function handleCampaignCreated($campaignId, $data): ?array
    {
        $data = array_merge(['campaign_id' => $campaignId], $data);

        return static::flowExecute(self::ENTITY_CAMPAIGN_CREATED, $data);
    }

    /**
     * Handle campaign email sent trigger
     *
     * @param array $data Campaign data
     *
     * @return array|null Success response or null on failure
     */
    public static function handleCampaignEmailSent($data): ?array
    {
        return static::flowExecute(self::ENTITY_CAMPAIGN_EMAIL_SENT, $data);
    }

    /**
     * Execute flow for given entity and data
     *
     * @param string     $entityId The entity ID
     * @param array      $data     The data to process
     * @param null|mixed $flows
     *
     * @return array|null Success response or null on failure
     */
    private static function flowExecute(string $entityId, $data, $flows = null): ?array
    {
        $formData = Helper::prepareFetchFormatFields($data);

        if (empty($formData) || !\is_array($formData)) {
            return null;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        if ($flows === null) {
            $flows = Flow::exists('MailMint', $entityId);
        }

        if (!$flows) {
            return null;
        }

        $data = array_column($formData, 'value', 'name');

        Flow::execute('MailMint', $entityId, $data, $flows);

        return self::SUCCESS_RESPONSE;
    }

    /**
     * Check if MailMint plugin is installed and active
     *
     * @return bool True if plugin is installed and active
     */
    private static function isPluginInstalled(): bool
    {
        return \defined(self::PLUGIN_CONSTANT);
    }

    /**
     * Validate form submission data
     *
     * @param int $formId    The form ID
     * @param int $contactId The contact ID
     *
     * @return bool True if valid
     */
    private static function isValidFormSubmission($formId, $contactId): bool
    {
        return !empty($formId)
            && class_exists('Mint\MRM\DataBase\Models\ContactModel')
            && ContactModel::is_contact_ids_exists([$contactId]);
    }

    /**
     * Validate contact ID
     *
     * @param int $contactId The contact ID
     *
     * @return bool True if valid
     */
    private static function isValidContactId($contactId): bool
    {
        return class_exists('Mint\MRM\DataBase\Models\ContactModel')
            && ContactModel::is_contact_ids_exists([$contactId]);
    }

    /**
     * Validate tags or lists data with contact ID
     *
     * @param array $items     Array of tags or lists
     * @param int   $contactId The contact ID
     *
     * @return bool True if valid
     */
    private static function isValidTagsOrListsData(array $items, $contactId): bool
    {
        return !empty($items)
            && class_exists('Mint\MRM\DataBase\Models\ContactGroupModel')
            && class_exists('Mint\MRM\DataBase\Models\ContactModel')
            && ContactModel::is_contact_ids_exists([$contactId]);
    }

    /**
     * Validate flow details and primary key matching
     *
     * @param object $flowDetails The flow details object
     * @param array  $formData    The form data
     *
     * @return bool True if valid
     */
    private static function isValidFlowDetails($flowDetails, array $formData): bool
    {
        return isset($flowDetails->primaryKey)
            && Helper::isPrimaryKeysMatch($formData, $flowDetails->primaryKey);
    }
}
