<?php

namespace BitApps\BTCBI_PRO\Triggers\PaidMembershipPro;

use BitCode\FI\Flow\Flow;

final class PaidMembershipProController
{
    private const ASSIGN_MEMBERSHIP_LEVEL = 1;

    private const CANCEL_MEMBERSHIP_LEVEL = 2;

    private const PURCHASE_MEMBERSHIP_LEVEL = 3;

    private const EXPIRY_MEMBERSHIP_LEVEL = 4;

    private const CHANGED_MEMBERSHIP_LEVEL = 5;

    public static function info()
    {
        $plugin_path = self::pluginActive('get_name');

        return [
            'name'           => 'Paid Membership Pro',
            'title'          => __('PaidMembershipPro', 'bit-integrations-pro'),
            'slug'           => $plugin_path,
            'pro'            => $plugin_path,
            'type'           => 'form',
            'is_active'      => is_plugin_active($plugin_path),
            'activation_url' => wp_nonce_url(self_admin_url('plugins.php?action=activate&amp;plugin=' . $plugin_path . '&amp;plugin_status=all&amp;paged=1&amp;s'), 'activate-plugin_' . $plugin_path),
            'install_url'    => wp_nonce_url(self_admin_url('update.php?action=install-plugin&plugin=' . $plugin_path), 'install-plugin_' . $plugin_path),
            'list'           => [
                'action' => 'paidmembershippro/get',
                'method' => 'get',
            ],
            'fields' => [
                'action' => 'paidmembershippro/get/form',
                'method' => 'post',
                'data'   => ['id']
            ],
            'isPro' => true
        ];
    }

    public static function pluginActive($option = null)
    {
        if (is_plugin_active('paid-memberships-pro/paid-memberships-pro.php')) {
            return $option === 'get_name' ? 'paid-memberships-pro/paid-memberships-pro.php' : true;
        }

        return false;
    }

    public function getAll()
    {
        if (!self::pluginActive()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'Paid Membership Pro'));
        }

        $types = [
            [
                'id'    => self::ASSIGN_MEMBERSHIP_LEVEL,
                'title' => __('An admin assigns a membership level to a user', 'bit-integrations-pro')
            ],
            [
                'id'    => self::CANCEL_MEMBERSHIP_LEVEL,
                'title' => __('User cancels a membership', 'bit-integrations-pro')
            ],
            [
                'id'    => self::PURCHASE_MEMBERSHIP_LEVEL,
                'title' => __('User purchases a membership', 'bit-integrations-pro')
            ],
            [
                'id'    => self::EXPIRY_MEMBERSHIP_LEVEL,
                'title' => __('User\'s subscription to a membership expires', 'bit-integrations-pro')
            ],
            [
                'id'    => self::CHANGED_MEMBERSHIP_LEVEL,
                'title' => __('User membership level changed', 'bit-integrations-pro')
            ]
        ];

        wp_send_json_success($types);
    }

    public function get_a_form($data)
    {
        if (!self::pluginActive()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'Paid Membership Pro'));
        }

        if (empty($data->id)) {
            wp_send_json_error(__('Trigger type doesn\'t exists', 'bit-integrations-pro'));
        }

        $fields = self::fields($data->id);

        if (empty($fields)) {
            wp_send_json_error(__('Trigger doesn\'t exists any field', 'bit-integrations-pro'));
        }

        if ($data->id != self::CHANGED_MEMBERSHIP_LEVEL) {
            $responseData['AllMembershipLevels'] = self::allMemberships();
        }

        $responseData['fields'] = $fields;
        wp_send_json_success($responseData);
    }

    public static function fields($id)
    {
        if (empty($id)) {
            wp_send_json_error(
                __(
                    'Requested parameter is empty',
                    'bit-integrations-pro'
                ),
                400
            );
        }

        $membershipFields = PaidMembershipProHelper::getPaidMembershipProField();
        $userFields = PaidMembershipProHelper::getUserField();
        $fields = array_merge($membershipFields, $userFields);

        foreach ($fields as $field) {
            $fieldsNew[] = [
                'name'  => $field->fieldKey,
                'type'  => 'text',
                'label' => $field->fieldName,
            ];
        }

        return $fieldsNew;
    }

    public static function assignMembershipLabelToUser($levelId, $userId)
    {
        if (empty($levelId)) {
            return;
        }

        $flows = Flow::exists('PaidMembershipPro', self::ASSIGN_MEMBERSHIP_LEVEL);

        if (empty($flows)) {
            return;
        }

        $flowDetails = json_decode($flows[0]->flow_details);
        $selectedMembershipLevel = !empty($flowDetails->selectedMembershipLevel) ? $flowDetails->selectedMembershipLevel : [];

        if ($levelId === $selectedMembershipLevel || $selectedMembershipLevel === 'any') {
            $data = self::getLevelCombineData($userId, $levelId);

            Flow::execute('PaidMembershipPro', self::ASSIGN_MEMBERSHIP_LEVEL, $data, $flows);
        }
    }

    public static function cancelMembershipLevel($levelId, $userId, $cancelLevelId)
    {
        if (empty($levelId)) {
            return;
        }

        $flows = Flow::exists('PaidMembershipPro', self::CANCEL_MEMBERSHIP_LEVEL);

        if (empty($flows)) {
            return;
        }

        $flowDetails = json_decode($flows[0]->flow_details);
        $selectedMembershipLevel = !empty($flowDetails->selectedMembershipLevel) ? $flowDetails->selectedMembershipLevel : [];

        if ($cancelLevelId == $selectedMembershipLevel || $selectedMembershipLevel === 'any') {
            $data = self::getLevelCombineData($userId, $cancelLevelId);
            Flow::execute('PaidMembershipPro', self::CANCEL_MEMBERSHIP_LEVEL, $data, $flows);
        }
    }

    public static function checkoutMembershipLevel($userId, $morder)
    {
        $user = $morder->getUser();
        $membership = $morder->getMembershipLevel();
        $userId = $user->ID;
        $membershipId = $membership->id;

        $flows = Flow::exists('PaidMembershipPro', self::PURCHASE_MEMBERSHIP_LEVEL);

        if (empty($flows)) {
            return;
        }

        $flowDetails = json_decode($flows[0]->flow_details);
        $selectedMembershipLevel = !empty($flowDetails->selectedMembershipLevel) ? $flowDetails->selectedMembershipLevel : [];

        if ($membershipId == $selectedMembershipLevel || $selectedMembershipLevel === 'any') {
            $data = self::getLevelCombineData($userId, $membershipId);

            Flow::execute('PaidMembershipPro', self::PURCHASE_MEMBERSHIP_LEVEL, $data, $flows);
        }
    }

    public static function expiryMembershipLevel($userId, $membershipId)
    {
        $flows = Flow::exists('PaidMembershipPro', self::EXPIRY_MEMBERSHIP_LEVEL);

        if (empty($flows)) {
            return;
        }

        $flowDetails = json_decode($flows[0]->flow_details);
        $selectedMembershipLevel = !empty($flowDetails->selectedMembershipLevel) ? $flowDetails->selectedMembershipLevel : [];

        if (($membershipId == $selectedMembershipLevel || $selectedMembershipLevel === 'any')) {
            $data = self::getLevelCombineData($userId, $membershipId);

            Flow::execute('PaidMembershipPro', self::EXPIRY_MEMBERSHIP_LEVEL, $data, $flows);
        }
    }

    public static function membershipLevelChanged($membership)
    {
        if (empty($membership) || !\function_exists('pmpro_getMembershipLevelsForUser')) {
            return;
        }

        $userId = key($membership);
        $userData = self::getUserInfo($userId);
        $levelData = pmpro_getMembershipLevelsForUser($userId);

        $flows = Flow::exists('PaidMembershipPro', self::CHANGED_MEMBERSHIP_LEVEL);

        if (empty($flows)) {
            return;
        }

        $finalData = array_merge($userData, (array) reset($levelData));

        Flow::execute('PaidMembershipPro', self::CHANGED_MEMBERSHIP_LEVEL, $finalData, $flows);
    }

    public static function allMemberships()
    {
        global $wpdb;
        $levels = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->pmpro_membership_levels} ORDER BY id ASC"
            )
        );

        $allLevels = [
            [
                'membershipId'    => 'any',
                'membershipTitle' => __('Any Membership Level', 'bit-integrations-pro')
            ]
        ];

        if (empty($levels)) {
            return $allLevels;
        }

        foreach ($levels as $level) {
            $allLevels[] = [
                'membershipId'    => $level->id,
                'membershipTitle' => $level->name,
            ];
        }

        return $allLevels;
    }

    public static function getAllPaidMembershipProLevel()
    {
        wp_send_json_success(self::allMemberships());
    }

    private static function getUserInfo($userId)
    {
        $userInfo = get_userdata($userId);
        $user = [];
        if ($userInfo) {
            $userData = $userInfo->data;
            $user_meta = get_user_meta($userId);
            $user = [
                'user_id'    => $userId,
                'first_name' => $user_meta['first_name'][0],
                'last_name'  => $user_meta['last_name'][0],
                'user_email' => $userData->user_email,
                'nickname'   => $userData->user_nicename,
                'avatar_url' => get_avatar_url($userId),
            ];
        }

        return $user;
    }

    private static function getLevelInfo($id)
    {
        global $wpdb;

        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->pmpro_membership_levels} WHERE id = %d",
                $id
            )
        );
    }

    private static function getLevelCombineData($userId, $levelId)
    {
        $userData = self::getUserInfo($userId);
        $levelData = self::getLevelInfo($levelId);

        return array_merge($userData, (array) reset($levelData));
    }
}
