<?php

namespace BitApps\BTCBI_PRO\Triggers\StoreEngine;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Flow\Flow;

final class StoreEngineController
{
    public static function info()
    {
        return [
            'name'              => 'StoreEngine',
            'type'              => 'custom_form_submission',
            'is_active'         => StoreEngineHelper::isPluginInstalled(),
            'documentation_url' => 'https://bit-integrations.com/wp-docs/trigger/storeengine-integrations/',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'store_engine/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public function getAllTasks()
    {
        if (!StoreEngineHelper::isPluginInstalled()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'StoreEngine'));
        }

        wp_send_json_success(
            StoreEngineHelper::tasks()
        );
    }

    public static function handleProductPurchased($order, $payload)
    {
        $orderId = $order->get_id();

        $formData = StoreEngineHelper::ProductPurchasedFormatFields(
            $orderId,
            $order,
            null,
            (array) $payload
        );

        return static::flowExecute('storeengine/checkout/after_place_order', $formData);
    }

    public static function handleOrderStatusUpdated($orderId, $oldStatus, $newStatus, $order)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields(
            $orderId,
            $order,
            null,
            [
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
            ]
        );

        return static::flowExecute('storeengine/order/status_changed', $formData);
    }

    public static function handleOrderRestored($orderId, $oldStatus, $newStatus, $order)
    {
        if ($oldStatus !== 'trash' || $newStatus === 'trash') {
            return;
        }

        $formData = StoreEngineHelper::ProductPurchasedFormatFields(
            $orderId,
            $order,
            null,
            [
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
            ]
        );

        return static::flowExecute('storeengine/order_restored', $formData);
    }

    public static function handleOrderStatusCompleted($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_completed', $formData);
    }

    public static function handleOrderStatusOnHold($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_on_hold', $formData);
    }

    public static function handleOrderStatusPendingPayment($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_pending_payment', $formData);
    }

    public static function handleOrderStatusProcessing($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_processing', $formData);
    }

    public static function handleOrderStatusDraft($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_draft', $formData);
    }

    public static function handleOrderStatusCancelled($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_cancelled', $formData);
    }

    public static function handleOrderStatusTrash($orderId, $order, $statusTransition)
    {
        $formData = StoreEngineHelper::ProductPurchasedFormatFields($orderId, $order, $statusTransition);

        return static::flowExecute('storeengine/order_status_trash', $formData);
    }

    public static function handleNewCustomerNote($note, $order)
    {
        $orderId = $order->get_id();
        $formData = StoreEngineHelper::ProductPurchasedFormatFields(
            $orderId,
            $order,
            null,
            ['note' => $note]
        );

        return static::flowExecute('storeengine/order/new_customer_note', $formData);
    }

    public static function handleNoteDeleted($noteId, $note)
    {
        $formData = Helper::prepareFetchFormatFields((array) $note);

        return static::flowExecute('storeengine/order/note_deleted', $formData);
    }

    private static function flowExecute($entityId, $formData)
    {
        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        $flows = Flow::exists('StoreEngine', $entityId);

        if (!$flows) {
            return;
        }

        $data = array_column($formData, 'value', 'name');

        Flow::execute('StoreEngine', $entityId, $data, $flows);

        return ['type' => 'success'];
    }
}
