<?php

namespace BitApps\BTCBI_PRO\Triggers\StoreEngine;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Core\Util\User;

class StoreEngineHelper
{
    /**
     * Define available tasks
     */
    public static function tasks()
    {
        return [
            [
                'form_name'           => __('Product Purchased', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/checkout/after_place_order',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Updated', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order/status_changed',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To On Hold', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_on_hold',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To Pending Payment', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_pending_payment',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To Processing', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_processing',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To Completed', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_completed',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To Cancelled', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_cancelled',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To Draft', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_draft',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Status Set To Trash', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_status_trash',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Order Restored', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order_restored',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Customer Note Added to Order', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order/new_customer_note',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Customer Note Deleted From Order', 'bit-integrations-pro'),
                'triggered_entity_id' => 'storeengine/order/note_deleted',
                'skipPrimaryKey'      => true
            ],
        ];
    }

    /**
     * Main entry for formatting purchased product fields
     *
     * @param mixed $orderId
     * @param mixed $order
     * @param mixed $statusTransition
     * @param mixed $additionalData
     */
    public static function ProductPurchasedFormatFields($orderId, $order, $statusTransition = null, $additionalData = [])
    {
        if (!self::isValidOrder($orderId, $order)) {
            return;
        }

        $userId = $order->get_customer_id();
        $items = $order->get_items();

        if (empty($items) || !\is_array($items)) {
            return;
        }

        $productData = [];
        foreach ($items as $item) {
            $productData[] = self::getProductDataFromItem($item, $userId);
        }

        $orderData = array_merge(
            $additionalData,
            [
                'order_id'     => $orderId,
                'order_status' => $order->get_status(),
                'order_total'  => $order->get_total(),
                'currency'     => $order->get_currency(),
            ],
            User::get($userId),
            $productData,
        );

        if (!empty($statusTransition)) {
            $orderData['status_transition'] = $statusTransition;
        }

        return Helper::prepareFetchFormatFields($orderData);
    }

    public static function isPluginInstalled()
    {
        return \defined('STOREENGINE_VERSION');
    }

    /**
     * Validate if the order is usable
     *
     * @param mixed $orderId
     * @param mixed $order
     */
    private static function isValidOrder($orderId, $order): bool
    {
        return !(
            empty($orderId)
            || empty($order)
            || !\is_object($order)
            || !method_exists($order, 'get_items')
            || !method_exists($order, 'get_customer_id')
            || !method_exists($order, 'get_status')
            || !method_exists($order, 'get_total')
            || !method_exists($order, 'get_currency')
        );
    }

    /**
     * Extract product + membership data from a single order item
     *
     * @param mixed $item
     * @param mixed $userId
     */
    private static function getProductDataFromItem($item, $userId): array
    {
        if (
            !\is_object($item)
            || !method_exists($item, 'get_product_id')
            || !method_exists($item, 'get_price_id')
        ) {
            return [];
        }

        $productId = (int) $item->get_product_id();
        if (!$productId) {
            return [];
        }

        $product = get_post($productId);
        if (!$product) {
            return [];
        }

        return array_merge(
            self::mapProductFields($product, $productId),
            self::getCustomMetadata($productId),
            self::getMembershipData($item, $userId)
        );
    }

    /**
     * Map WordPress product post fields
     *
     * @param mixed $product
     */
    private static function mapProductFields($product, int $productId): array
    {
        return [
            'product_id'               => $productId,
            'product_author'           => $product->post_author,
            'product_date'             => $product->post_date,
            'product_date_gmt'         => $product->post_date_gmt,
            'product_content'          => $product->post_content,
            'product_title'            => $product->post_title,
            'product_excerpt'          => $product->post_excerpt,
            'product_status'           => $product->post_status,
            'product_comment_status'   => $product->comment_status,
            'product_ping_status'      => $product->ping_status,
            'product_password'         => $product->post_password,
            'product_name'             => $product->post_name,
            'product_to_ping'          => $product->to_ping,
            'product_pinged'           => $product->pinged,
            'product_modified'         => $product->post_modified,
            'product_modified_gmt'     => $product->post_modified_gmt,
            'product_content_filtered' => $product->post_content_filtered,
            'product_parent'           => $product->post_parent,
            'product_guid'             => $product->guid,
            'product_menu_order'       => $product->menu_order,
            'product_type'             => $product->post_type,
            'product_mime_type'        => $product->post_mime_type,
            'product_comment_count'    => $product->comment_count,
            'filter'                   => 'raw',
            'featured_image'           => get_post_thumbnail_id($productId) ?: null,
            'permalink'                => get_permalink($productId),
        ];
    }

    /**
     * Get product custom metadata
     */
    private static function getCustomMetadata(int $productId): array
    {
        $customMeta = [];

        foreach (get_post_meta($productId) as $metaKey => $metaValue) {
            if (
                \is_string($metaKey)
                && strpos($metaKey, '_storeengine_') === 0
                && !empty($metaValue[0])
            ) {
                $customMeta[ltrim($metaKey, '_')] = maybe_unserialize($metaValue[0]);
            }
        }

        return $customMeta;
    }

    /**
     * Extract membership plan + user membership data
     *
     * @param mixed $item
     * @param mixed $userId
     */
    private static function getMembershipData($item, $userId): array
    {
        $result = [];

        if (!class_exists('\StoreEngine\Utils\Helper')) {
            return $result;
        }

        $membership = \StoreEngine\Utils\Helper::get_integration_repository_by_price_id(
            'storeengine/membership-addon',
            $item->get_price_id()
        );

        if (empty($membership->integration)) {
            return $result;
        }

        $membershipPlanId = $membership->integration->get_integration_id();
        $accessGroup = get_post($membershipPlanId);

        if (empty($accessGroup) || $accessGroup->post_type !== 'storeengine_groups') {
            return $result;
        }

        $result = [
            'membership_plan_id'   => $membershipPlanId,
            'membership_plan_name' => $accessGroup->post_title,
            'membership_plan_slug' => $accessGroup->post_name,
            'membership'           => array_filter(
                [
                    'user_roles'       => get_post_meta($membershipPlanId, '_storeengine_membership_user_roles', true),
                    'content_protects' => get_post_meta($membershipPlanId, '_storeengine_membership_content_protect_types', true),
                    'expiration'       => get_post_meta($membershipPlanId, '_storeengine_membership_expiration', true),
                    'priority'         => get_post_meta($membershipPlanId, '_storeengine_membership_priority', true),
                    'excluded_items'   => get_post_meta($membershipPlanId, '_storeengine_membership_content_protect_excluded_items', true),
                ]
            ),
        ];

        // User-specific membership data
        if ($userId) {
            $userMembershipData = get_user_meta($userId, '_storeengine_user_membership_data', true);

            if (!empty($userMembershipData[$membershipPlanId])) {
                $result['user_membership_data'] = $userMembershipData[$membershipPlanId];
            }
        }

        return $result;
    }
}
