<?php

namespace BitApps\BTCBI_PRO\Triggers\WPSubscription;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Core\Util\Post;
use BitCode\FI\Flow\Flow;
use BitCode\FI\Triggers\WC\WCController;
use BitCode\FI\Triggers\WC\WCHelper;

final class WPSubscriptionController
{
    public static function info()
    {
        return [
            'name'              => 'WPSubscription',
            'title'             => __('Take recurring payment and build subscription with WooCommerce', 'bit-integrations-pro'),
            'type'              => 'custom_form_submission',
            'is_active'         => static::isPluginInstalled(),
            'documentation_url' => '#',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'wp-subscription/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public function getAllTasks()
    {
        if (!static::isPluginInstalled()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'WPSubscription'));
        }

        wp_send_json_success([
            ['form_name' => __('Subscription Activated', 'bit-integrations-pro'), 'triggered_entity_id' => 'subscrpt_subscription_activated', 'skipPrimaryKey' => true],
            ['form_name' => __('Subscription Expired', 'bit-integrations-pro'), 'triggered_entity_id' => 'subscrpt_subscription_expired', 'skipPrimaryKey' => true],
            ['form_name' => __('Subscription Cancelled', 'bit-integrations-pro'), 'triggered_entity_id' => 'subscrpt_subscription_cancelled', 'skipPrimaryKey' => true],
            ['form_name' => __('Subscription Status Changed Admin Alert', 'bit-integrations-pro'), 'triggered_entity_id' => 'subscrpt_status_changed_admin_email_notification', 'skipPrimaryKey' => true],
        ]);
    }

    public static function handleSubscriptionActivated($subscriptionId)
    {
        return self::prepareAndExecute('subscrpt_subscription_activated', $subscriptionId);
    }

    public static function handleSubscriptionExpired($subscriptionId)
    {
        return self::prepareAndExecute('subscrpt_subscription_expired', $subscriptionId);
    }

    public static function handleSubscriptionCancelled($subscriptionId)
    {
        return self::prepareAndExecute('subscrpt_subscription_cancelled', $subscriptionId);
    }

    public static function handleSubscriptionStatusChangedEmailAlert($subscriptionId, $oldStatus, $newStatus)
    {
        return self::prepareAndExecute('subscrpt_status_changed_admin_email_notification', $subscriptionId, ['old_status' => $oldStatus, 'new_status' => $newStatus]);
    }

    private static function prepareAndExecute($triggerId, $subscriptionId, $additionalData = [])
    {
        if (empty($subscriptionId)) {
            return false;
        }

        $orderId = get_post_meta($subscriptionId, '_subscrpt_order_id', true);

        if (!$orderId) {
            return false;
        }

        $order = wc_get_order($orderId);

        if (empty($order)) {
            return false;
        }

        $customerId = $order->get_customer_id();
        $customer = WCController::getUserInfo($customerId);
        $post = Post::get($subscriptionId);

        if (empty($post)) {
            return false;
        }

        $subscriptionData = [
            'id'     => $subscriptionId,
            'title'  => $post['post_title'] ?? '',
            'status' => $post['post_status'] ?? '',
        ];

        $orderData = WCHelper::accessOrderData($order);

        $data = Helper::prepareFetchFormatFields(
            array_merge(
                [
                    'subscription' => $subscriptionData,
                    'customer'     => $customer,
                    'order'        => $orderData,
                ],
                $additionalData
            )
        );

        return self::flowExecute($triggerId, $data);
    }

    private static function flowExecute($triggeredEntityId, $data)
    {
        if (empty($data) || !\is_array($data)) {
            return;
        }

        Helper::setTestData("btcbi_{$triggeredEntityId}_test", array_values($data));

        $flows = Flow::exists('WPSubscription', $triggeredEntityId);

        if (empty($flows)) {
            return;
        }

        Flow::execute('WPSubscription', $triggeredEntityId, array_column($data, 'value', 'name'), $flows);

        return ['type' => 'success'];
    }

    private static function isPluginInstalled()
    {
        return class_exists('WooCommerce') && class_exists('Sdevs_Subscription');
    }
}
