<?php

namespace BitApps\BTCBI_PRO\Triggers\WPUserFrontend;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Core\Util\Post;
use BitCode\FI\Core\Util\User;
use BitCode\FI\Flow\Flow;

final class WPUserFrontendController
{
    public static function info()
    {
        return [
            'name'              => 'WP User Frontend (WPUF)',
            'type'              => 'custom_form_submission',
            'is_active'         => self::isPluginInstalled(),
            'documentation_url' => '#',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'wpuf/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public function getAllTasks()
    {
        if (!self::isPluginInstalled()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'WP User Frontend'));
        }

        wp_send_json_success(self::getForms());
    }

    public static function handlePostFormSubmission($postId, $formId, $formSettings, $metaData)
    {
        $userData = \function_exists('wpuf_get_user') ? wpuf_get_user() : [];

        $post = Post::get($postId);
        $metadata = Post::getMeta($postId);
        $postData = array_merge((array) $post, $metadata);

        $formData = Helper::prepareFetchFormatFields(
            [
                'post_id'      => $postId,
                'form_id'      => $formId,
                'formSettings' => $formSettings,
                'meta_data'    => $metaData,
                'post_data'    => $postData,
                'user_data'    => $userData,
            ]
        );

        return static::flowExecute('postFormSubmission', $formData);
    }

    public static function handleRegistrationFormSubmission($userId, $formId, $formSettings)
    {
        $formData = Helper::prepareFetchFormatFields(
            array_merge(
                [
                    'user_id' => $userId,
                    'form_id' => $formId,
                ],
                User::get($userId),
                $formSettings
            )
        );

        return static::flowExecute('wpuf_after_register', $formData);
    }

    public static function handleSubscriptionPackUpdated($id, $request)
    {
        $subscription = !empty($request['subscription']) ? $request['subscription'] : '';
        $transientKey = "btcbi_subscription_pack_updated_{$subscription['ID']}";

        if (get_transient($transientKey)) {
            return;
        }

        $formData = Helper::prepareFetchFormatFields($subscription);

        set_transient($transientKey, true, 5);

        return static::flowExecute('subscriptionPackUpdated', $formData);
    }

    public static function handleCouponCreated($postId, $postData, $update, $postBefore = null)
    {
        if (
            (
                !empty($postBefore->post_status)
                && !\in_array($postBefore->post_status, ['draft', 'auto-draft'])
            )
            || $postData->post_status !== 'publish'
            || $postData->post_type != 'wpuf_coupon'
        ) {
            return;
        }

        $userData = User::get($postData->post_author);

        $formData = Helper::prepareFetchFormatFields(
            array_merge(
                $userData,
                (array) $postData,
            )
        );

        return static::flowExecute('wpuf_coupon_created', $formData);
    }

    public static function handleCouponUpdated($postId, $post)
    {
        $userData = User::get($post['post_author']);

        $formData = Helper::prepareFetchFormatFields(
            array_merge(
                $userData,
                (array) $post,
            )
        );

        return static::flowExecute('wpuf_update_coupon', $formData);
    }

    public static function handleProfileUpdated($userId, $formId, $formSettings, $metaData)
    {
        $userData = array_merge(User::get($userId), $metaData);

        $formData = Helper::prepareFetchFormatFields(
            array_merge(
                ['form_id' => $formId, 'user_id' => $userId],
                $userData,
                $formSettings,
            )
        );

        return static::flowExecute('wpuf_update_profile', $formData);
    }

    public static function handleProfileMetaUpdated($userId, $postdata)
    {
        $formData = Helper::prepareFetchFormatFields(
            array_merge(
                User::get($userId),
                $postdata
            )
        );

        return static::flowExecute('wpuf_pro_frontend_form_update_user_meta', $formData);
    }

    private static function getForms()
    {
        return [
            [
                'form_name'           => __('Post Form Submitted', 'bit-integrations-pro'),
                'triggered_entity_id' => 'postFormSubmission',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Registration Form Submitted', 'bit-integrations-pro'),
                'triggered_entity_id' => 'wpuf_after_register',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Profile updated Form Submitted', 'bit-integrations-pro'),
                'triggered_entity_id' => 'wpuf_update_profile',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('User Metadata updated Via Profile Edit Form', 'bit-integrations-pro'),
                'triggered_entity_id' => 'wpuf_pro_frontend_form_update_user_meta',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Subscription Pack Updated', 'bit-integrations-pro'),
                'triggered_entity_id' => 'subscriptionPackUpdated',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Coupon Created', 'bit-integrations-pro'),
                'triggered_entity_id' => 'wpuf_coupon_created',
                'skipPrimaryKey'      => true
            ],
            [
                'form_name'           => __('Coupon updated', 'bit-integrations-pro'),
                'triggered_entity_id' => 'wpuf_update_coupon',
                'skipPrimaryKey'      => true
            ],
        ];
    }

    private static function flowExecute($entityId, $formData)
    {
        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        $flows = Flow::exists('WPUserFrontend', $entityId);

        if (!$flows) {
            return;
        }

        Flow::execute('WPUserFrontend', $entityId, array_column($formData, 'value', 'name'), $flows);

        return ['type' => 'success'];
    }

    private static function isPluginInstalled()
    {
        return class_exists('WP_User_Frontend');
    }
}
