<?php

namespace BitApps\PiPro\src\Integrations\FluentCart;

// Prevent direct script access
if (!\defined('ABSPATH')) {
    exit;
}

use BitApps\Pi\Services\FlowService;
use BitApps\PiPro\src\Integrations\IntegrationHelper;

final class FluentCartTrigger
{
    /**
     * Handle product purchased event.
     *
     * @param array $data
     */
    public static function productPurchased($data)
    {
        if (empty($data['cart']) || empty($data['order'])) {
            return;
        }

        $data['cart'] = $data['cart']->toArray();
        $data['order'] = $data['order']->toArray();

        return self::execute('productPurchased', $data);
    }

    /**
     * Handle product updated event.
     *
     * @param array $data
     */
    public static function productUpdated($data)
    {
        if (empty($data['product'])) {
            return;
        }

        $data['product'] = $data['product']->toArray();

        return self::execute('productUpdated', $data);
    }

    /**
     * Handle cart updated event.
     *
     * @param array $data
     */
    public static function cartUpdated($data)
    {
        if (empty($data['cart'])) {
            return;
        }

        $data['cart'] = $data['cart']->toArray();

        return self::execute('cartUpdated', $data);
    }

    /**
     * Handle order refunded event.
     *
     * @param array $data
     */
    public static function orderRefunded($data)
    {
        return self::executeOrderPaymentTrigger('orderRefunded', $data);
    }

    /**
     * Handle order partially refunded event.
     *
     * @param array $data
     */
    public static function orderPartiallyRefunded($data)
    {
        return self::executeOrderPaymentTrigger('orderPartiallyRefunded', $data);
    }

    /**
     * Handle order fully refunded event.
     *
     * @param array $data
     */
    public static function orderFullyRefunded($data)
    {
        return self::executeOrderPaymentTrigger('orderFullyRefunded', $data);
    }

    /**
     * Handle order cancelled event.
     *
     * @param array $data
     */
    public static function orderCancelled($data)
    {
        return self::executeOrderStatusTrigger('orderCancelled', $data);
    }

    /**
     * Handle order status changed event.
     *
     * @param array $data
     */
    public static function orderStatusChanged($data)
    {
        return self::executeOrderStatusTrigger('orderStatusChanged', $data);
    }

    /**
     * Handle order status changed to processing event.
     *
     * @param array $data
     */
    public static function orderStatusUpdatedToProcessing($data)
    {
        return self::executeOrderStatusTrigger('orderStatusUpdatedToProcessing', $data);
    }

    /**
     * Handle order status changed to completed event.
     *
     * @param array $data
     */
    public static function orderStatusUpdatedToCompleted($data)
    {
        return self::executeOrderStatusTrigger('orderStatusUpdatedToCompleted', $data);
    }

    /**
     * Handle order status changed to on-hold event.
     *
     * @param array $data
     */
    public static function orderStatusUpdatedToOnHold($data)
    {
        return self::executeOrderStatusTrigger('orderStatusUpdatedToOnHold', $data);
    }

    /**
     * Handle order shipping status changed event.
     *
     * @param array $data
     */
    public static function orderShippingStatusChanged($data)
    {
        return self::executeOrderStatusTrigger('orderShippingStatusChanged', $data);
    }

    /**
     * Handle order shipping status changed to shipped event.
     *
     * @param array $data
     */
    public static function orderShipped($data)
    {
        return self::executeOrderStatusTrigger('orderShipped', $data);
    }

    /**
     * Handle order shipping status changed to unshipped event.
     *
     * @param array $data
     */
    public static function orderUnshipped($data)
    {
        return self::executeOrderStatusTrigger('orderUnshipped', $data);
    }

    /**
     * Handle order shipping status changed to unshippable event.
     *
     * @param array $data
     */
    public static function orderUnshippable($data)
    {
        return self::executeOrderStatusTrigger('orderUnshippable', $data);
    }

    /**
     * Handle order delivered event.
     *
     * @param array $data
     */
    public static function orderDelivered($data)
    {
        return self::executeOrderStatusTrigger('orderDelivered', $data);
    }

    /**
     * Handle order marked as paid event.
     *
     * @param array $data
     */
    public static function orderMarkedAsPaid($data)
    {
        return self::executeOrderPaymentTrigger('orderMarkedAsPaid', $data);
    }

    /**
     * Execute order status related triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeOrderStatusTrigger($machineSlug, $data)
    {
        if (empty($data['order'])) {
            return;
        }

        $data['order'] = $data['order']->toArray();

        return self::execute($machineSlug, $data);
    }

    /**
     * Execute order payment related triggers.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function executeOrderPaymentTrigger($machineSlug, $data)
    {
        if (empty($data['order']) || empty($data['customer']) || empty($data['transaction'])) {
            return;
        }

        $data['order'] = $data['order']->toArray();
        $data['customer'] = $data['customer']->toArray();
        $data['transaction'] = $data['transaction']->toArray();

        return self::execute($machineSlug, $data);
    }

    /**
     * Execute the trigger.
     *
     * @param string $machineSlug
     * @param array  $data
     */
    private static function execute($machineSlug, $data)
    {
        $flows = FlowService::exists('FluentCart', $machineSlug);

        if (!$flows) {
            return;
        }

        IntegrationHelper::handleFlowForForm($flows, $data);
    }
}
