<?php

namespace BitApps\PiPro\src\Integrations\MailMint;

use Mint\MRM\DataStores\TagData;
use Mint\MRM\DataStores\ListData;
use Mint\MRM\DataStores\ContactData;
use Mint\MRM\DataBase\Models\NoteModel;
use BitApps\Pi\src\Flow\NodeInfoProvider;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataBase\Models\CampaignModel;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Models\ContactGroupPivotModel;

if (!\defined('ABSPATH')) {
    exit;
}


final class MailMintServices
{
    private $nodeInfoProvider;

    public function __construct(NodeInfoProvider $nodeInfoProvider)
    {
        $this->nodeInfoProvider = $nodeInfoProvider;
    }

    /**
     * Get All List.
     *
     * @return collection
     */
    public function getAllList()
    {
        $classValidationErrors = MailMintHelper::validateClassExistence('ContactGroupModel');

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $lists = ContactGroupModel::get_all_lists_or_tags('lists');

        return [
            'response'    => $lists,
            'payload'     => [],
            'status_code' => 200
        ];
    }

    /**
     * Get List By ID.
     *
     * @return collection
     */
    public function getListById()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'ContactGroupModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $list = ContactGroupModel::get($fieldsMapping['id']);

        return [
            'response'    => $list,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Create List.
     *
     * @return collection
     */
    public function createList()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['title' => ['required', 'sanitize:text']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ListData'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $listId = ContactGroupModel::insert(
            new ListData($fieldsMapping),
            'lists'
        );

        if (!$listId) {
            return [
                'response'    => __('List could not be created.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => ContactGroupModel::get($listId),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Update List.
     *
     * @return collection
     */
    public function updateList()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ListData'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = ContactGroupModel::update(
            new ListData($fieldsMapping),
            $fieldsMapping['id'],
            'lists'
        );

        if (!$status) {
            return [
                'response'    => __('List could not be updated.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('List updated successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Delete List.
     *
     * @return collection
     */
    public function deleteList()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'ContactGroupModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = ContactGroupModel::destroy($fieldsMapping['id']);

        if (!$status) {
            return [
                'response'    => __('List not found or could not be deleted.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('List deleted successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Get All Tag.
     *
     * @return collection
     */
    public function getAllTag()
    {
        $classValidationErrors = MailMintHelper::validateClassExistence('ContactGroupModel');

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $tags = ContactGroupModel::get_all_lists_or_tags('tags');

        return [
            'response'    => $tags,
            'payload'     => [],
            'status_code' => 200
        ];
    }

    /**
     * Get Tag By ID.
     *
     * @return collection
     */
    public function getTagById()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'ContactGroupModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $tag = ContactGroupModel::get($fieldsMapping['id']);

        return [
            'response'    => $tag,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Create Tag.
     *
     * @return collection
     */
    public function createTag()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['title' => ['required', 'sanitize:text']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'TagData'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $tagId = ContactGroupModel::insert(
            new TagData($fieldsMapping),
            'tags'
        );

        if (!$tagId) {
            return [
                'response'    => __('Tag could not be created.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => ContactGroupModel::get($tagId),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Update Tag.
     *
     * @return collection
     */
    public function updateTag()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'TagData'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = ContactGroupModel::update(
            new TagData($fieldsMapping),
            $fieldsMapping['id'],
            'tags'
        );

        if (!$status) {
            return [
                'response'    => __('Tag could not be updated.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('Tag updated successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Delete Tag.
     *
     * @return collection
     */
    public function deleteTag()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'ContactGroupModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = ContactGroupModel::destroy($fieldsMapping['id']);

        if (!$status) {
            return [
                'response'    => __('Tags not found or could not be deleted.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('Tags deleted successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Get All Campaign.
     *
     * @return collection
     */
    public function getAllCampaign()
    {
        $classValidationErrors = MailMintHelper::validateClassExistence('CampaignModel');

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        global $wpdb;
        $campaigns = CampaignModel::get_all($wpdb, 0, 1000);

        return [
            'response'    => $campaigns,
            'payload'     => [],
            'status_code' => 200
        ];
    }

    /**
     * Get Campaign By ID.
     *
     * @return collection
     */
    public function getCampaignById()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'CampaignModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $campaign = CampaignModel::get($fieldsMapping['id']);

        return [
            'response'    => $campaign,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Get Campaign Emails.
     *
     * @return collection
     */
    public function getCampaignEmails()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'CampaignModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $emails = CampaignModel::get_campaign_email($fieldsMapping['id']);

        return [
            'response'    => $emails,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Add Recipients To Campaign.
     *
     * @return collection
     */
    public function addRecipientsToCampaign()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'campaign_id' => ['required', 'integer'],
            'lists'       => ['array'],
            'tags'        => ['array'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'CampaignModel'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $campaignId = $fieldsMapping['campaign_id'];

        if (!CampaignModel::is_campaign_exist($campaignId)) {
            return [
                'response'    => __('Campaign does not exist.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $recipients = [];

        $recipients['lists'] = array_map(
            function ($listId) {
                $list = ContactGroupModel::get($listId);

                return ContactGroupModel::get_or_insert_contact_group_by_title($list['title'], 'lists');
            },
            $fieldsMapping['lists']
        );

        $recipients['tags'] = array_map(
            function ($tagId) {
                $tag = ContactGroupModel::get($tagId);

                return ContactGroupModel::get_or_insert_contact_group_by_title($tag['title'], 'tags');
            },
            $fieldsMapping['tags']
        );

        $status = CampaignModel::insert_or_update_campaign_meta(
            $campaignId,
            'recipients',
            maybe_serialize($recipients)
        );

        if (!$status) {
            return [
                'response'    => __('Recipients could not be added to the campaign.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $campaign = CampaignModel::get($campaignId);
        $campaign['meta_value'] = unserialize($campaign['meta_value']);

        return [
            'response'    => $campaign,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Delete Campaign.
     *
     * @return collection
     */
    public function deleteCampaign()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'CampaignModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = CampaignModel::destroy($fieldsMapping['id']);

        if (!$status) {
            return [
                'response'    => __('Campaign not found or could not be deleted.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('Campaign deleted successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Get All Contact.
     *
     * @return collection
     */
    public function getAllContact()
    {
        $classValidationErrors = MailMintHelper::validateClassExistence('ContactModel');

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contacts = ContactModel::get_all(0, 1000);

        return [
            'response'    => $contacts,
            'payload'     => [],
            'status_code' => 200
        ];
    }

    /**
     * Get Contact By ID.
     *
     * @return collection
     */
    public function getContactById()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'ContactModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contact = ContactModel::get($fieldsMapping['id']);

        return [
            'response'    => $contact,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Add Lists To Contact.
     *
     * @return collection
     */
    public function addListsToContact()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'contact_id' => ['required', 'integer'],
            'lists'      => ['required', 'array'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ContactModel'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contactId = $fieldsMapping['contact_id'];
        if (!ContactModel::is_contact_ids_exists([$contactId])) {
            return [
                'response'    => __('Contact does not exist.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        if (!MailMintHelper::setListsToContact($contactId, $fieldsMapping['lists'])) {
            return [
                'response'    => __('Lists could not be added to the contact.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $contact = ContactModel::get($contactId);
        $contact['lists'] = ContactGroupModel::get_lists_to_contact(['id' => $contactId])['lists'] ?? [];

        return [
            'response'    => $contact,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Remove Lists From Contact.
     *
     * @return collection
     */
    public function removeListsFromContact()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'contact_id' => ['required', 'integer'],
            'lists'      => ['required', 'array'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ContactModel', 'ContactGroupPivotModel'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contactId = $fieldsMapping['contact_id'];
        if (!ContactModel::is_contact_ids_exists([$contactId])) {
            return [
                'response'    => __('Contact does not exist.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $lists = array_map(
            function ($listId) {
                return ['id' => $listId];
            },
            $fieldsMapping['lists']
        );

        $status = ContactGroupPivotModel::remove_groups_from_contacts($lists, [$contactId]);

        if (!$status) {
            return [
                'response'    => __('Lists could not be removed from the contact.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $contact = ContactModel::get($contactId);
        $contact['lists'] = ContactGroupModel::get_lists_to_contact(['id' => $contactId])['lists'] ?? [];

        return [
            'response'    => $contact,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Add Tags To Contact.
     *
     * @return collection
     */
    public function addTagsToContact()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'contact_id' => ['required', 'integer'],
            'tags'       => ['required', 'array'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ContactModel'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contactId = $fieldsMapping['contact_id'];

        if (!ContactModel::is_contact_ids_exists([$contactId])) {
            return [
                'response'    => __('Contact does not exist.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        if (!MailMintHelper::setTagsToContact($contactId, $fieldsMapping['tags'])) {
            return [
                'response'    => __('Tags could not be added to the contact.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $contact = ContactModel::get($contactId);
        $contact['tags'] = ContactGroupModel::get_tags_to_contact(['id' => $contactId])['tags'] ?? [];

        return [
            'response'    => $contact,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Remove Tags From Contact.
     *
     * @return collection
     */
    public function removeTagsFromContact()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'contact_id' => ['required', 'integer'],
            'tags'       => ['required', 'array'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ContactModel', 'ContactGroupPivotModel'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contactId = $fieldsMapping['contact_id'];
        if (!ContactModel::is_contact_ids_exists([$contactId])) {
            return [
                'response'    => __('Contact does not exist.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $tags = array_map(
            function ($tagId) {
                return ['id' => $tagId];
            },
            $fieldsMapping['tags']
        );

        $status = ContactGroupPivotModel::remove_groups_from_contacts($tags, [$contactId]);

        if (!$status) {
            return [
                'response'    => __('Tags could not be removed from the contact.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $contact = ContactModel::get($contactId);
        $contact['tags'] = ContactGroupModel::get_tags_to_contact(['id' => $contactId])['tags'] ?? [];

        return [
            'response'    => $contact,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Create Contact.
     *
     * @return collection
     */
    public function createContact()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'email' => ['required', 'sanitize:email'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            ['ContactGroupModel', 'ContactModel', 'ContactData'],
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $contactData = new ContactData($fieldsMapping['email'], $fieldsMapping);
        $contactId = ContactModel::insert($contactData);

        if (!$contactId) {
            return [
                'response'    => __('Contact could not be created.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 500
            ];
        }

        if (!MailMintHelper::setListsToContact($contactId, $fieldsMapping['lists'] ?? [])) {
            return [
                'response'    => __('Lists could not be added to the contact.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        if (!MailMintHelper::setTagsToContact($contactId, $fieldsMapping['tags'] ?? [])) {
            return [
                'response'    => __('Tags could not be added to the contact.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        $contact = ContactModel::get($contactId);
        $contact['lists'] = ContactGroupModel::get_lists_to_contact(['id' => $contactId])['lists'] ?? [];
        $contact['tags'] = ContactGroupModel::get_tags_to_contact(['id' => $contactId])['tags'] ?? [];

        return [
            'response'    => $contact,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Delete Contact.
     *
     * @return collection
     */
    public function deleteContact()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'ContactModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = ContactModel::destroy($fieldsMapping['id']);

        if (!$status) {
            return [
                'response'    => __('Contact not found or could not be deleted.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('Contact deleted successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Get Contact Notes.
     *
     * @return collection
     */
    public function getContactNotes()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'NoteModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $notes = NoteModel::get_notes_to_contact($fieldsMapping['id']);

        return [
            'response'    => $notes,
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Add Contact Note.
     *
     * @return collection
     */
    public function addContactNote()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = [
            'id'   => ['required', 'integer'],
            'note' => ['required', 'sanitize:text'],
        ];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'NoteModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $note = [
            'description' => $fieldsMapping['note'],
            'type'        => 'note',
            'is_public'   => 1,
            'status'      => 1,
            'created_by'  => get_current_user_id(),
        ];

        $status = NoteModel::insert($note, $fieldsMapping['id']);

        if (!$status) {
            return [
                'response'    => __('Note could not be added.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('Note added successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Delete Contact Note.
     *
     * @return collection
     */
    public function deleteContactNote()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();

        $rules = ['id' => ['required', 'integer']];

        $validationErrors = MailMintHelper::validateFieldMap($fieldsMapping, $rules);

        if (!empty($validationErrors)) {
            return $validationErrors;
        }

        $classValidationErrors = MailMintHelper::validateClassExistence(
            'NoteModel',
            $fieldsMapping
        );

        if (!empty($classValidationErrors)) {
            return $classValidationErrors;
        }

        $status = NoteModel::destroy($fieldsMapping['id']);

        if (!$status) {
            return [
                'response'    => __('Note not found or could not be deleted.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        return [
            'response'    => __('Note deleted successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }
}
