<?php

namespace BitApps\PiPro\src\Integrations\Stripe;

// Prevent direct script access
if (!\defined('ABSPATH')) {
    exit;
}


use BitApps\Pi\Helpers\Utility;
use BitApps\Pi\src\Authorization\AuthorizationFactory;
use BitApps\Pi\src\Authorization\AuthorizationType;
use BitApps\Pi\src\Flow\NodeInfoProvider;
use BitApps\Pi\src\Interfaces\ActionInterface;

class StripeAction implements ActionInterface
{
    private NodeInfoProvider $nodeInfoProvider;

    public function __construct(NodeInfoProvider $nodeInfoProvider)
    {
        $this->nodeInfoProvider = $nodeInfoProvider;
    }

    public function execute(): array
    {
        $executedNodeAction = $this->executeStripeAction();

        return Utility::formatResponseData(
            $executedNodeAction['status_code'],
            $executedNodeAction['payload'],
            $executedNodeAction['response']
        );
    }

    private function executeMachine($machineSlug, $data, $apiKey)
    {
        $header = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type'  => 'application/x-www-form-urlencoded'
        ];
        $stripeHelperObj = new StripeHelper($header);
        $metaData = $this->nodeInfoProvider->getFieldMapRepeaters('metaData-field-properties.value', false, false);

        switch ($machineSlug) {
            case 'createCustomer':
                return $stripeHelperObj->createCustomer($data, $metaData);

                break;

            case 'createProduct':
                return $stripeHelperObj->createProduct($data, $metaData);

                break;

            case 'createSubscription':
                return $stripeHelperObj->createSubscription($data, $metaData);

                break;

            case 'retrieveCustomer':
                return $stripeHelperObj->retrieveCustomer($data);

                break;

            case 'retrieveCustomerById':
                return $stripeHelperObj->retrieveCustomerById($data['customer_id'] ?? '');

                break;

            case 'retrieveBalance':
                return $stripeHelperObj->retrieveBalance($data);

                break;

            case 'retrievePrice':
                return $stripeHelperObj->retrievePrice($data);

                break;

            case 'retrieveSubscription':
                return $stripeHelperObj->retrieveSubscription($data);

                break;

            case 'retrieveSubscriptionById':
                return $stripeHelperObj->retrieveSubscriptionById($data['subscription_id'] ?? '');

                break;

            case 'retrieveSubscriptionCustomerId':
                return $stripeHelperObj->retrieveSubscriptionCustomerId($data['customer_id'] ?? '');

                break;

            case 'createPaymentIntent':
                return $stripeHelperObj->createPaymentIntent($data, $metaData);

                break;

            case 'createInvoice':
                return $stripeHelperObj->createInvoice($data, $metaData);

                break;

            case 'cancelSubscription':
                return $stripeHelperObj->cancelSubscription($data, $metaData);

                break;
        }
    }

    private function executeStripeAction()
    {
        $machineSlug = $this->nodeInfoProvider->getMachineSlug();
        $connectionId = $this->nodeInfoProvider->getFieldMapConfigs('connection-id.value');
        $data = $this->nodeInfoProvider->getFieldMapData();
        $tokenAuthorization = AuthorizationFactory::getAuthorizationHandler(
            AuthorizationType::API_KEY,
            $connectionId
        );
        $apiKey = $tokenAuthorization->getAccessToken();

        return $this->executeMachine($machineSlug, $data, $apiKey);
    }
}
