<?php

namespace BitApps\PiPro\src\Integrations\SureMembers;

use BitApps\Pi\Deps\BitApps\WPValidator\Validator;
use BitApps\Pi\src\Flow\NodeInfoProvider;
use SureMembers\Inc\Access;

if (!\defined('ABSPATH')) {
    exit;
}


final class SureMembersServices
{
    private $nodeInfoProvider;

    public function __construct(NodeInfoProvider $nodeInfoProvider)
    {
        $this->nodeInfoProvider = $nodeInfoProvider;
    }

    /**
     * Add user to access group.
     *
     * @return collection
     */
    public function addUserToAccessGroup()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();
        $rules = [
            'email'    => ['required', 'email', 'sanitize:email'],
            'groupIds' => ['required', 'array'],
        ];

        $validationError = self::validateFields($fieldsMapping, $rules);

        if (!empty($validationError)) {
            return $validationError;
        }

        $user = get_user_by('email', $fieldsMapping['email']);

        if (!$user) {
            return [
                'response'    => __('User not found with the provided email.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        Access::grant($user->ID, $fieldsMapping['groupIds']);

        return [
            'response'    => __('User added to access group successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    /**
     * Remove SureMembers User from Access Group.
     *
     * @return collection
     */
    public function removeUserFromAccessGroup()
    {
        $fieldsMapping = $this->nodeInfoProvider->getFieldMapData();
        $rules = [
            'email'    => ['required', 'email', 'sanitize:email'],
            'groupIds' => ['required', 'array'],
        ];

        $validationError = self::validateFields($fieldsMapping, $rules);

        if (!empty($validationError)) {
            return $validationError;
        }

        $user = get_user_by('email', $fieldsMapping['email']);

        if (!$user) {
            return [
                'response'    => __('User not found with the provided email.', 'bit-pi'),
                'payload'     => $fieldsMapping,
                'status_code' => 404
            ];
        }

        Access::revoke($user->ID, $fieldsMapping['groupIds']);

        return [
            'response'    => __('User removed from access group successfully.', 'bit-pi'),
            'payload'     => $fieldsMapping,
            'status_code' => 200
        ];
    }

    private static function validateFields($fieldMappings, $validationRules)
    {
        $validator = new Validator();
        $validation = $validator->make($fieldMappings, $validationRules);

        if ($validation->fails()) {
            return [
                'response'    => $validation->errors(),
                'payload'     => $payload ?? $fieldMappings,
                'status_code' => 400
            ];
        }
    }
}
