<?php

namespace BitApps\PiPro\src\Integrations\WPSubscription;

// Prevent direct script access
if (!\defined('ABSPATH')) {
    exit;
}

use BitApps\Pi\Helpers\Utility;
use BitApps\Pi\Services\FlowService;
use BitApps\PiPro\src\Integrations\IntegrationHelper;

final class WPSubscriptionTrigger
{
    /**
     * Triggered when a subscription is activated.
     *
     * @param int $subscriptionId
     *
     * @return bool|void
     */
    public static function subscriptionActivated($subscriptionId)
    {
        return self::prepareDataAndExecute($subscriptionId, 'subscriptionActivated');
    }

    /**
     * Triggered when a subscription is cancelled.
     *
     * @param int $subscriptionId
     *
     * @return bool|void
     */
    public static function subscriptionCancelled($subscriptionId)
    {
        return self::prepareDataAndExecute($subscriptionId, 'subscriptionCancelled');
    }

    /**
     * Triggered when a subscription status is changed.
     *
     * @param int $subscriptionId
     * @param string $oldStatus
     * @param string $newStatus
     *
     * @return bool|void
     */
    public static function subscriptionStatusChanged($subscriptionId, $oldStatus, $newStatus)
    {
        $additionalData = [
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
        ];

        return self::prepareDataAndExecute($subscriptionId, 'subscriptionStatusChanged', $additionalData);
    }

    /**
     * Triggered when a subscription is expired.
     *
     * @param int $subscriptionId
     *
     * @return bool|void
     */
    public static function subscriptionExpired($subscriptionId)
    {
        return self::prepareDataAndExecute($subscriptionId, 'subscriptionExpired');
    }

    /**
     * Prepare data and execute the flow.
     *
     * @param int   $subscriptionId
     * @param string $machineSlug
     * @param array  $additionalData
     *
     * @return bool|void
     */
    private static function prepareDataAndExecute($subscriptionId, $machineSlug, $additionalData = [])
    {
        if (empty($subscriptionId)) {
            return false;
        }

        $flows = FlowService::exists('WPSubscription', $machineSlug);

        if (!$flows) {
            return;
        }

        $orderId = get_post_meta($subscriptionId, '_subscrpt_order_id', true);

        if ($orderId <= 0) {
            return;
        }

        $order = wc_get_order($orderId);

        if (!$order) {
            return;
        }

        $post = get_post($subscriptionId);

        if (!$post) {
            return;
        }

        $additionalData['subscription_data'] = [
            'id'     => $subscriptionId,
            'title'  => $post->post_title ?? '',
            'status' => $post->post_status ?? '',
        ];

        $customerId = $order->get_customer_id();
        $additionalData['order_data'] = $order->get_data();
        $additionalData['customer_data'] = Utility::getUserInfo($customerId);

        IntegrationHelper::handleFlowForForm($flows, $additionalData);
    }

    private static function isPluginInstalled()
    {
        return class_exists('Sdevs_Subscription');
    }
}
