<?php

namespace BitCode\BitFormPro\API\Controller;

use BitCode\BitForm\Core\Integration\IntegrationHandler;
use BitCode\BitForm\Core\Util\HttpHelper;
use BitCode\BitForm\Core\Util\FieldValueHandler;
use WP_Error;

class MolliePaymentController extends PaymentServices
{
    protected $paymentName = 'mollie';
    protected $paymentType = 'order';

    private const MOLLIE_API_ENDPOINT = 'https://api.mollie.com/v2/payments';

    public function handlePayment()
    {
        $mollieDataString = file_get_contents('php://input');
        $data = json_decode($mollieDataString);
        $transactionId = $data->id;
        $paymentIntegId = sanitize_text_field($_GET['integID']);

        if (!$paymentIntegId) {
            return new WP_Error('no_id', 'Payment Integration ID parameter is missing', array('status' => 400));
        }

        $mollieData = $this->getMolliePaymentInfo($paymentIntegId, $transactionId);

        if ($mollieData->status === 'paid') {
            $this->saveMolliePaymentInfo($mollieData);
        }

    }

    private function getIntegrationDetails($paymentIntegId)
    {
        $integrationHandler = new IntegrationHandler(0);
        $integrationDetails = $integrationHandler->getAllIntegration('app', 'payments', '', $paymentIntegId);
        $integrationDetails = json_decode($integrationDetails[0]->integration_details);
        return $integrationDetails;
    }

    public function createPayment($request)
    {
        if (!$request) {
            wp_send_json_error('Request is required', 400);
        }
        $data = is_string($request) ? \json_decode($request) : $request;
        $payIntegID = $data->payIntegID;
        $amount = $data->amount;
        $currency = $data->currency;
        $meta = $data->metadata;
        $method = $data->method;
        $description = $data->description;
        $redirectUrl = $data->redirectUrl;
        $entryID = $meta->entryID;

        $integrationDetails = $this->getIntegrationDetails($payIntegID);
        $apiKey = $integrationDetails->apiKey;
        $redirectUrl = $redirectUrl . '?integID=' . $payIntegID;
        $webhook_url = get_rest_url() . 'bitform/v1/payments/mollie' . '?integID=' . $payIntegID;

        // get file value
        $submittedValue = $this->getSubmittedValue($entryID);
        $replaceDescription = FieldValueHandler::replaceFieldWithValue($description, $submittedValue);

        $headers = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
        ];
        $data = [
            'amount' => [
                'currency' => $currency,
                'value' => $amount,
            ],
            'description' => $replaceDescription,
            'redirectUrl' => $redirectUrl,
            'webhookUrl' => $webhook_url,
            'metadata' => (array) $meta,
            'method' => $method,
        ];

        $jsonData = wp_json_encode($data);
        $apiResponse = HttpHelper::post(self::MOLLIE_API_ENDPOINT, $jsonData, $headers);

        if (isset($apiResponse->status) && is_int($apiResponse->status) && $apiResponse->status >= 400) {
            wp_send_json_error($apiResponse, $apiResponse->status);
        } else {
            $transactionId = $apiResponse->id;
            update_option('bf_mollie_transaction_id', $transactionId);
            wp_send_json_success($apiResponse, 200);
        }
    }


    public function saveMolliePaymentInfo($data)
    {
        $this->transactionId = $this->sanitize($data->id);
        $this->formId = $this->sanitize($data->metadata->formID);
        $this->entryId = $this->sanitize($data->metadata->entryID);
        $this->fieldKey = $this->sanitize($data->metadata->fieldKey);
        $this->apiResponseData = $data;

        return $this->savePaymentInfo();
    }

    public function getMolliePaymentInfo($paymentIntegId, $transactionID)
    {
        $endpoint = self::MOLLIE_API_ENDPOINT . '/' . $transactionID;

        $integrationDetails = $this->getIntegrationDetails($paymentIntegId);

        $apiKey = $integrationDetails->apiKey;

        $headers = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
        ];

        $apiResponse = HttpHelper::get($endpoint, '', $headers);

        if (isset($apiResponse->status) && is_int($apiResponse->status) && $apiResponse->status >= 400) {
            return new WP_Error('api_error', $apiResponse->detail, array('status' => $apiResponse->status));
        } else {
            return $apiResponse;
        }
    }


    public function getSubmittedValue($entryId)
    {
        $formEntryMeta = $this->formMetaModel->get(
            [
                'meta_key',
                'meta_value',
            ],
            [
                'bitforms_form_entry_id' => $entryId,
            ]
        );
        $entries = [];
        foreach ($formEntryMeta as $key => $value) {
            $entries[$value->meta_key] = $value->meta_value;
        }
        return $entries;
    }

}

/**
 * https://bitform.xyz/wp-json/bitform/v1/payments/razorpay
 */