import { post, get } from "../utils/ajax";
import notificationToast from "../utils/notification-toast";
import { Offcanvas } from 'bootstrap';

export default class Tax {

    // Constructor method
    constructor() {
        this.taxName = jQuery('#tax_name');
        this.taxRate = jQuery('#tax_rate');
        this.taxType = jQuery('#tax_type');
        this.taxForm = jQuery('#add-tax-form');
        this.taxTable = jQuery('#wpb-tax-tbl');
        this.offcanvasElement = document.getElementById('tax-form');
        this.offcanvas = new Offcanvas(this.offcanvasElement);

        this.errorMessages = {
            tax_name: {
                selector: "#tax_name_error",
                blank: "Please enter tax name",
            },
            tax_booking_type: {
                selector: "#booking_type_error",
                blank: "Please select booking type",
            },
        };

        this.taxDatatable = this.initializeTaxTable();
        this.setupEventHandlers();
    }

    // Method to initialize the tax table
    initializeTaxTable() {
        return new DataTable('#wpb-tax-tbl', {
            "searching": false,
            "processing": true,
            "serverSide": true,
            "order": [
                [0, 'DESC']
            ],
            "paging": true,
            "lengthChange": false,
            "ajax": (data, callback, settings) => {
                return get('tax_list', data).then(resData => {
                    callback(resData);
                });
            },
            "columns": [
                {
                    "data": "name",
                    "name": 'tax_name',
                    "searchable": false
                },
                {
                    "data": "rate",
                    "name": 'tax_rate',
                    "searchable": false
                },
                {
                    "data": "booking_type",
                    "name": 'tax_booking_type',
                    "render": function (data, type, row) {
                        return data.map(type => `<span class="badge bg-primary-subtle p-2 text-primary me-2">${type}</span>`).join('');
                    },
                    "searchable": false
                },
                {
                    "data": "priority",
                    "name": 'tax_priority',
                    "render": function (data, type, row) {
                        return `<span>${row.tax_priority}</span>`;
                    },
                    "searchable": false
                },
                {
                    "data": "status",
                    "render": function (data, type, row) {
                        let checked = data == 1 ? 'checked' : '';
                        return `
                            <div class="form-check form-switch">
                                <input class="form-check-input tax_status" type="checkbox" data-id="${row.id}" ${checked}>   
                            </div>
                        `;
                    },
                    "searchable": false,
                    "orderable": false
                },
                {
                    "data": "actions",
                    "render": function (data, type, row) {
                        return `
                            <a href="#" data-id="${row.id}" class="me-3 edit-tax-button">
                                <img class="edit-icon" src="${wpbookit.wpb_plugin_url}core/admin/assets/images/edit-icon.svg" alt="edit-icon">
                                <svg class="spinner wpb-tax-edit-svg d-none" height="18" width="18" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
                                    <path fill="#d3d3d3" d="M304 48c0 26.5-21.5 48-48 48s-48-21.5-48-48 21.5-48 48-48 48 21.5 48 48zm-48 368c-26.5 0-48 21.5-48 48s21.5 48 48 48 48-21.5 48-48-21.5-48-48-48zm208-208c-26.5 0-48 21.5-48 48s21.5 48 48 48 48-21.5 48-48-21.5-48-48-48zM96 256c0-26.5-21.5-48-48-48S0 229.5 0 256s21.5 48 48 48 48-21.5 48-48zm12.9 99.1c-26.5 0-48 21.5-48 48s21.5 48 48 48 48-21.5 48-48c0-26.5-21.5-48-48-48zm294.2 0c-26.5 0-48 21.5-48 48s21.5 48 48 48 48-21.5 48-48c0-26.5-21.5-48-48-48zM108.9 60.9c-26.5 0-48 21.5-48 48s21.5 48 48 48 48-21.5 48-48-21.5-48-48-48z"></path>
                                </svg>
                            </a>
                            <a href="#" data-id="${row.id}" data-name="${row.name}" class="delete-tax-button">
                                <img src="${wpbookit.wpb_plugin_url}core/admin/assets/images/delete-icon.svg" alt="delete-icon">
                            </a>
                        `;
                    },
                    "searchable": false,
                    "orderable": false
                },
            ],
            language: window.wpbookit.datatable_language
        });
    }

    // Method to set up event handlers
    setupEventHandlers() {
        this.taxForm.on('submit', this.handleFormSubmit.bind(this));

        jQuery(document.body).on("click", ".edit-tax-button", this.handleEditTax.bind(this));
        jQuery(document.body).on("click", ".delete-tax-button", this.handleDeleteTax.bind(this));
        jQuery(document.body).on("change", ".tax_status", this.handleStatusChange.bind(this));
        jQuery(document.body).on("change", "#tax_booking_type", this.handleBookingTypeChange.bind(this));

        this.offcanvasElement.addEventListener('show.bs.offcanvas', (e) => {
            if (e.relatedTarget === undefined) {
                this.offcanvasElement.querySelector('.offcanvas-title-edit').classList.remove('d-none');
                this.offcanvasElement.querySelector('.offcanvas-title-add').classList.add('d-none');
                jQuery('#tax_name_error').text('').hide();
                jQuery('#tax_rate_error').text('').hide();
                jQuery('#booking_type_error').text('').hide();
                jQuery('#inclusive_tax_error').text('').hide();
                jQuery('#priority_error').text('').hide();
            } else {
                this.offcanvasElement.querySelector('.offcanvas-title-edit').classList.add('d-none');
                this.offcanvasElement.querySelector('.offcanvas-title-add').classList.remove('d-none');

                this.taxForm[0].reset();
                jQuery('#tax_type').trigger('change');
                jQuery('#tax_booking_type').trigger('change');
                jQuery('input[name="inclusive_tax"]').prop("checked", false);
                jQuery('#edit-tax-id').val('');
            }
        });

        jQuery(document.body).on("change keyup keydown", "#tax_rate", this.handleTaxRateChange.bind(this));
    }

    // Method to validate tax rate input
    validateTaxRateInput(value) {
        if (value.trim() === '') {
            this.showError("#tax_rate_error", "Please enter tax rate");
            return false;
        } else if (isNaN(value)) {
            this.showError("#tax_rate_error", "Please enter a valid number.");
            return false;
        } else if (parseFloat(value) <= 0) {
            this.showError("#tax_rate_error", "Tax rate must be greater than 0.");
            return false;
        } else {
            this.hideError("#tax_rate_error");
            return true;
        }
    }

    // Validate priority field
    validateTaxPriorityInput(value) {
        if (value.trim() === '') {
            this.showError("#priority_error", "Please enter your priority.");
            return false;
        } else if (isNaN(value)) {
            this.showError("#priority_error", "Please enter a valid number.");
            return false;
        } else if (parseInt(value, 10) < 1) {
            this.showError("#priority_error", "Priority must be at least 1.");
            return false;
        } else {
            this.hideError("#priority_error");
            return true;
        }
    }

    // Method to handle tax rate change
    handleTaxRateChange(event) {
        const input = event.currentTarget;
        const isValid = this.validateTaxRateInput(input.value);
    }

    // Method to validate a field
    validateField(value, errorSelector, blankErrorMessage) {
        if (!value || (Array.isArray(value) && value.length === 0)) {
            this.showError(errorSelector, blankErrorMessage);
            return false;
        } else {
            this.hideError(errorSelector);
            return true;
        }
    }

    // Method to validate form fields
    validateFormFields(formData) {
        let isValid = true;

        Object.keys(this.errorMessages).forEach(fieldName => {
            const fieldValue = formData.get(fieldName);
            const errorMessage = this.errorMessages[fieldName];
            if (fieldName === "tax_booking_type") {
                const bookingTypeValue = formData.getAll("tax_booking_type[]");
                const isFieldValid = this.validateField(bookingTypeValue.length > 0 ? bookingTypeValue : null, errorMessage.selector, errorMessage.blank);
                if (!isFieldValid) {
                    isValid = false;
                }
            } else {
                const isFieldValid = this.validateField(fieldValue, errorMessage.selector, errorMessage.blank);
                if (!isFieldValid) {
                    isValid = false;
                }
            }
        });

        // Validate tax rate separately
        const taxRateValue = formData.get('tax_rate');
        const isTaxRateValid = this.validateTaxRateInput(taxRateValue);
        if (!isTaxRateValid) {
            isValid = false;
        }

        // Validate priority field separately
        const taxPriorityValue = formData.get('tax_priority');
        const isTaxPriorityValid = this.validateTaxPriorityInput(taxPriorityValue);
        if (!isTaxPriorityValid) {
            isValid = false;
        }

        return isValid;
    }

    // Method to show an error message
    showError(elementId, errorMessage) {
        jQuery(elementId).text(errorMessage).show();
    }

    // Method to hide an error message
    hideError(elementId) {
        jQuery(elementId).text('').hide();
    }

    // Add a new method to handle booking type change
    handleBookingTypeChange(event) {
        const selectedValues = jQuery(event.currentTarget).val();

        if (selectedValues && selectedValues.includes("0")) {
            jQuery('#tax_booking_type option').each(function () {
                if (jQuery(this).val() !== "0") {
                    jQuery(this).prop('disabled', true);
                }
            });
        } else {
            jQuery('#tax_booking_type option').each(function () {
                if (jQuery(this).val() === "0") {
                    jQuery(this).prop('disabled', selectedValues.length > 0);
                } else {
                    jQuery(this).prop('disabled', false);
                }
            });
        }
    }

    // Method to handle form submission
    handleFormSubmit(event) {
        event.preventDefault();
        const formData = new FormData(this.taxForm[0]);
        const taxId = formData.get('edit-tax-id');
        this.inclusiveTax = document.querySelector('input[name="inclusive_tax"]:checked');

        const isValid = this.validateFormFields(formData);
        if (!isValid) { return; }

        formData.append('tax_booking_type', jQuery('#tax_booking_type').val());
        formData.append('inclusive_tax', this.inclusiveTax ? this.inclusiveTax.value : '');

        if (taxId) {
            jQuery('#wpb-submit-tax').prop('disabled', true);
            jQuery('.wpb-tax-submit-svg').removeClass('d-none');
            post('update_tax', formData).then(response => {
                this.handleFormResponse(response);
            })
                .catch(error => {
                    jQuery('#wpb-submit-tax').prop('disabled', false);
                    jQuery('.wpb-tax-submit-svg').addClass('d-none');
                    console.error('Error :', error);
                });
        } else {
            jQuery('#wpb-submit-tax').prop('disabled', true);
            jQuery('.wpb-tax-submit-svg').removeClass('d-none');
            post('add_tax', formData).then(response => {
                this.handleFormResponse(response);
            })
                .catch(error => {
                    jQuery('#wpb-submit-tax').prop('disabled', false);
                    jQuery('.wpb-tax-submit-svg').addClass('d-none');
                    console.error('Error :', error);
                });
        }
    }

    // Method to handle edit tax action
    handleEditTax(event) {
        event.preventDefault();
        var _this = this,
            taxId = jQuery(event.currentTarget).data('id');
        jQuery(event.currentTarget).find('.edit-icon').hide();
        jQuery(event.currentTarget).find('.wpb-tax-edit-svg').removeClass('d-none');

        get('edit_tax_details', { tax_id: taxId }).then(response => {
            const tax = response;
            _this.taxForm.find('#edit-tax-id').val(tax.id);
            _this.taxForm.find('#tax_name').val(tax.tax_name);
            _this.taxForm.find('#tax_rate').val(tax.tax_rate);
            _this.taxForm.find('#tax_type').val(tax.tax_type).trigger('change');
            _this.taxForm.find('#tax_booking_type').val(tax.tax_booking_type_ids).trigger('change');
            if (tax.inclusive_tax == 'yes') {
                _this.taxForm.find('#inclusive-tax-yes').prop("checked", true);
            } else {
                _this.taxForm.find('#inclusive-tax-no').prop("checked", true);
            }
            _this.taxForm.find('#tax_priority').val(tax.tax_priority);
            jQuery(event.currentTarget).find('.edit-icon').show();
            jQuery(event.currentTarget).parent().find('.wpb-tax-edit-svg').addClass('d-none');
            _this.offcanvas.show();
        });
    }

    // Method to handle delete tax action
    handleDeleteTax(event) {
        event.preventDefault();
        const taxId = jQuery(event.currentTarget).data('id');
        const taxName = jQuery(event.currentTarget).data('name');

        if (confirm(`Do you want to delete this tax: ${taxName}?`)) {
            post('delete_tax', { tax_id: taxId }).then(response => {
                const { message, status } = response;
                if (status === 'success') {
                    this.taxDatatable.ajax.reload();
                }
                notificationToast[status](message, status.toUpperCase(), { autoClose: true });
            });
        }
    }

    // Method to handle status change
    handleStatusChange(event) {
        const checkbox = event.currentTarget;
        const taxId = jQuery(checkbox).data('id');
        const isActive = checkbox.checked;

        // Update the status on the server
        post('update_tax_status', { tax_id: taxId, status: isActive ? 1 : 0 }).then(response => {
            const { message, status } = response;
            notificationToast[status](message, status.toUpperCase(), { autoClose: true });
        });
    }

    // Method to handle form response
    handleFormResponse(response) {
        const { message, status } = response;
        if (status == 'success') {
            this.taxDatatable.ajax.reload();
            this.offcanvas.hide();
        }
        jQuery('#wpb-submit-tax').prop('disabled', false);
        jQuery('.wpb-tax-submit-svg').addClass('d-none');
        notificationToast[status](message, status.toUpperCase(), { autoClose: true });
    }
}