<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://iqonic.design
 * @since      1.0.0
 *
 * @package    Wpb_Zoom_Booking
 * @subpackage Wpb_Zoom_Booking/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Wpb_Zoom_Booking
 * @subpackage Wpb_Zoom_Booking/admin
 * @author     Iqonic Design <hello@iqonic.design>
 */
class Wpb_Zoom_Booking_Create_Addon {

    /**
     * The meeting url of the zoom.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $create_meeting_url    The meeting url of the zoom.
     */
    protected $create_meeting_url;


    /**
     * The token url of the zoom.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $zoom_token_url    The token url of the zoom.
     */
    protected $zoom_token_url;


    /**
     * The authorize url of the zoom.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $zoom_authorize_url    The authorize url of the zoom.
     */
    protected $zoom_authorize_url;

    protected $telemed_service;


    /**
     * Initialize the class and set its properties.
     *
     * Sets the plugin name, version, and default URLs for the Zoom API.
     * Adds an action to handle the creation of Zoom meetings.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( ) {


        // Set the default URLs for the Zoom API
        $this->create_meeting_url = 'https://api.zoom.us/v2/users/me/meetings';
        $this->zoom_token_url = 'https://zoom.us/oauth/token';
        $this->zoom_authorize_url = 'https://zoom.us/oauth/authorize';
        $this->telemed_service = 'zoom_telemed';
        // Add an action to handle the creation of Zoom meetings
        add_action(
            'wpb_booking_zoom_create',
            array( $this, 'wpb_booking_zoom_create' ),
            10,
            2
        );
        add_action('wpb_get_available_telemed',array($this,'wpb_add_telemed'));
    }

    public function wpb_add_telemed($active_telemed)
	{
		$active_telemed[$this->telemed_service] = esc_html__("Zoom Telemed", 'wpbookit');
		return $active_telemed;
	}

    /**
     * Get the URL for the Zoom OAuth authorization.
     *
     * This function returns the URL to be used for the Zoom OAuth authorization
     * process. It is used to redirect the user to the Zoom authorization page
     * where they can authorize the application.
     *
     * @since    1.0.0
     * @return   string  The URL for the Zoom OAuth authorization.
     */
    public function wpb_get_consent_screen_url(){
        // Get the URL for the Zoom OAuth authorization.
        return $this->wpb_get_auth_url();
    }


    /**
     * Handle Zoom OAuth callback.
     *
     * This function is called when the user is redirected back to the site
     * after authorizing the application on the Zoom OAuth consent screen.
     * It retrieves the authorization code from the query parameters, exchanges
     * it for an access token, and stores the access token in the appropriate
     * place depending on the user's role.
     *
     * @return void
     */
    public function wpb_handle_zoom_callback() {
        // Check if the authorization code is present in the query parameters.
        if (isset($_GET['code'])) {
            // Exchange the authorization code for an access token.
            $token = $this->wpb_get_access_token($_GET['code']);
            // Get the current user.
            $current_user = wp_get_current_user();
            // If the token is successfully retrieved, store it in the appropriate place.
            if ($token) {
                if (in_array('administrator', (array)$current_user->roles)) {
                    update_option('wpb_zoom_access_token', json_encode($token));
                } else {
                    update_user_meta(get_current_user_id(), 'wpb_zoom_access_token', json_encode($token));
                }
            }
            // Redirect the user to a confirmation page.
            wp_redirect(admin_url('admin.php?page=wpbookit-dashboard&tab=settings'));
            exit();
        }
    }



    /**
     * Get the authentication URL for Zoom OAuth.
     *
     * @return string The URL for the Zoom OAuth authorization.
     */
    public function wpb_get_auth_url()
    {
        // Get the current user and the Zoom settings.
        $current_user = wp_get_current_user();
        $get_config = get_option('wpb_zoom_settings');

        // Get the client id and redirect URI from the Zoom settings.
        $wpb_zoom_client_id = isset($get_config['wpb_zoom_client_id']) ? $get_config['wpb_zoom_client_id'] : '';
        $wpb_zoom_client_redirect_uri = isset($get_config['wpb_zoom_client_redirect_uri']) ? $get_config['wpb_zoom_client_redirect_uri'] : '';

        // If the current user is staff, get the client id and redirect URI from their user meta.
        // if ( in_array( 'staff', (array) $current_user->roles ) ) {
        //     $get_config = get_user_meta($current_user->ID, 'wpb_zoom_settings', true);
        //     $wpb_zoom_client_id = isset($get_config['wpb_zoom_client_id']) ? $get_config['wpb_zoom_client_id'] : '';
        //     $wpb_zoom_client_redirect_uri = isset($get_config['wpb_zoom_client_redirect_uri']) ? $get_config['wpb_zoom_client_redirect_uri'] : '';
        // }

        // Build and return the Zoom OAuth authorization URL.
        return add_query_arg(array(
            'client_id' => $wpb_zoom_client_id,
            'redirect_uri' => $wpb_zoom_client_redirect_uri,
            'response_type' => 'code',
            'scope' => '', // Specify the required Zoom scopes here.
            'state' => 'a',
        ), $this->zoom_authorize_url);
    }

    /**
     * Get an access token using an authorization code.
     *
     * @param string $code The authorization code to exchange for an access token.
     * @return array|false The decoded response from Zoom, or false if there was an error.
     */
    public function wpb_get_access_token($code)
    {
        // Get the client id, secret, and redirect URI from settings
        $get_config = get_option('wpb_zoom_settings');
        $wpb_zoom_client_id = isset($get_config['wpb_zoom_client_id']) ? $get_config['wpb_zoom_client_id'] : '';
        $wpb_zoom_client_secret = isset($get_config['wpb_zoom_client_secret']) ? $get_config['wpb_zoom_client_secret'] : '';
        $wpb_zoom_client_redirect_uri = isset($get_config['wpb_zoom_client_redirect_uri']) ? $get_config['wpb_zoom_client_redirect_uri'] : '';
        
        // If the current user is staff, get the client id and secret from user meta
        $current_user = wp_get_current_user();
        // if ( in_array( 'staff', (array) $current_user->roles ) ) {
        //     $get_config = get_user_meta($current_user->ID, 'wpb_zoom_settings', true);
        //     $wpb_zoom_client_id = isset($get_config['wpb_zoom_client_id']) ? $get_config['wpb_zoom_client_id'] : '';
        //     $wpb_zoom_client_secret = isset($get_config['wpb_zoom_client_secret']) ? $get_config['wpb_zoom_client_secret'] : '';
        //     $wpb_zoom_client_redirect_uri = isset($get_config['wpb_zoom_client_redirect_uri']) ? $get_config['wpb_zoom_client_redirect_uri'] : '';
        // }
        
        // Make the request to get the access token
        $tokenURL = $this->zoom_token_url;
        $response = wp_remote_post($tokenURL, array(
            'body' => array(
                'code' => $code,
                'grant_type' => 'authorization_code',
                'client_id' => $wpb_zoom_client_id,
                'client_secret' => $wpb_zoom_client_secret,
                'redirect_uri' => $wpb_zoom_client_redirect_uri,
            ),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => '*',
            ),
            'timeout' => 30,
        ));

        // Check for errors
        if (is_wp_error($response)) {
            // return new WP_Error('zoom_oauth_failed', 'Failed to retrieve OAuth token from Zoom.', array('status' => 500));
            return false;
        }
        // return false;
        // Parse the response
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        return $response_data;
    }


    /**
     * Get a new access token using a refresh token.
     *
     * @param string $refresh_token The refresh token to use.
     * @return array|false The decoded response from Zoom, or false if there was an error.
     */
    public function wpb_get_access_token_by_refresh($refresh_token)
    {
        // Get the client id and secret from settings
        $get_config = get_option('wpb_zoom_settings');
        $wpb_zoom_client_id = isset($get_config['wpb_zoom_client_id']) ? $get_config['wpb_zoom_client_id'] : '';
        $wpb_zoom_client_secret = isset($get_config['wpb_zoom_client_secret']) ? $get_config['wpb_zoom_client_secret'] : '';

        // If the current user is staff, get the client id and secret from user meta
        $current_user = wp_get_current_user();
        // if ( in_array( 'staff', (array) $current_user->roles ) ) {
        //     $get_config = get_user_meta($current_user->ID, 'wpb_zoom_settings', true);
        //     $wpb_zoom_client_id = isset($get_config['wpb_zoom_client_id']) ? $get_config['wpb_zoom_client_id'] : '';
        //     $wpb_zoom_client_secret = isset($get_config['wpb_zoom_client_secret']) ? $get_config['wpb_zoom_client_secret'] : '';
        // }

        // Make the request to get the new access token
        $tokenURL = $this->zoom_token_url;
        $response = wp_remote_post($tokenURL, array(
            'body' => array(
                'grant_type' => 'refresh_token',
                'refresh_token' => $refresh_token,
                'client_id' => $wpb_zoom_client_id,
                'client_secret' => $wpb_zoom_client_secret,
            ),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => '*',
            ),
            'timeout' => 30,
        ));

        // Check for errors
        if (is_wp_error($response)) {
            return false;
            // return new WP_Error('zoom_oauth_failed', 'Failed to retrieve OAuth token from Zoom.', array('status' => 500));
        }

        // Parse the response
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        return $response_data;
    }

    /**
     * Check if the given access token is expired.
     *
     * @param string $access_token The access token to check.
     * @return bool Returns true if the access token is expired, false otherwise.
     */
    public function wpb_is_access_token_expired($access_token) {
        // Decode the access token
        $token_parts = explode('.', $access_token);
        $token_payload = base64_decode($token_parts[1]);
        $token_data = json_decode($token_payload, true);
    
        // Check if 'exp' or 'expires_in' exists and is not expired
        if (isset($token_data['exp'])) {
            $expiry_time = $token_data['exp'];
            $current_time = time();
            return $expiry_time < $current_time;
        } elseif (isset($token_data['expires_in'])) {
            $expiry_time = time() + $token_data['expires_in'];
            $current_time = time();
            return $expiry_time < $current_time;
        } else {
            // No expiry information found, assume expired
            return true;
        }
    }

    /**
     * Is User Authorized Or Not.
     *
     * @since    1.0.0
     */
    public function wpb_is_user_authorized() {

        $token_data = get_option('wpb_zoom_access_token');

        $current_user = wp_get_current_user();
        if ( in_array( 'staff', (array) $current_user->roles ) ) {
            $token_data = get_user_meta($current_user->ID, 'wpb_zoom_access_token', true);
        }

        if ($token_data) {
            $token_data = json_decode($token_data, true);
            $token_data = (array) $token_data;
            if(isset($token_data['access_token']) && $token_data['access_token']) {
                return true;
            }
        }
        return false;
    }


    /**
     * Creates a Zoom meeting for the customer based on the provided booking details.
     *
     * @param array $booking_details The details of the booking.
     *                               - booking_date (string): The date of the booking.
     *                               - timeslot (string): The time slot of the booking.
     *                               - topic (string): The topic of the meeting (default: 'New Meeting General Talk').
     * @param int $booking_id The ID of the booking.
     * @return void
     */
    public function wpb_booking_zoom_create($booking_details, $booking_id){
        global $wpdb;
       
        // Get the user ID of the booking type's staff member
        $booking_type_id = $booking_details['booking_type_id'];
        $user_id = get_metadata( 'wpb_booking_type', $booking_type_id, 'staff', true );
        $meeting_link_type = get_metadata( 'wpb_booking_type', $booking_type_id, 'meeting_link_type', true );
        $current_user = new  WP_User(get_metadata( 'wpb_booking_type', $booking_type_id, 'staff', true ));

        
        if ( in_array( 'staff', (array) $current_user->roles ) ) {
           // $get_config = get_user_meta($current_user->ID, 'wpb_zoom_settings', true);
            $get_config = get_option( 'wpb_zoom_settings',[] );

        }elseif (in_array( 'administrator', (array) $current_user->roles ) ){
           
            $get_config = get_option( 'wpb_zoom_settings',[] );
        }
        $wpb_zoom_status =  $get_config['wpb_zoom_status']??"off";
        
        if($wpb_zoom_status=='off'){
            return ; 
        }


        $booking_type_instance = new WPB_Booking_Type($booking_type_id);
        $enable_group_booking = $booking_type_instance->get_meta('enable_group_booking') ?? false;

        if($meeting_link_type != 'zoom'){
            return;
        }

        if($enable_group_booking != false){
            
            $booking_query = $wpdb->prepare(
                "SELECT * FROM $wpdb->wpb_bookings WHERE booking_date = %s AND timeslot = %s AND status = %s",
                $booking_details['booking_date'], $booking_details['timeslot'], 'wpb-approved'
            );
            $booking_results = $wpdb->get_results($booking_query);
            if( is_array($booking_results) && count($booking_results) > 0 ){
                foreach($booking_results as $booking){
                    $location_source = get_metadata( 'wpb_bookings', $booking->id, 'location_source', true );
                    if(!empty($location_source)){
                        update_metadata('wpb_bookings', (int)$booking_id, 'location','online_video');
                        update_metadata('wpb_bookings', (int)$booking_id, 'location_source', $location_source);
                        delete_metadata('wpb_bookings', (int)$booking_id, 'zoom_error');
                        return;
                    }
                }
            }
        }
        // Get the access token data
        $token_data = json_decode( get_option('wpb_zoom_access_token') );
        
        // If the user ID is available, get the access token data from the user meta
        if(!empty($user_id)){
            $token_data = json_decode( get_user_meta($user_id, 'wpb_zoom_access_token', true) );
            // If the access token data is not available, get it from the option
            if(!$token_data){
                $token_data = json_decode( get_option('wpb_zoom_access_token') );
            }
        }
        // If the user ID is not available, get the access token data from the option
        else{
            // If the user is logged in, get the ID of the current user
            if(is_user_logged_in()){
                $user_id = get_current_user_id();
            }
            // If the user is not logged in, get the ID of the first administrator
            else{
                $admins = get_users(array('role' => 'administrator'));
                $user_id = $admins[0]->ID;
                $token_data = json_decode( get_option('wpb_zoom_access_token') );
            }
        }
        $token_data = (array) $token_data;
        if(isset($token_data['error']) && $token_data['error']){
            wpb_error_log($token_data['error'], 'zoom_error');
            update_metadata('wpb_bookings', (int)$booking_id, 'zoom_error', $token_data['reason'], '');
            if($booking_details['re_generate']){
                $response_data = [
                    "status"    => 'error',
                    "message"   => $token_data['reason'],
                ];
                wp_send_json_error($response_data);
            }
        }
        // Get the user data
        $user_data = get_userdata( $user_id );

        // Refresh the access token if it is expired
        if ( $this->wpb_is_access_token_expired($token_data['access_token']) ) {
            if ($token_data['refresh_token']) {
                $token = $this->wpb_get_access_token_by_refresh($token_data['refresh_token']);
                if( $token ){
                    if ( in_array( 'administrator', (array) $user_data->roles ) ) {
                        update_option('wpb_zoom_access_token', json_encode($token));
                        $token_data = json_decode( get_option('wpb_zoom_access_token') );
                    }
                    else{
                        update_user_meta($user_id, 'wpb_zoom_access_token', json_encode($token));
                        $token_data = json_decode( get_user_meta($user_id, 'wpb_zoom_access_token', true) );
                    }
                }
            } 
        }


        // Add the access token and timezone to the booking details
        $booking_details['access_token'] = $token_data['access_token'];
        $booking_details['timezone'] = date_default_timezone_get();

        // Create the Zoom meeting
        $response = $this->wpb_create_zoom_meeting($booking_details);

        // Update the meeting link in the booking metadata
        if(isset($response['join_url']) && !empty($response['join_url'])){
            update_metadata('wpb_bookings', (int)$booking_id, 'location','online_video');
            update_metadata('wpb_bookings', (int)$booking_id, 'location_source', $response['join_url']);
            delete_metadata('wpb_bookings', (int)$booking_id, 'zoom_error');
        }
    }
    

    /**
     * Creates a Zoom meeting based on the provided booking details.
     *
     * @param array $booking_details The details of the booking.
     *                               - booking_date (string): The date of the booking.
     *                               - timeslot (string): The time slot of the booking.
     *                               - topic (string): The topic of the meeting (default: 'New Meeting General Talk').
     *                               - type (int): The type of the meeting (default: 2).
     *                               - duration (int): The duration of the meeting (default: 30).
     *                               - password (string): The password of the meeting (default: random).
     *                               - timezone (string): The timezone of the meeting (default: 'Asia/Kolkata').
     *                               - agenda (string): The agenda of the meeting (default: 'General Talk Meeting').
     *                               - access_token (string): The access token for authentication.
     *
     * @return array|WP_Error The join URL of the created meeting if successful, otherwise a WP_Error object.
     */
    public function wpb_create_zoom_meeting($booking_details = [])
    
    {
        // Generate the start time for the meeting in ISO 8601 format.
        $start_time = $booking_details['booking_date']."T".$booking_details['timeslot']."Z";

        // Prepare the request body for creating the meeting.
        $requestBody = [
            'topic'      => $booking_details['topic'] ?? 'New Meeting General Talk', // The topic of the meeting.
            'type'       => $booking_details['type'] ?? 2, // The type of the meeting.
            'start_time' => $start_time, // The start time of the meeting.
            'duration'   => $booking_details['duration'] ?? 30, // The duration of the meeting.
            'password'   => $booking_details['password'] ?? mt_rand(), // The password of the meeting.
            'timezone'   => $booking_details['timezone'] ?? 'Asia/Kolkata', // The timezone of the meeting.
            'agenda'     => $booking_details['agenda'] ?? 'General Talk Meeting', // The agenda of the meeting.
            'settings'   => [
                'host_video' => false, // Whether the host video is enabled.
                'participant_video' => false, // Whether the participant video is enabled.
                'mute_upon_entry' => true, // Whether to mute upon entry.
                'waiting_room' => true // Whether the waiting room is enabled.
            ],
        ];

        // Prepare the API URL for creating the meeting.
        $api_url = $this->create_meeting_url;
        // Send a POST request to create the meeting.
        $response = wp_remote_post($api_url, array(
            'body' => json_encode( $requestBody ),
            'headers' => array(
                'Authorization' => 'Bearer ' . $booking_details['access_token'], // The access token for authentication.
                'Content-Type' => 'application/json',
                'Accept' => '*',
            ),
            'timeout' => 30,
        ));

        // Check if there was an error in the request.
        if (is_wp_error($response)) {
            return new WP_Error('zoom_oauth_failed', 'Failed to retrieve OAuth token from Zoom.', array('status' => 500));
        }
        // Retrieve the response body.
        $response_body = wp_remote_retrieve_body($response);

        // Decode the response data.
        $response_data = json_decode($response_body, true);
       // print_r($response_data);die;

        return ['join_url' => $response_data['join_url']]; // Return the join URL of the created meeting.
    }

    public function wpb_get_zoom_saved_token() {
        $token_data = get_option('wpb_zoom_access_token');

        $current_user = wp_get_current_user();
        if ( in_array( 'staff', (array) $current_user->roles ) ) {
            $token_data = get_user_meta($current_user->ID, 'wpb_zoom_access_token', true);
        }

        if ($token_data) {
            $token_data = json_decode($token_data, true);
            return $token_data;
        }

        return false;
    }
}

$Wpb_Zoom_Booking_Create_Addon = new Wpb_Zoom_Booking_Create_Addon();