<?php
/**
 * Regular Payment
 *
 */

defined('ABSPATH') || exit;

/**
 * Payment Class.
 */
class WPB_Payment extends WPB_Data
{

	/**
	 * ID for this object.
	 */
	protected $id = 0;

	/**
	 * Payment Data array. This is the core payment data exposed in APIs.
	 */
	protected $data = array(
		'bookings_id' => 0,
		'payment_mode' => '',
		'total_amount' => 0,
		'paid_amount' => 0,
		'payment_status' => '',
	);

	/**
	 * Stores additional object_read
	 */
	protected $object_read = null;

	/**
	 * Stores meta in cache for future reads.
	 */
	protected $cache_group = 'payment';

	/**
	 * Get the payment if ID is passed, otherwise the payment is new and empty.
	 **/
	public function __construct($payment = 0)
	{
		if (is_numeric($payment) && $payment > 0):
			$this->set_id($payment);

		elseif ($payment instanceof self):
			$this->set_id($payment->get_id());

		elseif (!empty($payment->id)):
			$this->set_id($payment->id);

		else:
			$this->set_object_read(true);

		endif;

		if ($this->get_id() > 0):
			$this->read($payment);
		endif;
	}

	/**
	 * Set ID.
	 */
	public function set_id($id)
	{
		$this->id = absint($id);
	}

	/**
	 * Set object read property.
	 */
	public function set_object_read($read = true)
	{
		$this->object_read = (bool) $read;
	}

	/**
	 * Sets order bookings_id.
	 */
	public function set_bookings_id($bookings_id)
	{
		$this->set_prop('bookings_id', $bookings_id);
	}

	/**
	 * Sets order payment_mode.
	 */
	public function set_payment_mode($payment_mode)
	{
		$this->set_prop('payment_mode', $payment_mode);
	}

	/**
	 * Sets order total_amount.
	 */
	public function set_total_amount($total_amount)
	{
		$this->set_prop('total_amount', $total_amount);
	}

	/**
	 * Sets order paid_amount.
	 */
	public function set_paid_amount($paid_amount)
	{
		$this->set_prop('paid_amount', $paid_amount);
	}

	/**
	 * Sets payment_status.
	 */
	public function set_payment_status($payment_status)
	{
		$this->set_prop('payment_status', $payment_status);
	}

	/**
	 * Sets order status.
	 */
	public function set_status($status)
	{
		$this->set_prop('status', $status);
	}

	/**
	 * Sets a prop for a setter method of the current object.
	 */
	protected function set_prop($prop, $value)
	{
		if (array_key_exists($prop, $this->data)) {
			$this->changes[$prop] = $value;
		}
	}

	public function read($payment)
	{
		foreach ($payment as $prop => $value):
			if (array_key_exists($prop, $this->data)) {
				$this->changes[$prop] = $value;
			}
		endforeach;
	}

	/**
	 * Returns the unique ID for payment object.
	 **/
	public function get_id()
	{
		return $this->id;
	}

	/**
	 * Gets order bookings_id.
	 */
	public function get_bookings_id($context = 'view')
	{
		return $this->get_prop('bookings_id', $context);
	}

	/**
	 * Gets order payment_mode.
	 */
	public function get_customer_id($context = 'view')
	{
		return $this->get_prop('customer_id', $context);
	}

	/**
	 * Gets order customer.
	 */
	public function get_customer($context = 'view')
	{
		$customerID = $this->get_customer_id();
		if ($customerID):
			return get_userdata($customerID);
		endif;
		return $customerID;
	}

	/**
	 * Gets order total_amount.
	 */
	public function get_timeslot($context = 'view')
	{
		return $this->get_prop('total_amount', $context);
	}

	/**
	 * Gets order paid_amount.
	 */
	public function get_transaction_id($context = 'view')
	{
		return $this->get_prop('paid_amount', $context);
	}

	/**
	 * Get payment_status.
	 */
	public function get_date_created($context = 'view')
	{
		return $this->get_prop('payment_status', $context);
	}

	/**
	 * Return the payment statuses without wpb- internal prefix.
	 **/
	public function get_status($context = 'view')
	{
		$status = $this->get_prop('status', $context);

		if (empty($status) && 'view' === $context):
			$status = apply_filters('wpb_default_payment_status', 'pending');
		endif;
		return $status;
	}

	/**
	 * Get basic order data in array format.
	 */
	public function get_base_data()
	{
		return array_merge(
			array('id' => $this->get_id()),
			$this->data,
		);
	}

	/**
	 * Get all class data in array format.
	 * @return array
	 */
	public function get_data()
	{
		return array_merge(
			$this->get_base_data(),
			array(
				'meta_data' => $this->get_meta_data(),
			)
		);
	}

	/**
	 * Gets a prop for a getter method of the current object.
	 */
	protected function get_prop($prop, $context = 'view')
	{
		$value = null;

		if (array_key_exists($prop, $this->data)):
			$value = array_key_exists($prop, $this->changes) ? $this->changes[$prop] : $this->data[$prop];

			if ('view' === $context):
				$value = apply_filters('wpb_get_' . $prop, $value, $this);
			endif;
		endif;
		return $value;
	}

	/**
	 * Helper method to compute meta cache key. Different from WP Meta cache key in that meta data cached using this key also contains meta_id column.
	 */
	public function get_payment_cache_key()
	{
		if (!$this->get_id())
			return false;
		return self::generate_cache_key($this->get_id(), $this->cache_group);
	}

	/**
	 * Generate cache key from id and group.
	 */
	public static function generate_cache_key($id, $cache_group)
	{
		return WPB_Cache_Handler::get_cache_prefix($cache_group) . WPB_Cache_Handler::get_cache_prefix('object_' . $id) . 'object_payment_' . $id;
	}

	public function get_payment($paymentID = false)
	{
		if (!is_numeric($paymentID))
			return false;

		if (!empty($this->cache_group)):
			$cache_key = $this->get_payment_cache_key();

			if (!empty($cache_key)):
				$cached_payment = wp_cache_get($cache_key, $this->cache_group);
			endif;
		endif;

		try {
			$payment = new WPB_Booking($paymentID);
			if (!$cached_payment && !empty($this->cache_group)):
				wp_cache_set($cache_key, $payment, $this->cache_group);
			endif;
			return $payment;
		} catch (Exception $e) {
			$error = new WP_Error('payment_error', $e->getMessage());
			return false;
		}
	}

	public function get_payments($args = array(), $defaults = array())
{
    global $wpdb;
    $payments = [];
    $query = '';

    // Extract individual arguments
    $per_page = !empty($args['per_page']) ? $args['per_page'] : 5;
    $offset = !empty($args['offset']) ? $args['offset'] : 0;

    $data_range = !empty($args['date_range']) ? $args['date_range'] : '';
    $payment_amount = !empty($args['payment_amount']) ? $args['payment_amount'] : '';
    $payment_status = !empty($args['payment_status']) ? $args['payment_status'] : '';
    $payment_mode = !empty($args['payment_mode']) ? $args['payment_mode'] : '';
    $payment_name = !empty($args['payment_name']) ? $args['payment_name'] : '';
    $end_date = !empty($args['date_to']) ? $args['date_to'] : '';
    $start_date = !empty($args['date_from']) ? $args['date_from'] : '';

    $payment_id = !empty($args['payment_id']) ? $args['payment_id'] : '';
    $staff_id = $args['staff'] ;

    $order = sanitize_text_field($args['order'] ?? "");
    $order_by = sanitize_text_field($args['order_by'] ?? "id");

    $query = $wpdb->prepare("SELECT SQL_CALC_FOUND_ROWS * , payments.id AS payment_id ,payments.date_created as p_date_created 
        FROM %i AS payments 
        LEFT JOIN %i AS bookings ON payments.bookings_id = bookings.id 
        LEFT JOIN %i AS bookingsmeta ON bookings.id = bookingsmeta.wpb_bookings_id AND bookingsmeta.meta_key = 'staff_id'
        WHERE 1 = 1",
        $wpdb->wpb_payments,
        $wpdb->wpb_bookings,
        $wpdb->wpb_bookingsmeta,

    );

    if (!empty($start_date) && !empty($end_date)) {
        if ($start_date == $end_date) {
            $query .= $wpdb->prepare(" AND DATE(payments.date_created) = %s ", $start_date);
        } else {
            $query .= $wpdb->prepare(" AND payments.date_created BETWEEN %s AND %s", $start_date, $end_date);
        }
    }

    if (!empty($payment_amount)) {
        $query .= $wpdb->prepare(" AND payments.paid_amount = %f", $payment_amount);
    }

    if (!empty($payment_status) && count($payment_status) > 0) {
        $placeholders = rtrim(str_repeat('%s,', count($payment_status)), ',');
        $query .= $wpdb->prepare(" AND payments.payment_status IN ({$placeholders})", ...$payment_status);
    }

    if (!empty($payment_mode)) {
        $placeholders = rtrim(str_repeat('%s,', count($payment_mode)), ',');
        $query .= $wpdb->prepare(" AND payments.payment_mode IN ({$placeholders})", ...$payment_mode);
    }

    if (!empty($payment_name)) {
        $query .= $wpdb->prepare(" AND bookings.booking_name LIKE %s", $payment_name);
    }

    if (!empty($payment_id)) {
        $query .= $wpdb->prepare(" AND payments.id LIKE %s", $payment_id);
	}
		
	if($staff_id != 0){
		$query .= $wpdb->prepare(" AND bookingsmeta.meta_value LIKE %s", $staff_id);
	}

    $count_query_selection = "SELECT FOUND_ROWS()";

    // Add LIMIT clause
    $query .= $wpdb->prepare(" ORDER BY {$order_by} {$order} LIMIT %d, %d", $offset, $per_page);
    $results = $wpdb->get_results($query, ARRAY_A);
    $total_count = $wpdb->get_var($count_query_selection);

    foreach ($results as $result) {
        $booking_type = new WPB_Booking_Type($result["booking_type_id"]);
        $original_price = $booking_type->get_meta('price');
        $wpb_prefix = wpb_get_general_settings()['prefix'] ?? '';
        $wpb_postfix = wpb_get_general_settings()['postfix'] ?? '';

		$tax_data_meta = get_metadata('wpb_bookings', $result["bookings_id"], 'tax_data', true);

		if (is_array($tax_data_meta)) {
			$tax_data = $tax_data_meta; 
		} else {
			$tax_data = json_decode($tax_data_meta, true) ?? [];
		}
		extract( $tax_data  );
		$tax_data = iqwpb_generate_html_output($tax_data ?? [], $price_excluding_tax, $main_total, $result['total_amount'], $result['paid_amount'],'',$result["bookings_id"]);
        $payment = new stdClass();
        $payment->payment_id = $result["payment_id"];
        $payment->transaction_id = $result["transaction_id"] ?? '-';
        $payment->bookings_id = $result["bookings_id"];
        $payment->booking_type_id = $result["booking_type_id"];
        $payment->tax_data = $tax_data['tax_values'];
        $payment->price = $original_price;
        $payment->id = $result['id'];
        $payment->customer_email = $result['booking_email'];
        $payment->bookings_id = $result['bookings_id'];
        $payment->payment_mode = $result['payment_mode'];
        $payment->total_amount = $result['total_amount'];
        $payment->paid_amount = $result['paid_amount'];
        $payment->payment_status = $result['payment_status'];
        $payment->date_created = wp_date(get_option('date_format'), strtotime($result['p_date_created'] ?? ''));
        $payment->customer_name = $result['booking_name'];
        $payment->booking_date = wp_date(get_option('date_format'), strtotime($result['booking_date'] ?? ''));
        $payment->booking_type = $result['booking_type'];
        $payment->customer_id = $result['customer_id'];
        $payment->customer_gender = ucfirst(get_user_meta($result['customer_id'], 'gender', true) ?? "-");
        $payment->customer_birthday = get_user_meta($result['customer_id'], 'date_of_birth', true) ?? "-";
        $payment->prefix = $wpb_prefix;
        $payment->postfix = $wpb_postfix;

        $user_avatar_id = get_user_meta($payment->customer_id, "wp_user_avatar", true);
        if ($user_avatar_id) {
            $user__profile_url = wp_get_attachment_url($user_avatar_id);
        } else {
            $user__profile_url = get_avatar_url($payment->customer_id);
        }

        $payment->customer_avatar = $user__profile_url ? $user__profile_url : $user_avatar_id;

        $payments[] = $payment;
    }

    $payments = array_map(function ($payment) {
        $payment->customer_gender = $payment->customer_gender == false ? '-' : $payment->customer_gender;
        $payment->customer_birthday = $payment->customer_birthday == false ? '-' : $payment->customer_birthday;
        return apply_filters('wpb_payment_object', $payment);
    }, $payments);

    return array(
        'payments' => $payments,
        'total_count' => $total_count
    );
}



}
