<?php

if (!defined('ABSPATH')) {
    exit;
}

class WPB_Email_Notification_Handler {

    // List of actions and their respective callback functions and parameter counts
    private $actions = [
       
    ];

    public function __construct() {
        add_action( 'wpb_customer_booking_reminder', [ $this, 'trigger_customer_booking_email'], 10, 1 );
        add_action( 'wpb_customer_registration', [ $this, 'trigger_user_registration_email'], 10, 2 );
        add_action( 'wpb_staff_registration', [ $this, 'trigger_staff_registration_email'], 10, 2 );
        add_action( 'wpb_customer_cancel_booking', [ $this, 'trigger_cancellation_email'], 10, 1 );
        add_action( 'wpb_customer_booking_approval', [ $this, 'trigger_approval_email'], 10, 1 );
        add_action( 'wpb_customer_booking_confirmation', [ $this, 'trigger_confirmation_email'], 10, 1 );
        add_action( 'wpb_staff_booking_reminder', [ $this, 'trigger_staff_booking_email'], 10, 1 );
        add_action( 'wpb_staff_booking_request', [ $this, 'trigger_staff_booking_request_email'], 10, 1 );
        add_action( 'wpb_staff_booking_cancellation', [ $this, 'trigger_staff_booking_cancellation_email'], 10, 1 );
        // add_action('user_register', [$this, 'handle_user_registration'], 10, 2);
    }

    /**
     * Handle user registration to trigger appropriate email notifications.
     *
     * @param int $user_id
     * @param array $userdata
     */
    public function handle_user_registration($user_id, $userdata) {
        $user = get_userdata($user_id);

        // Check user role and trigger email accordingly
        if (in_array('customer', $user->roles)) {
            $this->trigger_user_registration_email($user, $userdata['user_pass']);
        } elseif (in_array('staff', $user->roles)) {
            $this->trigger_staff_registration_email($user, $userdata['user_pass']);
        }
    }

    /**
     * Fetch booking instances.
     *
     * @param int $booking_id
     * @return array
     */
    private function get_booking_instances($booking_id) {
        $booking = new WPB_Booking($booking_id);
        $booking_type_id = $booking->get_booking_type_id();
        $customer_id = $booking->get_customer_id();

        $wp_customer_instance = !empty($customer_id) ? new WP_User($customer_id) : (object) [
            'user_login' => $booking->get_booking_name(),
            'user_email' => $booking->get_booking_email(),
        ];

        $booking_type = new WPB_Booking_Type($booking_type_id);
        $staff_id = get_metadata('wpb_booking_type', $booking_type_id, 'staff', true);
        $wp_staff_instance = new WP_User($staff_id);

        return [$booking, $booking_type, $wp_customer_instance, $wp_staff_instance];
    }

    /**
     * Generic email trigger function.
     *
     * @param string $email_title
     * @param string $to_email
     * @param array $message_replacements
     * @param string $custom_email_content
     */
    private function trigger_email($email_title, $to_email, $message_replacements, $custom_email_content = '') {
        $email_data = self::get_email_data($email_title);
        if ($email_data->status) {
            $email_subject = $email_data->emails_subject;

            if (!empty($custom_email_content)) {
                $email_data->emails_content = $custom_email_content;
            }

            $email_message = $this->email_content_key_replace($email_data->emails_content, $message_replacements);
            wpb_mailer($to_email, $email_subject, $email_message);
        }
    }

    /**
     * Trigger customer booking email.
     *
     * @param int $booking_id
     */
    public function trigger_customer_booking_email($booking_id) {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);

        $this->trigger_email(
            'Customer Booking Reminder',
            $booking->get_booking_email(),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type]
        );
    }

    /**
     * Trigger approval email.
     *
     * @param int $booking_id
     */
    public function trigger_approval_email($booking_id) {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);

        $this->trigger_email(
            'Customer Booking Approval',
            $booking->get_booking_email(),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type]
        );
    }

    /**
     * Trigger confirmation email.
     *
     * @param int $booking_id
     */
    public function trigger_confirmation_email($booking_id) {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);
        $email_reminder = $booking_type->get_booking_type_meta('email_reminder');
        $custom_email_content = '';

        if ($email_reminder == 'true') {
            $custom_email_content = $booking_type->get_booking_type_meta('email_content_editor');
        }

        $this->trigger_email(
            'Customer Booking Confirmation',
            $booking->get_booking_email(),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type],
            $custom_email_content
        );

        $this->trigger_email(
            'Staff Booking Confirmation',
            apply_filters('wpb_staff_booking_confirmation_email',$wp_staff_instance->user_email),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type],
            $custom_email_content
        );
    }

    /**
     * Trigger cancellation email.
     *
     * @param int $booking_id
     */
    public function trigger_cancellation_email($booking_id) {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);

        $this->trigger_email(
            'Customer Booking Cancellation',
            $booking->get_booking_email(),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type]
        );
    }

    /**
     * Trigger user registration email for customers.
     *
     * @param WP_User $user
     * @param string $password
     */
    public function trigger_user_registration_email(WP_User $user, $password) {
        $this->trigger_email(
            'Customer Registration',
            $user->user_email,
            ['user' => $user, 'password' => $password]
        );
    }

    /**
     * Trigger user registration email for staff.
     *
     * @param WP_User $user
     * @param string $password
     */
    public function trigger_staff_registration_email(WP_User $user, $password) {
        $this->trigger_email(
            'Staff Registration',
            $user->user_email,
            ['user' => $user, 'password' => $password]
        );
    }

    /**
     * Trigger staff booking email.
     *
     * @param int $booking_id
     */
    public function trigger_staff_booking_email($booking_id) {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);

        $this->trigger_email(
            'Staff Booking Reminder',
            apply_filters('wpb_staff_booking_reminder_email',$wp_staff_instance->user_email),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type]
        );
    }

    /**
     * Trigger staff booking cancellation email.
     *
     * @param int $booking_id
     */
    public function trigger_staff_booking_cancellation_email($booking_id) {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);

        $this->trigger_email(
            'Staff Booking Cancellation',
            apply_filters('wpb_staff_booking_cancellation_email', $wp_staff_instance->user_email),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type]
        );
    }

    public function trigger_staff_booking_request_email($booking_id)
    {
        list($booking, $booking_type, $wp_customer_instance, $wp_staff_instance) = $this->get_booking_instances($booking_id);
        $this->trigger_email(
            'Staff Booking Request',
            apply_filters('wpb_staff_booking_request_email',$wp_staff_instance->user_email),
            ['user' => $wp_customer_instance, 'staff' => $wp_staff_instance, 'booking' => $booking, 'booking_type' => $booking_type]
        );
    }

    /**
     * Fetch email data from the database.
     *
     * @param string $emails_title
     * @return object
     */
     public static function get_email_data($emails_title) {
        global $wpdb;

        $sql = "SELECT * FROM {$wpdb->wpb_booking_emails} WHERE emails_title = %s";
        return $wpdb->get_row($wpdb->prepare($sql, $emails_title));
    }

    /**
     * Replace dynamic keys in the email content with actual values.
     *
     * @param string $email_message
     * @param array $replacements
     * @return string
     */
    private function email_content_key_replace($email_message, $replacements) {
        $site_path = home_url('/') . 'wp-admin/';
        $keys = [
            '{{customer_name}}' => ($replacements['user']->data->display_name) ?
                $replacements['user']->data->display_name : (isset($replacements['booking']) ? $replacements['booking']->get_booking_name() : ''),
            '{{customer_email}}' => (isset($replacements['user']->data->user_email)) ?
                $replacements['user']->data->user_email : (isset($replacements['booking']) ? $replacements['booking']->get_booking_email() : ''),
            '{{booking_type}}' => ($replacements['booking'] && $replacements['booking']->get_booking_type()) ? $replacements['booking']->get_booking_type()['name'] : '',
            '{{booking_status}}' => ($replacements['booking'] && $replacements['booking']->get_status('view')) ? (ucfirst(str_replace('wpb-', '', $replacements['booking']->get_status('view')))) : '',
            '{{booking_date}}' => ($replacements['booking'] && $replacements['booking']->get_booking_date()) ? wpb_get_formated_date($replacements['booking']->get_booking_date()) : '',
            '{{booking_questions_ans}}' => $replacements['booking'] ? $this->get_questions_ans(booking: $replacements['booking']) : '',
            '{{booking_time}}' => ($replacements['booking'] && $replacements['booking']->get_timeslot('view', true)) ? wpb_get_formated_time($replacements['booking']->get_timeslot('view', true)) : '',
            '{{staff_name}}' => ($replacements['staff'] && $replacements['staff']->data->display_name) ? $replacements['staff']->data->display_name : '',
            '{{login_url}}' => get_first_wpb_profile_page_url() ?? '',
            '{{meeting_url}}' => ($replacements['booking'] && get_metadata('wpb_bookings', $replacements['booking']->get_id(), 'location_source', true)) ? get_metadata('wpb_bookings', $replacements['booking']->get_id(), 'location_source', true) : '',
            '{{customer_phone_no}}' => ( empty(get_user_meta($replacements['user']->ID, "phone", true)) ? '-' : get_user_meta($replacements['user']->ID, "phone", true)).( empty(get_user_meta($replacements['user']->ID, "dialCode", true)) ?  '' : get_user_meta($replacements['user']->ID, "dialCode", true)),
            '{{password}}' => isset($replacements['password']) ? $replacements['password'] : '',
            '{{staff_login}}' => $site_path,
        ];

        $keys = apply_filters('wpb_template_dynamic_keys_value', $keys, $email_message, $replacements);
        return str_replace(array_keys($keys), array_values($keys), $email_message);
    }
    private function get_questions_ans($booking) {
        $questions_data = json_decode($booking->get_meta('questions_answers') ?? '{}', true);
        
        if (empty($questions_data)) {
            return '';
        }
        
        $questions_data = array_map(function ($item) {
            $item['question'] = wpb_unicode_to_utf8($item['question']);
            $item['ans'] = wpb_unicode_to_utf8($item['ans']);
            return $item;
        }, $questions_data);
        
        $html_output = '';
        
        foreach ($questions_data as $question) {
            $html_output .= '<p><strong>' . htmlspecialchars($question['question']) . '</strong></p>';
            $html_output .= '<p>' . htmlspecialchars($question['ans']) . '</p>';
        }
        
        return $html_output;
    }
}

// Initialize the email notification handler
new WPB_Email_Notification_Handler();
