<?php

/**
 * Get all Booking statuses.
 * @return array
 **/

function wpb_get_booking_statuses() {
    return apply_filters(
        'wpb_booking_statuses',
        array(
            'wpb-pending'   => _x( 'Pending', 'Booking status', 'wpbookit' ), // warning
            'wpb-approved'  => _x( 'Approved', 'Booking status', 'wpbookit' ), // info
            'wpb-cancelled' => _x( 'Cancelled', 'Booking status', 'wpbookit' ), // danger
            'wpb-completed' => _x( 'Completed', 'Booking status', 'wpbookit' ), // primary
        )
    );
}

function wpb_get_booking_payment_statuses() {
    return apply_filters(
        'wpb_booking_payment_statuses',
        array(
            '1'   => _x( 'Paid', 'Payment status', 'wpbookit' ),
            '0'   => _x( 'Unpaid', 'Payment status', 'wpbookit' ),
        )
    );
}

function wpb_get_booking_modes() {
    $payment_mode=[];
    foreach (get_option('wpb_offline_payment_modes',[]) as $key => $value) {
        $payment_mode[str_replace(' ','-',strtolower($value['name']))]=$value['name'];
    }
    $payments = apply_filters('wpb_booking_shortcode_active_payment_gateway',$payment_mode);

    return $payments;
}

/**
 * See if a string is an booking status.
 *
 * @param  string $has_status Status, including any wpb-prefix.
 * @return bool
 */
function wpb_is_booking_status($has_status) {
    $booking_statuses = wpb_get_booking_statuses();
    return isset( $booking_statuses[$has_status] );
}

/**
 * Get a string is an booking status.
 *
 * @param  string $has_status Status, including any wpb-prefix.
 * @return bool
 */
function wpb_booking_status_label($statusKey) {
    $booking_statuses = wpb_get_booking_statuses();
    return isset( $booking_statuses[$statusKey] ) ? $booking_statuses[$statusKey] : false;
}

/**
 * Main function for returning booking, uses the WPB_Booking class.
 * @return bool|WPB_Booking
 */
function wpb_get_bookings($args = array()) {  
    // Default arguments
    $defaults = array(
        'user_id'       => 0,
        'paged'         => 1,
        'per_page'      => 31,
        'status'        => [],
        'booking_type'  => [],
        'date_from'     => '',
        'date_to'       => '',
        'order'         => 'DESC',
        'order_by'      => 'booking_date',
        'booking_name'  => '',
        'staff'         => 0,
        'offset'        => '',
        'is_paid'       => false    
    );
    $args = wp_parse_args($args, $defaults);
    return (new WPB_Booking)->get_bookings( $args );
}

/**
 * Main function for returning booking, uses the WPB_Booking class.
 * @return bool|WPB_Booking
 */
function wpb_get_booking($booking = false) {
    if (!$booking)
        return $booking;
    return (new WPB_Booking)->get_booking($booking);
}
/**
 * Deletes a specific booking along with its associated meta data and payments.
 *
 * This function removes a booking record from the database, including related meta data and payments.
 * It also triggers actions for before and after booking deletion.
 *
 * @since 1.0.0
 * @global wpdb $wpdb WordPress database abstraction object
 *
 * @param int $booking_id The ID of the booking to delete.
 * @return array An associative array containing the status and message of the operation.
 *              - 'status': 'success' | 'error'
 *              - 'message': String describing the result of the operation
 *
 * @throws Exception If the booking_id is not valid or the deletion process fails.
 *
 * @example
 **/
function wpb_delete_booking(int $booking_id):array{
    global $wpdb;
    do_action('wpbookit_booking_deleted', $booking_id); //remove this hooks once hooks dependency removed from addons.
    do_action('wpb_after_booking_deleted', $booking_id);
    
    if ($booking_id) :
        $resposne = $wpdb->delete(
            $wpdb->wpb_bookings,
            array('id' => $booking_id),
            array('%d')
        );

        $wpdb->delete(
            $wpdb->wpb_bookingsmeta,
            array('wpb_bookings_id' => $booking_id),
            array('%d')
        );
        $wpdb->delete(
            $wpdb->wpb_payments,
            array('bookings_id' => $booking_id),
            array('%d')
        );

        if ($resposne) :
            return [
                'status'  => 'success',
                'message' => esc_html__('Booking Deleted Successfully.', 'wpbookit')
            ];
        else :
            return [
                'status' => 'error',
                'message' => esc_html__('Error while deleting booking.', 'wpbookit')
            ];
        endif;
    else :
        return array(
            'status'  => 'success',
            'message' => esc_html__('Booking Deleted Successfully.', 'wpbookit')
        );
    endif;
}

/**
 * Fetch booking instances.
 *
 * @param int $booking_id
 * @return array
 */
function wpb_get_booking_instances($booking_id) {
    $booking = new WPB_Booking($booking_id);
    $booking_type_id = $booking->get_booking_type_id();
    $customer_id = $booking->get_customer_id();

    $wp_customer_instance = !empty($customer_id) ? new WP_User($customer_id) : (object) [
        'user_login' => $booking->get_booking_name(),
        'user_email' => $booking->get_booking_email(),
    ];

    $booking_type = new WPB_Booking_Type($booking_type_id);
    $staff_id = get_metadata('wpb_booking_type', $booking_type_id, 'staff', true);
    $wp_staff_instance = new WP_User($staff_id);

    return [$booking, $booking_type, $wp_customer_instance, $wp_staff_instance];
}