<?php
if (!function_exists('iqwpb_get_active_taxs')) {
    /**
     * Fetch tax data based on booking type ID.
     *
     * @param int $booking_type_id The ID of the booking type.
     * @return array Array of tax data.
     **/
    function iqwpb_get_active_taxs($booking_type_id)
    {
        global $wpdb;
        $table_name = $wpdb->wpb_tax;
        $include_exclude_tax=  (wpb_get_general_settings()['include_exclude_tax']??"incl" )=="incl"?"yes":"no" ;
        $query = $wpdb->prepare(
            "SELECT * FROM $table_name WHERE (FIND_IN_SET('0', tax_booking_type) OR FIND_IN_SET(%s, tax_booking_type)) AND status = '1' AND inclusive_tax LIKE %s",
            $booking_type_id,
            $include_exclude_tax
        );
        return $wpdb->get_results($query, ARRAY_A);
    }
}

if (!function_exists('iqwpb_process_tax_data')) {
    /**
     * Process tax data and calculate totals.
     *
     * @param array $results Array of tax data fetched from the database.
     * @param float $original_price The original price of the booking.
     * @return array Processed tax data along with main total and price excluding tax.
     **/
    function iqwpb_process_tax_data($tax, $total_price) {
        $general_setting    = wpb_get_general_settings();
        $inclusive_tax      = isset( $general_setting['include_exclude_tax'] ) && $general_setting['include_exclude_tax'] =="incl" ? true : false;
        $tax_data = array();
        
        // Sort the tax array based on tax priority
        usort($tax, function($a, $b) {
            return $a['tax_priority'] <=> $b['tax_priority'];
        });

        if ( $inclusive_tax ) :
            $taxes = iqwpb_calc_inclusive_tax( $total_price, $tax );
            $price_excluding_tax = $total_price;
        else :
            $taxes = iqwpb_calc_exclusive_tax( $total_price, $tax );
        endif;        // Calculate original prices and tax amounts
        
        $total_tax  = array_sum( (array)array_column( $taxes, 'tax_amount' ));
        $price_including_tax = $inclusive_tax ? $total_price : $total_price + $total_tax;
        $price_excluding_tax = $inclusive_tax ? $total_price - $total_tax : $total_price;
        // Format the prices

        return array(
            'tax_data'              => $taxes,
            'main_total'            => number_format($total_price, 2),
            'price_excluding_tax'   => $price_excluding_tax,
            'price_including_tax'   => $price_including_tax
        );
    }
    
    
}

if (!function_exists('iqwpb_calc_inclusive_tax')) {
    /**
     * Calculate inclusive taxes for a given price and tax rates.
     *
     * This function accepts a price and a list of tax rates (both fixed and regular), 
     * and calculates the inclusive tax amount for each rate. It returns the updated 
     * tax rates with the corresponding tax amounts.
     *
     * @param float $price The original price before tax.
     * @param array $rates Array of tax rates, each containing tax type and rate.
     * @return array Updated rates with tax amounts.
     */
    function iqwpb_calc_exclusive_tax( $price, $rates ) {
        // Convert price to cents (100ths of the currency unit) for precise calculation
        $price = $price * 100;
        foreach ( $rates as $key => $rate ) :
            if( $rate['tax_type'] == 'fixed' ) :
                $rates[$key]['tax_amount'] = number_format( $rate['tax_rate'], 2 );   // Set the fixed tax amount
            else :
                $tax_amount = $price * ( floatval( $rate['tax_rate'] ) / 100 );
                $rates[ $key ]['tax_amount'] = number_format( (float) $tax_amount/100, 2);
            endif;
        endforeach;

        return $rates;
    }

}

if (!function_exists('iqwpb_calc_inclusive_tax')) {
    /**
     * Calculate inclusive taxes for a given price and tax rates.
     *
     * This function accepts a price and a list of tax rates (both fixed and regular), 
     * and calculates the inclusive tax amount for each rate. It returns the updated 
     * tax rates with the corresponding tax amounts.
     *
     * @param float $price The original price before tax.
     * @param array $rates Array of tax rates, each containing tax type and rate.
     * @return array Updated rates with tax amounts.
     */
    function iqwpb_calc_inclusive_tax( $price, $rates ) {
        // Convert price to cents (100ths of the currency unit) for precise calculation
        $price = $price * 100;
        // Arrays to store fixed and regular tax rates
        $regular_rates = array();

        // Loop through each tax rate to classify and process fixed and regular rates
        foreach ( $rates as $key => $rate ) :
            // Check if the rate is a fixed tax rate
            if( $rate['tax_type'] == 'fixed' ) :
                $rates[$key]['tax_amount'] = number_format( $rate['tax_rate'], 2 );  // Set the fixed tax amount
                $price -= $rate['tax_rate'] * 100;  // Deduct the fixed tax from the price
            else :
                // Regular (percentage-based) tax rates are stored separately
                $regular_rates[$key] = $rate['tax_rate'];
            endif;
        endforeach;

        // Calculate the cumulative regular tax rate (in decimal form)
        $regular_tax_rate = 1 + (array_sum($regular_rates) / 100);

        // Loop through regular tax rates to calculate the inclusive tax amounts
        foreach ( $regular_rates as $key => $regular_rate ) :
            // Calculate the net price (excluding the regular tax) for each regular tax rate
            $the_rate = ( $regular_rate / 100 ) / $regular_tax_rate;
            $net_price = $price - ( $the_rate * $price );  // Subtract the tax portion from the price
            $tax_amount = $price - $net_price;  // The tax amount is the difference between price and net price

            // Update the rate with the calculated tax amount, formatted to 2 decimal places
            $rates[$key]['tax_amount'] = number_format($tax_amount / 100, 2);
        endforeach;

        // Return the updated rates array with calculated tax amounts
        return $rates;
    }

}

if (!function_exists('iqwpb_generate_html_output')) {
    /**
     * Generate HTML output for taxes and totals.
     *
     * @param array $tax_data Array of tax data to display.
     * @param float $price_excluding_tax Price excluding tax.
     * @param float $main_total Total price including all taxes.
     * @param float $original_price Original price of the booking.
     * @return string HTML output for displaying taxes and totals.
     **/
    function iqwpb_generate_html_output($tax_data, $price_excluding_tax, $price_including_tax, $main_total, $original_price, $coupon_price='',$booking_id=0)
    {
        $tax_values = array();
        if (!empty($original_price)) :
            $html_output = sprintf(
                '<div class="d-flex justify-content-between booking_price">
                    <h6 class="mb-0">%s:</h6>
                    <p class="m-0"><span class="text-primary price_text">%s</span></p>
                </div>',
                apply_filters('wpb_sub_total_label',esc_html__("Sub Total", 'wpbookit')),
                apply_filters('wpb_sub_total_value_with_prefix', wpb_get_prefix_postfix_price( number_format($price_excluding_tax, 2,'.','') ),$price_excluding_tax,$booking_id ),
            );

            $tax_values[apply_filters('wpb_sub_total_value_key','sub_total')] = apply_filters('wpb_sub_total_value', number_format($price_excluding_tax, 2,'.','') ,$price_excluding_tax,$booking_id );
   
            if(count($tax_data)>=1){
                $html_output.='<hr/>';
            }

            if( is_array( $tax_data ) ) {
                foreach ($tax_data as $tax) {
                    if(!is_array($tax)) continue;
                    $class_name = 'booking_' . (str_replace(' ', '_', $tax['tax_name']??""));
                    if ($tax['tax_type'] == 'fixed') {
                        $tax_amount = $tax['tax_amount'];
                        $key = (preg_replace('/[\s()-]+/', '_', $tax['tax_name']));
                        $tax_values[$key] = number_format($tax_amount, 2,'.','');
                        $html_output .= sprintf(
                            '<div class="d-flex justify-content-between mt-3 %s">
                                <h6 class="mb-0">%s:</h6>
                                <p class="m-0"><span class="text-primary price_text">%s</span></p>
                            </div>',
                            esc_html($class_name),
                            esc_html($tax['tax_name']),
                            wpb_get_prefix_postfix_price($tax['tax_amount']),
                        );
                    } else {
                        $key = ( preg_replace('/[\s()-]+/', '_', $tax['tax_name'] ) );
                        $tax_values[$key] = number_format($tax['tax_amount'], 2,'.','');
                        $html_output .= sprintf(
                            '<div class="d-flex justify-content-between mt-3 %s">
                                <h6 class="mb-0">%s:</h6>
                                <p class="m-0"><span class="text-primary price_text">%s</span></p>
                            </div>',
                            esc_html($class_name),
                            esc_html($tax['tax_name']),
                            wpb_get_prefix_postfix_price($tax['tax_amount']),
                        );
                    }
                }
            }

            $html_output .= sprintf(
                '<hr><div class="d-flex justify-content-between mt-3 main_booking_total">
                    <h6 class="mb-0">%s:</h6>
                    <p class="m-0"><span class="text-primary price_text">%s</span></p>
                </div>',
                esc_html__("Total", 'wpbookit'),
                wpb_get_prefix_postfix_price($price_including_tax ),
            );


            $tax_values['total'] = is_numeric($price_including_tax)? number_format($price_including_tax, 2,'.',''):$price_including_tax;

        else :
            $html_output = sprintf(
                '<div class="d-flex justify-content-between main_booking_total">
                    <h6 class="mb-0">%s:</h6>
                    <p class="m-0"><span class="text-primary price_text">%s</span></p>
                </div>',
                esc_html__("Total", 'wpbookit'),
                esc_html__("Free", 'wpbookit'),
            );

            $tax_values['total'] = esc_html__("Free", 'wpbookit');
        endif;

        return array(
            'html_output' => $html_output,
            'tax_values' => $tax_values
        );
    }
}

if (!function_exists('iqwpb_get_booking_totals_and_html')) {
    /**
     * Get totals and generate HTML output for booking taxes and totals.
     *
     * @param int $booking_type_id The ID of the booking type.
     * @param float $original_price The original price of the booking.
     * @return array Array containing the HTML output and the tax values.
     */
    function iqwpb_get_booking_totals_and_html($booking_type_id, $original_price,$booking_id='',$is_price_already_exclude_with_tax=false,$total_paid=false,$coupon_code=false)
    {
        // Fetch tax data

        if($booking_id==''){
            $tax_results = iqwpb_get_active_taxs($booking_type_id);
        }else{
            $tax_results =  json_decode(get_metadata('wpb_bookings', $booking_id, 'tax_data', true),true)['tax_data'] ?? [];
        }
        $coupen_results = '';
        
        $coupen_results = apply_filters('wpb_before_add_tax_check_coupon',$coupon_code,$booking_id,$original_price,$booking_type_id);
        // Process tax data
        $processed_data         = iqwpb_process_tax_data($tax_results, $original_price);
        $tax_data               = isset($processed_data['tax_data']) ? $processed_data['tax_data'] : array();
        $main_total             = isset($processed_data['main_total']) ? $processed_data['main_total'] : 0;
        $price_excluding_tax    = isset($processed_data['price_excluding_tax']) ? $processed_data['price_excluding_tax'] : 0;
        $price_including_tax    = isset($processed_data['price_including_tax']) ? $processed_data['price_including_tax'] : 0;

        if( $is_price_already_exclude_with_tax ) :
            $price_excluding_tax = $original_price;
        endif;
        // Generate HTML output and get tax values
        $html_and_values = iqwpb_generate_html_output($tax_data, $price_excluding_tax, $price_including_tax, $main_total, $original_price,'',$booking_id);
        if(has_filter('wpb_before_add_tax_check_coupon') && $coupen_results !== false){
            $html_and_values = $coupen_results;
        }
        return $html_and_values;
    }
}
