<?php

namespace WeDevs\DokanPro\Modules\RMA\Emails;

use WC_Email;

defined( 'ABSPATH' ) || exit;

/**
 * Dokan Send Coupon Email
 *
 * An email sent to the admin and customer when a vendor generate a coupon for customer
 *
 * @class       Dokan_Send_Coupon_Email
 * @version     2.9.3
 * @author      weDevs
 * @extends     WC_Email
 */
class SendCouponEmail extends WC_Email {

        /**
         * Constructor.
         */
        public function __construct() {
            $this->id             = 'Dokan_Send_Coupon_Email';
            $this->title          = __( 'Dokan Send Coupon To Customer', 'dokan' );
            $this->description    = __( 'This email send to customer when customer send return request and vendor send a store credit to customer', 'dokan' );
            $this->template_base  = DOKAN_RMA_DIR . '/templates/';
            $this->template_html  = 'emails/send-coupon.php';
            $this->template_plain = 'emails/plain/send-coupon.php';
            $this->customer_email = true;
            $this->placeholders   = array(
                '{vendor_name}' => '',
                '{coupon_code}' => '',
                '{rma_id}'      => '',
                // Only for backward compatibility.
                '{site_name}'   => $this->get_from_name(),
            );

            // Triggers for this email
            add_action( 'dokan_send_coupon_to_customer', [ $this, 'trigger' ], 30, 2 );

            // Call parent constructor
            $this->recipient = 'customer@oftheorder';
            parent::__construct();
        }

        /**
         * Get email subject.
         *
         * @since  3.1.0
         * @return string
         */
        public function get_default_subject() {
            return __( '[{site_title}] A new coupon is generated by ({vendor_name})', 'dokan' );
        }

        /**
         * Get email heading.
         *
         * @since  3.1.0
         * @return string
         */
        public function get_default_heading() {
            return __( 'New Coupon is Generated', 'dokan' );
        }

        /**
         * Trigger the sending of this email.
         *
         * @param WC_Coupon $coupon The product ID.
         * @param array $data.
         */
        public function trigger( $coupon, $data ) {
            if ( ! $this->is_enabled() ) {
                return;
            }

            $this->object = $coupon;

            if ( ! $data ) {
                return;
            }

            $this->setup_locale();
            $rma_id         = $data['request_id'] ?? '';
            $vendor_id      = $data['refund_vendor_id'] ?? '';
            $vendor         = dokan()->vendor->get( $vendor_id );
            $customer_email = $coupon->get_email_restrictions();
            $customer_email = is_array( $customer_email ) ? $customer_email[0] : $customer_email;

            $this->placeholders['{vendor_name}'] = $vendor->get_name();
            $this->placeholders['{rma_id}']      = $rma_id;
            $this->placeholders['{coupon_code}'] = $coupon->get_code();
            $this->recipient = $customer_email;

            $this->send( "{$customer_email}, {$this->get_recipient()}", $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
            $this->restore_locale();
        }

        /**
         * Get content html.
         *
         * @access public
         *
         * @return string
         */
        public function get_content_html() {
            return wc_get_template_html(
                $this->template_html,
                array(
                    'coupon'             => $this->object,
                    'email_heading'      => $this->get_heading(),
                    'additional_content' => $this->get_additional_content(),
                    'plain_text'         => false,
                    'email'              => $this,
                    'data'               => $this->placeholders,
                ),
                'dokan/',
                $this->template_base
            );
        }

        /**
         * Get content plain.
         *
         * @access public
         *
         * @return string
         */
        public function get_content_plain() {
            return wc_get_template_html(
                $this->template_plain,
                array(
                    'coupon'             => $this->object,
                    'email_heading'      => $this->get_heading(),
                    'additional_content' => $this->get_additional_content(),
                    'plain_text'         => true,
                    'email'              => $this,
                    'data'               => $this->placeholders,
                ),
                'dokan/',
                $this->template_base
            );
        }


        /**
         * Initialise settings form fields.
         */
        public function init_form_fields() {
            $placeholders = $this->placeholders;
            unset( $placeholders['{site_name}'] );
            /* translators: %s: list of placeholders */
            $placeholder_text  = sprintf( __( 'Available placeholders: %s', 'dokan' ), '<code>' . implode( '</code>, <code>', array_keys( $placeholders ) ) . '</code>' );
            $this->form_fields = array(
                'enabled' => array(
                    'title'         => __( 'Enable/Disable', 'dokan' ),
                    'type'          => 'checkbox',
                    'label'         => __( 'Enable this email notification', 'dokan' ),
                    'default'       => 'yes',
                ),
                'recipient' => array(
                    'title'         => __( 'Recipient(s)', 'dokan' ),
                    'type'          => 'text',
                    // translators: Default email address.
                    'description'   => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.', 'dokan' ), '<code>' . esc_attr( $this->recipient ) . '</code>' ),
                    'placeholder'   => $this->recipient,
                    'default'       => '',
                    'desc_tip'      => true,
                ),
                'subject' => array(
                    'title'         => __( 'Subject', 'dokan' ),
                    'type'          => 'text',
                    'desc_tip'      => true,
                    'description'   => $placeholder_text,
                    'placeholder'   => $this->get_default_subject(),
                    'default'       => '',
                ),
                'heading' => array(
                    'title'         => __( 'Email heading', 'dokan' ),
                    'type'          => 'text',
                    'desc_tip'      => true,
                    'description'   => $placeholder_text,
                    'placeholder'   => $this->get_default_heading(),
                    'default'       => '',
                ),
                'additional_content' => array(
                    'title'       => __( 'Additional content', 'dokan' ),
                    'description' => __( 'Text to appear below the main email content.', 'dokan' ) . ' ' . $placeholder_text,
                    'css'         => 'width:400px; height: 75px;',
                    'placeholder' => __( 'N/A', 'dokan' ),
                    'type'        => 'textarea',
                    'default'     => $this->get_default_additional_content(),
                    'desc_tip'    => true,
                ),
                'email_type' => array(
                    'title'         => __( 'Email type', 'dokan' ),
                    'type'          => 'select',
                    'description'   => __( 'Choose which format of email to send.', 'dokan' ),
                    'default'       => 'html',
                    'class'         => 'email_type wc-enhanced-select',
                    'options'       => $this->get_email_type_options(),
                    'desc_tip'      => true,
                ),
            );
        }
    }
