"use strict";
(function ($) {
	const initPower = () => {
		function countIndentationLevelChange(str) {
			let smatches = str.match(/{[A-Za-z\-][A-Za-z0-9\-_]*:/g);
			let ematches = str.match(/}/g);
			let sn = smatches ? smatches.length : 0;
			let en = ematches ? ematches.length : 0;
			return sn - en;
		}

		function formatJsString(input) {
			let lines = input.split("\n");
			let currentIndent = 0;
			const indentString = "    "; // 4 spaces for indentation

			lines = lines.map((l) => {
				l = l.trim();
				if (l === "}") {
					currentIndent = Math.max(currentIndent - 1, 0);
					return indentString.repeat(currentIndent) + l;
				}
				l = indentString.repeat(currentIndent) + l;
				let diff = countIndentationLevelChange(l);
				currentIndent = currentIndent + diff;
				currentIndent = Math.max(currentIndent, 0);
				return l;
			});
			return lines.join("\n");
		}
		const form = document.getElementById("power-shortcodes-form");
		if (!form) {
			return;
		}
		let licenseActive = form.dataset.licenseActive === "yes";
		const fieldWrapper = document.getElementById(
			"power-shortcodes-wrapper",
		);
		const saveButtons = document.getElementsByClassName(
			"power-shortcodes-save",
		);

		const formMessage = document.getElementById(
			"power-shortcodes-form-message",
		);
		const oldJson = form.elements.power_shortcodes.value;
		const itemTemplate = document.getElementById(
			"power-shortcodes-template",
		);
		const repeaterWrapper = document.getElementById(
			"power-shortcodes-repeater",
		);
		const addButton = document.getElementById("power-shortcodes-add");
		const expandAllLink = document.getElementById("power-shortcodes-expand-all");
		const collapseAllLink = document.getElementById("power-shortcodes-collapse-all")

		let isDirty = false;

		const getCurrentStateArray = () => {
			const items = document.getElementsByClassName(
				"power-shortcodes-repeater-item",
			);
			let res = [];
			for (const item of items) {
				const name = item.getElementsByClassName("power-shortcodes-name")[0]
					.value;
				const description =
					item.getElementsByClassName("power-shortcodes-description")[0]?.value ||
					"";
				const code = item.getElementsByClassName("power-shortcodes-code")[0].value;
				res.push({ name: name, description: description, code: code });
			}
			return res;
		};

		const getCurrentJsonString = () => {
			try {
				return JSON.stringify(getCurrentStateArray());
			} catch (e) {
				return "";
			}
		};

		const setDirty = (dirty) => {
			isDirty = !!dirty;
		};

		const updateStatusCounters = () => {};

		// Search / Filter via shared utility
		const searchInput = document.getElementById("power-shortcodes-search");
		const noResultsBox = document.getElementById("power-shortcodes-no-results");
		const getItemText = (li) => {
			const summary = li.querySelector(".power-shortcodes-summary-name");
			const nameInput = li.querySelector(".power-shortcodes-name");
			const descTextarea = li.querySelector(".power-shortcodes-description");
			const codeTextarea = li.querySelector(".power-shortcodes-code");
			return [
				summary ? summary.textContent : "",
				nameInput ? nameInput.value : "",
				descTextarea ? descTextarea.value : "",
				codeTextarea ? codeTextarea.value : "",
			].join(" ");
		};
		const toggleItem = (li, visible) => {
			li.classList.toggle("hidden", !visible);
		};
		const handleEmpty = (anyVisible, total) => {
			if (!noResultsBox) return;
			if (total > 0 && !anyVisible) {
				noResultsBox.classList.remove("hidden");
			} else {
				noResultsBox.classList.add("hidden");
			}
		};
		const runFilter = () => {
			if (!repeaterWrapper) return;
			const items = repeaterWrapper.querySelectorAll(
				"li.power-shortcodes-repeater-item",
			);
			window.DSHFilters.applyFilter(
				items,
				scrollInputValue(searchInput),
				getItemText,
				toggleItem,
				handleEmpty,
			);
		};
		function scrollInputValue(input){ return (input && input.value) ? input.value : ""; }
		if (searchInput) {
			searchInput.addEventListener("input", runFilter);
		}

		let oldValue = [];
		if (typeof oldJson === "string" && oldJson !== "") {
			try {
				oldValue = JSON.parse(oldJson);
			} catch (e) {
				console.error("json parse error");
			}
		}

		const showFormError = (message) => {
			formMessage.textContent = message;
			formMessage.classList.remove("hidden");
			formMessage.classList.remove("notice-success");
			formMessage.classList.add("notice-error");
		};

		const showFormSuccess = (message) => {
			formMessage.textContent = message;
			formMessage.classList.add("notice-success");
			formMessage.classList.remove("notice-error");
			formMessage.classList.remove("hidden");
		};

		const showItemsErrors = (errors) => {
			const errorLists = repeaterWrapper.getElementsByClassName(
				"power-shortcodes-item-error",
			);
			for (let i = 0; i < errorLists.length; i++) {
				errorLists[i].innerHTML = "";
			}
			for (let e of errors) {
				const wrapper = repeaterWrapper.childNodes[e.index];
				const ul = wrapper.getElementsByClassName(
					"power-shortcodes-item-error",
				)[0];
				const li = document.createElement("li");
				li.textContent = e.message;
				ul.appendChild(li);
				ul.classList.remove("hidden");
				// Link errors to inputs via aria-describedby
				try {
					const errId = `power-error-${e.index}-${ul.children.length}`;
					li.id = errId;
					const nameInput = wrapper.getElementsByClassName("power-shortcodes-name")[0];
					const desc = wrapper.getElementsByClassName("power-shortcodes-description")[0];
					const code = wrapper.getElementsByClassName("power-shortcodes-code")[0];
					const target = nameInput || desc || code;
					if (target) {
						const prev = target.getAttribute("aria-describedby");
						target.setAttribute(
							"aria-describedby",
							prev ? `${prev} ${errId}` : errId,
						);
					}
				} catch (ex) {}
			}
			// Scroll to first error
			if (Array.isArray(errors) && errors.length > 0) {
				try {
					const idx = Math.max(0, Number(errors[0].index || 0));
					const item = repeaterWrapper.childNodes[idx];
					if (item) {
						const details = item.getElementsByClassName("power-shortcodes-details")[0];
						if (details) details.open = true;
						item.scrollIntoView({ behavior: "smooth", block: "center" });
						// Focus preferred field
						const nameInput = item.getElementsByClassName("power-shortcodes-name")[0];
						const desc = item.getElementsByClassName("power-shortcodes-description")[0];
						const code = item.getElementsByClassName("power-shortcodes-code")[0];
						const target = nameInput || desc || code;
						if (target) {
							target.focus({ preventScroll: true });
							if (target.select) target.select();
						}
						flashError(item);
					}
				} catch (err) {}
			}
		};

		const toast = document.getElementById("power-toast");
		const toastBackdrop = document.getElementById("power-toast-backdrop");
		const showToast = (message, type) => {
			if (!toast) return;
			toast.textContent = message || "";
			toast.classList.remove("hidden", "error", "success");
			toast.classList.add(type === "error" ? "error" : "success");
			if (toastBackdrop) toastBackdrop.classList.remove("hidden");
			setTimeout(() => {
				toast.classList.add("hidden");
				if (toastBackdrop) toastBackdrop.classList.add("hidden");
			}, 2200);
		};

		const saveCallback = (e) => {
			const httpRequest = e.target;
			if (httpRequest.readyState === XMLHttpRequest.DONE) {
				if (httpRequest.status === 200) {
					var response;
					try {
						response = JSON.parse(httpRequest.responseText);
					} catch (e) {
						showFormError("Response Error");
						showToast("Response Error", "error");
						return;
					}
					if (!response?.success) {
						showFormError(response?.data.message);
						showToast(response?.data.message || "Save failed", "error");
						showItemsErrors(response?.data.items_errors);
					} else {
						showFormSuccess(response?.data.message);
						showToast(response?.data.message || "Saved", "success");
						// Reset dirty baseline and flag
						try { oldJson = getCurrentJsonString(); } catch (e) {}
						setDirty(false);
						if (pendingRedirectToList) {
							pendingRedirectToList = false;
							try {
								const href = String(window.location.href || "");
								window.location.href = href.replace(
									"dynamic-shortcodes-power",
									"dynamic-shortcodes",
								);
							} catch (e) {
								window.location.href = "admin.php?page=dynamic-shortcodes";
							}
						}
					}
				} else {
					showFormError(
						"There was a problem with the admin ajax request.",
					);
					showToast("Network error", "error");
				}
			}
		};

		for (let saveButton of saveButtons) {
			saveButton.addEventListener("click", (event) => {
				// Hide all previous message:
				formMessage.classList.add("hidden");
				const divs = document.getElementsByClassName(
					"power-shortcodes-item-error",
				);
				for (let div of divs) {
					div.classList.add("hidden");
				}
				const items = document.getElementsByClassName(
					"power-shortcodes-repeater-item",
				);
				let res = [];
				for (const item of items) {
					const name = item.getElementsByClassName(
						"power-shortcodes-name",
					)[0].value;
					const description = item.getElementsByClassName(
						"power-shortcodes-description",
					)[0]?.value || "";
					const code = item.getElementsByClassName(
						"power-shortcodes-code",
					)[0].value;
					res.push({ name: name, description: description, code: code });
				}
				const value = JSON.stringify(res);
				let data = new FormData(form);
				data.set("power_shortcodes", value);
				const httpRequest = new XMLHttpRequest();
				httpRequest.onreadystatechange = saveCallback;
				const url = form.getAttribute("action");
				httpRequest.open("POST", url, true);
				httpRequest.send(data);
			});
		}

		const addRepeaterItem = (values, insertAt) => {
			let isNew = false;
			if (typeof values === "undefined") {
				isNew = true;
				values = { name: "", description: "", code: "" };
			}
			const repeaterItem =
				itemTemplate.content.firstElementChild.cloneNode(true);
			if (isNew) {
				repeaterItem.dataset.testid = "new-power";
			}
			const usageInfo = repeaterItem.getElementsByClassName(
				"power-shortcodes-usage",
			)[0];
			const details = repeaterItem.getElementsByClassName(
				"power-shortcodes-details",
			)[0];
			const summaryName = repeaterItem.getElementsByClassName(
				"power-shortcodes-summary-name",
			)[0];
			const nameInput = repeaterItem.getElementsByClassName(
				"power-shortcodes-name",
			)[0];
			nameInput.value = values.name;
			summaryName.textContent = values.name || "Untitled";
			const descriptionArea = repeaterItem.getElementsByClassName(
				"power-shortcodes-description",
			)[0];
			descriptionArea.value = values.description || "";
			let codeArea = repeaterItem.getElementsByClassName(
				"power-shortcodes-code",
			)[0];
			codeArea.value = values.code;
			const delButton = repeaterItem.getElementsByClassName(
				"power-shortcodes-del",
			)[0];
			const copyButton = repeaterItem.getElementsByClassName(
				"power-shortcodes-copy",
			)[0];
			const formatButton = repeaterItem.getElementsByClassName(
				"power-shortcodes-format",
			)[0];
			if (!licenseActive) {
				nameInput.disabled = true;
				descriptionArea.disabled = true;
				codeArea.disabled = true;
				delButton.disabled = true;
				formatButton.disabled = true;
				copyButton.disabled = true;
			}
			const updateUsageInfo = (name) => {
				name = name.trim();
				let msg = "";
				if (/^[a-zA-Z_$][a-zA-Z0-9_\-$]*$/.test(name)) {
					msg = `You can use it with <code>{power:${name}}</code>`;
				}
				usageInfo.innerHTML = msg;
			};
			updateUsageInfo(values.name);
			formatButton.addEventListener("click", () => {
				codeArea.value = formatJsString(codeArea.value);
			});
			nameInput.addEventListener("input", (e) => {
				const name = e.target.value;
				updateUsageInfo(name);
				summaryName.textContent = name.trim() ? name.trim() : "Untitled";
				setDirty(true);
			});
			descriptionArea.addEventListener("input", () => setDirty(true));
			codeArea.addEventListener("input", () => setDirty(true));
			copyButton.addEventListener("click", async () => {
				const name = nameInput.value.trim();
				if (!/^[a-zA-Z_$][a-zA-Z0-9_\-$]*$/.test(name)) {
					return;
				}
				const text = `{power:${name}}`;
				try {
					await navigator.clipboard.writeText(text);
					copyButton.textContent = "Copied";
					setTimeout(() => (copyButton.textContent = "Copy"), 1200);
				} catch (e) {
					copyButton.textContent = "Error";
					setTimeout(() => (copyButton.textContent = "Copy"), 1200);
				}
			});
			delButton.addEventListener("click", () => {
				const confirmDelete = confirm(
					"Delete this Power Shortcode? The change takes effect after you save.",
				);
				if (confirmDelete) {
					const li = delButton.closest(".power-shortcodes-repeater-item");
					if (li) {
						li.classList.add("removing");
						setTimeout(() => { li.remove(); }, 200);
					} else {
						// Fallback for unexpected DOM changes
						delButton.parentElement.remove();
					}
					updateStatusCounters();
					setDirty(true);
				}
			});
			if (insertAt === "top") {
				repeaterWrapper.insertBefore(repeaterItem, repeaterWrapper.firstChild);
			} else {
				repeaterWrapper.appendChild(repeaterItem);
			}
			updateStatusCounters();
			return repeaterItem;
		};

		if (oldValue.length) {
			for (const item of oldValue) {
				addRepeaterItem(item);
			}
		}

		const flashHighlight = (li) => {
 			if (!li) return;
 			li.classList.add("power-new-highlight");
 			setTimeout(() => li.classList.remove("power-new-highlight"), 1200);
 		};
		const flashError = (li) => {
			if (!li) return;
			li.classList.add("power-error-highlight");
			setTimeout(() => li.classList.remove("power-error-highlight"), 1200);
		};

		// Add new item from left button (may be absent)
		if (addButton) {
			addButton.addEventListener("click", () => {
				const li = addRepeaterItem();
				setDirty(true);
				try {
					const details = li.getElementsByClassName("power-shortcodes-details")[0];
					if (details) details.open = true;
					const input = li.getElementsByClassName("power-shortcodes-name")[0];
					if (input) {
						input.focus({ preventScroll: true });
						if (input.select) input.select();
					}
					li.scrollIntoView({ behavior: "smooth", block: "center" });
					flashHighlight(li);
				} catch (e) {}
			});
		}
		// Only sidebar New Power (append at bottom)
		(() => {
			const btn = document.getElementById("power-actions-new");
			if (!btn) return;
			btn.addEventListener("click", () => {
				const li = addRepeaterItem(undefined, "bottom");
				setDirty(true);
				try {
					const details = li.getElementsByClassName("power-shortcodes-details")[0];
					if (details) details.open = true;
					const input = li.getElementsByClassName("power-shortcodes-name")[0];
					if (input) {
						input.focus({ preventScroll: true });
						if (input.select) input.select();
					}
					li.scrollIntoView({ behavior: "smooth", block: "center" });
					flashHighlight(li);
				} catch (e) {}
			});
		})();

		// Bottom New button
		(() => {
			const btn = document.getElementById("power-actions-new-bottom");
			if (!btn) return;
			btn.addEventListener("click", () => {
				const li = addRepeaterItem(undefined, "bottom");
				setDirty(true);
				try {
					const details = li.getElementsByClassName("power-shortcodes-details")[0];
					if (details) details.open = true;
					const input = li.getElementsByClassName("power-shortcodes-name")[0];
					if (input) {
						input.focus({ preventScroll: true });
						if (input.select) input.select();
					}
					li.scrollIntoView({ behavior: "smooth", block: "center" });
					flashHighlight(li);
				} catch (e) {}
			});
		})();

		// Expand / Collapse all
		if (expandAllLink && collapseAllLink) {
			expandAllLink.addEventListener("click", (e) => {
				e.preventDefault();
				const all = repeaterWrapper.querySelectorAll(".power-shortcodes-details");
				all.forEach((d) => (d.open = true));
			});
			collapseAllLink.addEventListener("click", (e) => {
				e.preventDefault();
				const all = repeaterWrapper.querySelectorAll(".power-shortcodes-details");
				all.forEach((d) => (d.open = false));
			});
		}

		// Initial counters and filtered state
		runFilter();
	};

	// initialize Select2 with ajax for searching posts.
	const getSelect2Options = (postType) => {
		return {
			ajax: {
				url: ajaxurl,
				dataType: "json",
				delay: 250,
				data: function (params) {
					let args = {
						q: params.term,
						action: "dsh_get_posts",
					};
					if (postType) {
						args.dsh_post_type = postType;
					}
					return args;
				},
				processResults: function (data) {
					var options = [];
					if (data) {
						jQuery.each(data, function (index, text) {
							options.push({ id: text[0], text: text[1] });
						});
					}
					return {
						results: options,
					};
				},
				cache: true,
			},
			minimumInputLength: 3, // the minimum of symbols to input before perform a search
		};
	};

	const initDemosPage = () => {
		let $demosPage = $(".wrap.dynamic-shortcodes-demo");
		if (!$demosPage.length) {
			return;
		}
		let $postSelectorWrapper = $(".demos-preview-select-wrapper");
		let $postSelector = $("#dsh-demos-post-selector");
		$postSelector.select2(getSelect2Options(false));
		$postSelector.on("change", () => {
			let val = $postSelector.val();
			const url = new URL(window.location);
			url.searchParams.set("demos_post_id", val);
			window.location.href = url.toString();
		});

		let $barButtons = $demosPage.find("ul#demo-tabs-bar li a");
		// restore tab if present in url fragment:
		let fragment = window.location.hash;
		let selectedTab = $barButtons.first().attr("href");
		if (fragment) {
			selectedTab = fragment;
		}
		const showTab = (tab) => {
			$barButtons.removeClass("active");
			let $a = $barButtons.filter(`[href="${tab}"]`);
			if (
				$a.data("showPostSelector") === "yes" &&
				$a.parent().hasClass("enabled")
			) {
				$postSelectorWrapper.show();
			} else {
				$postSelectorWrapper.hide();
			}
			$a.addClass("active");
			$demosPage.find(".dsh-tab").hide();
			$(tab).fadeIn();
		};
		showTab(selectedTab);
		$barButtons.click(function (e) {
			e.preventDefault();
			let selectTab = $(this).attr("href");
			// setting the url fragment causes scrolling, the following is to
			// avoid it:
			let scrollPosition = $(window).scrollTop();
			window.location.hash = selectTab;
			$(window).scrollTop(scrollPosition);
			showTab(selectTab);
		});
	};
	initPower();
	initDemosPage();

	// Dismissable Admin Notices (from https://www.alexgeorgiou.gr/persistently-dismissible-notices-wordpress/):
	jQuery(".dynamic-shortcodes-dismissible-notice").on(
		"click",
		".notice-dismiss",
		function (event, el) {
			var $notice = jQuery(this).parent(".notice.is-dismissible");
			var dismiss_url = $notice.attr("data-dismiss-url");
			if (dismiss_url) {
				jQuery.get(dismiss_url);
			}
		},
	);
})(jQuery);
