<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms
namespace DynamicShortcodes\Core\Settings\AdminPages;

use DynamicShortcodes\Plugin;
use DSHDeps\DynamicOOO\PluginUtils\AdminPages\Pages\Base;
use DynamicShortcodes\Core\Shortcodes\ShortcodeParser;
use DynamicShortcodes\Core\Shortcodes\ParseError;
use DynamicShortcodes\Core\Settings\Manager as SettingsManager;

class PowerShortcodes extends Base {

	/**
	 * @return void
	 */
	public function init() {
		add_action( 'wp_ajax_dynamic_shortcodes_power_shortcodes_save', [ $this, 'power_shortcodes_save_ajax' ] );
	}

	public function render_content() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$is_license_active = Plugin::instance()->license_facade->get_active_license();

		$id          = SettingsManager::OPTIONS_PREFIX . '_power_shortcodes_json';
		$value       = esc_attr( get_option( $id, '' ) );
		$ajax_url    = admin_url( 'admin-ajax.php' );
		$ajax_action = 'dynamic_shortcodes_power_shortcodes_save';
		$nonce       = wp_create_nonce( 'power_shortcodes_save' );
		$title       = esc_html( get_admin_page_title() );
		// translators: specifiers are html tags.
		$info = esc_html__( 'Power Shortcodes are a type of Dynamic Shortcodes that come without the usual security restrictions. Please note that Power Shortcodes can only be created by administrators, but can be utilized by contributors as well.', 'dynamic-shortcodes' );

		$help_links                     = [
			[
				'title' => __( 'How to Build a Power Shortcode', 'dynamic-shortcodes' ),
				'url' => 'https://dnmc.ooo/dsh-doc-power',
			],
			[
				'title' => __( 'The Syntax of Dynamic Shortcodes', 'dynamic-shortcodes' ),
				'url' => 'https://dnmc.ooo/dsh-doc-syntax',
			],
		];
		$data_license_active            = $is_license_active ? 'yes' : 'no';
		$maybe_disabled_because_license = $is_license_active ? '' : 'disabled';
		$search_placeholder             = esc_attr__( 'Search by name, description or content...', 'dynamic-shortcodes' );
		$no_results_text                = esc_html__( 'No results match your search.', 'dynamic-shortcodes' );

		$link_html = '';
		foreach ( $help_links as $link ) {
			$link_html .= '<li><a href="' . $link['url'] . '">' . $link['title'] . '</a></li>';
		}

		$license_notice = '';
		if ( ! $is_license_active ) {
			$license_url = esc_attr( admin_url( '/admin.php?page=' . Plugin::instance()->plugin_utils->get_license_admin_page() ) );
			// translators: 1 and 2 are html tags.
			$msg            = esc_html__( 'You are not able to edit the Power Shortcodes because the license is not active. Please %1$sactivate the license%2$s.', 'dynamic-shortcodes' );
			$msg            = sprintf( $msg, "<a href=$license_url>", '</a>' );
			$license_notice = '<div class="notice notice-error">' . $msg . '</div>';
		}
		?>
		<div class="wrap dsh-power">

			<?php
			//phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $license_notice;
			?>
			<template id="power-shortcodes-template">
				<li class="power-shortcodes-repeater-item">
					<details class="power-shortcodes-details" open>
						<summary class="power-shortcodes-summary">
							<svg class="power-caret power-caret-right" viewBox="0 0 16 16" aria-hidden="true">
								<path d="M6 4l4 4-4 4" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
							</svg>
							<svg class="power-caret power-caret-down" viewBox="0 0 16 16" aria-hidden="true">
								<path d="M4 6l4 4 4-4" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
							</svg>
							<span class="power-shortcodes-summary-name"></span>
							<div class="power-summary-actions">
								<button class="power-shortcodes-del" type="button" title="<?php esc_attr_e( 'Delete', 'dynamic-shortcodes' ); ?>"><?php esc_html_e( 'Delete', 'dynamic-shortcodes' ); ?></button>
							</div>
						</summary>
						<ul class="power-shortcodes-item-error hidden"></ul>
						<div class="power-shortcodes-name-field">
							<label>
								<span class="power-field-label"><?php esc_html_e( 'Name', 'dynamic-shortcodes' ); ?></span>
								<input type="text" class="power-shortcodes-name" placeholder="<?php esc_attr_e( 'Enter a custom name for this Power Shortcode, so you can easily recall it later', 'dynamic-shortcodes' ); ?>" autocomplete="off" data-1p-ignore="true">
							</label>
						</div>
						<div class="power-shortcodes-description-field">
							<label>
								<span class="power-field-label"><?php esc_html_e( 'Description', 'dynamic-shortcodes' ); ?></span>
								<textarea cols="80" rows="1" spellcheck="false" class="power-shortcodes-description" placeholder="<?php esc_attr_e( 'Add a short description (optional)', 'dynamic-shortcodes' ); ?>" autocomplete="off" data-1p-ignore="true"></textarea>
							</label>
						</div>
						<div class="power-shortcodes-code-field">
							<label>
								<span class="power-field-label"><?php esc_html_e( 'Content', 'dynamic-shortcodes' ); ?></span>
								<div class="power-shortcodes-code-wrapper">
									<textarea cols="80" rows="4" spellcheck="false" class="power-shortcodes-code" placeholder="<?php esc_attr_e( 'Insert text with Dynamic Shortcodes', 'dynamic-shortcodes' ); ?>" autocomplete="off" data-1p-ignore="true"></textarea>
									<button class="power-shortcodes-format" type="button"><?php esc_html_e( 'Indent', 'dynamic-shortcodes' ); ?></button>
								</div>
							</label>
						</div>
						<div class="power-shortcodes-usage"></div>
						<button class="power-shortcodes-copy" type="button" title="<?php esc_attr_e( 'Copy shortcode to clipboard', 'dynamic-shortcodes' ); ?>"><?php esc_html_e( 'Copy', 'dynamic-shortcodes' ); ?></button>
					</details>
				</li>
			</template>
			<div class="content-wrapper">
				<div class="dsh-left">
					<form id="power-shortcodes-form" class="dynamic-shortcodes-tab" action="<?php echo esc_url( $ajax_url ); ?>" data-license-active="<?php echo esc_attr( $data_license_active ); ?>" autocomplete="off">
						<input type="hidden" name="nonce" value="<?php echo esc_attr( $nonce ); ?>">
						<input type="hidden" name="action" value="<?php echo esc_attr( $ajax_action ); ?>">
						<input type="hidden" name="power_shortcodes" value="<?php echo $value; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>">
						<div class="power-shortcodes-info"><?php echo esc_html( $info ); ?></div>
						<div id="power-shortcodes-form-message" class="notice hidden sr-only" role="status" aria-live="polite" tabindex="-1"></div>
						<div id="power-shortcodes-wrapper">
							<div class="power-shortcodes-bulk-toggles">
								<div class="power-shortcodes-bulk-left">
									<button type="button" id="power-shortcodes-expand-all"><?php esc_html_e( 'Expand all', 'dynamic-shortcodes' ); ?></button> · <button type="button" id="power-shortcodes-collapse-all"><?php esc_html_e( 'Collapse all', 'dynamic-shortcodes' ); ?></button>
								</div>
							</div>
							<ul id="power-shortcodes-repeater"></ul>
							<div id="power-shortcodes-no-results" class="power-shortcodes-no-results hidden"><?php echo esc_html( $no_results_text ); ?></div>
							<div class="power-shortcodes-bottom-actions">
								<button class="button" id="power-actions-new-bottom" type="button" aria-label="<?php esc_attr_e( 'Add new Power Shortcode at the bottom', 'dynamic-shortcodes' ); ?>" <?php echo $maybe_disabled_because_license; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>><?php esc_html_e( '+ New Power Shortcode', 'dynamic-shortcodes' ); ?></button>
							</div>
						</div>
					</form>
				</div>
				<div id="dsh-help" role="region" aria-label="<?php esc_attr_e( 'Power Shortcodes actions', 'dynamic-shortcodes' ); ?>">
					<div class="dsh-sticky-actions">
						<h3><?php esc_html_e( 'Search', 'dynamic-shortcodes' ); ?></h3>
						<div class="power-shortcodes-search-container">
							<input type="text" id="power-shortcodes-search" placeholder="<?php echo $search_placeholder; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>" aria-label="<?php esc_attr_e( 'Power Shortcodes search', 'dynamic-shortcodes' ); ?>" autocomplete="off" data-1p-ignore="true">
						</div>
						<hr>
						<h3><?php esc_html_e( 'Documentation', 'dynamic-shortcodes' ); ?></h3>
						<ul class="dsh-help-links">
							<?php
							//phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
							echo $link_html;
							?>
						</ul>
						<hr>
						<h3><?php esc_html_e( 'Actions', 'dynamic-shortcodes' ); ?></h3>
						<div class="dsh-actions-group">
							<button class="button button-primary power-shortcodes-save" type="button" <?php echo $maybe_disabled_because_license; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>><?php esc_html_e( 'Save Changes', 'dynamic-shortcodes' ); ?></button>
							<button class="button" id="power-actions-new" type="button" aria-label="<?php esc_attr_e( 'Add new Power Shortcode at the bottom', 'dynamic-shortcodes' ); ?>" <?php echo $maybe_disabled_because_license; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>><?php esc_html_e( '+ New Power Shortcode', 'dynamic-shortcodes' ); ?></button>
						</div>
					</div>
				</div>
				<div id="power-toast-backdrop" class="power-toast-backdrop hidden" aria-hidden="true"></div>
				<div id="power-toast" class="power-toast hidden" role="status" aria-live="polite"></div>
			</div>
		</div>
		<?php
	}

	/**
	 * If there are no errors two options are saved on the DB, one for the
	 * parsed shortcodes one for the json as sent by the frontend. This is useful
	 * for recreating the interface.
	 */
	public function power_shortcodes_save_ajax() {
		if ( ! current_user_can( 'manage_options' ) ||
			 ! wp_verify_nonce( $_POST['nonce'], 'power_shortcodes_save' ) ) {
			wp_send_json_error(
				[
					'message' => esc_html__( 'There was a problem while saving the Powers, you might want to copy your changes and reload the page before saving again.', 'dynamic-shortcodes' ),
				]
			);
		}
		$shortcodesjson = stripslashes( $_POST['power_shortcodes'] ?? '' );
		$shortcodes     = json_decode( $shortcodesjson, true );
		if ( ! is_array( $shortcodes ) ) {
			wp_send_json_error( [ 'message' => 'Malformed data' ] );
		}
		$res = $this->save_power_shortcodes( $shortcodes );
		if ( $res !== true ) {
			wp_send_json_error(
				[
					'message' => 'There was an error saving the shortcodes',
					'items_errors' => $res,
				]
			);
		}
		update_option( SettingsManager::OPTIONS_PREFIX . '_power_shortcodes_json', $shortcodesjson );
		wp_send_json_success(
			[
				'message' => esc_html__( 'Power Shortcodes were saved successfully', 'dynamic-shortcodes' ),
			]
		);
	}

	public function save_power_shortcodes( $data ) {
		[ $ok, $res ] = $this->parse_shortcodes( $data );
		if ( ! $ok ) {
			return $res;
		}
		update_option( SettingsManager::OPTION_POWER_SHORTCODES, $res );
		return true;
	}

	protected function parse_shortcodes( $shortcodes ) {
		$errors            = [];
		$parsed_shortcodes = [];
		foreach ( $shortcodes as $index => $shortcode ) {
			$ok = preg_match( '/^\s*(' . ShortcodeParser::REGEX_IDENTIFIER . ')\s*$/', $shortcode['name'], $matches );
			if ( ! $ok ) {
				$errors[] = [
					'index' => $index,
					'message' => 'Invalid syntax for the name',
				];
			}
			$name = $matches[1] ?? false;
			if ( $name && isset( $parsed_shortcodes[ $name ] ) ) {
				$errors[] = [
					'index' => $index,
					'message' => esc_html__( 'This name has already been used.', 'dynamic-shortcodes' ),
				];
			}
			$shortcodes_manager = \DynamicShortcodes\Plugin::instance()->shortcodes_manager;
			try {
				$ast = $shortcodes_manager->parse_shortcodes( $shortcode['code'], 0, false );
				if ( $name ) {
					$parsed_shortcodes[ $name ] = $ast;
				}
			} catch ( ParseError $e ) {
				$errors[] = [
					'index' => $index,
					'message' => esc_html__( 'There is a syntax error in the shortcode code', 'dynamic-shortcodes' ),
				];
			}
		}
		if ( ! empty( $errors ) ) {
			return [ false, $errors ];
		}
		return [ true, $parsed_shortcodes ];
	}
}
