<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms

namespace DynamicShortcodes\Core\Shortcodes;

class KeyargsConstraintChecker {

	public static function make_keyarg_err_message( $keyarg, $type ) {
		return sprintf( 'The keyarg %s is not allowed in shortcode %s ', $keyarg, $type );
	}

	public static function make_keyarg_conflict_err_message( $key1, $key2, $type ) {
		return sprintf( 'The keyargs %s and %s in shortcode %s are not compatible', $key1, $key2, $type );
	}

	public static function make_keyarg_requirment_err_message( $key, $required_key, $type ) {
		return sprintf( 'In shortcode %s, the keyarg %s is also required when the keyarg %s is selected. ', $type, $required_key, $key );
	}

	public static function is_global_keyarg( $key ) {
		$globals = [
			'raw!' => true,
			'expand!' => true,
		];
		return isset( $globals[ $key ] );
	}

	public static function is_keyarg_special( $s ) {
		return substr( $s, -1 ) === '!';
	}

	public static function normalize_keyargs( $keyargs, $allowed_keyargs, $type, $specials_only = false ) {
		$lc_keyargs = array_change_key_case( $keyargs );
		if ( $specials_only ) {
			$lc_keyargs = array_filter(
				$lc_keyargs,
				function ( $i ) {
					return self::is_keyarg_special( $i );
				},
				ARRAY_FILTER_USE_KEY
			);
		}
		$normalized_keyargs = [];
		foreach ( $allowed_keyargs as $ak => $aliases ) {
			if ( isset( $lc_keyargs[ $ak ] ) ) {
				$normalized_keyargs[ $ak ] = $lc_keyargs[ $ak ];
				unset( $lc_keyargs[ $ak ] );
			}
			foreach ( $aliases as $alias ) {
				if ( isset( $lc_keyargs[ $alias ] ) ) {
					$normalized_keyargs[ $ak ] = $lc_keyargs[ $alias ];
					unset( $lc_keyargs[ $alias ] );
				}
			}
		}
		// the remaining keys are either global keyargs or not allowed.
		foreach ( $lc_keyargs as $key => $val ) {
			if ( self::is_global_keyarg( $key ) ) {
				$normalized_keyargs[ $key ] = $val;
			} else {
				$msg = self::make_keyarg_err_message( array_keys( $lc_keyargs )[0], $type );
				//phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
				throw new EvaluationError( $msg );
			}
		}
		return $normalized_keyargs;
	}

	public static function check_conflicts( $normalized_keyargs, $conflicts, $type ) {
		foreach ( $conflicts as $conflict ) {
			$hits = [];
			foreach ( $conflict as $key ) {
				if ( isset( $normalized_keyargs[ $key ] ) ) {
					$hits[] = $key;
				}
				if ( count( $hits ) >= 2 ) {
					$msg = self::make_keyarg_conflict_err_message( $hits[0], $hits[1], $type );
					//phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
					throw new EvaluationError( $msg );
				}
			}
		}
	}

	public static function check_requirements( $normalized_keyargs, $requirements, $type ) {
		foreach ( $requirements as $key => $required_keys ) {
			if ( isset( $normalized_keyargs[ $key ] ) ) {
				foreach ( $required_keys as $rkey ) {
					if ( ! isset( $normalized_keyargs[ $rkey ] ) ) {
						$msg = self::make_keyarg_requirment_err_message( $key, $rkey, $type );
						//phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
						throw new EvaluationError( $msg );
					}
				}
			}
		}
	}

	public static function validate_and_normalize_keyargs( $keyargs, $allowed_keyargs, $type, $conflicts = [], $requirements = [], $specials_only = false ) {
		$normalized_keyargs = self::normalize_keyargs( $keyargs, $allowed_keyargs, $type, $specials_only );
		self::check_conflicts( $normalized_keyargs, $conflicts, $type );
		self::check_requirements( $normalized_keyargs, $requirements, $type );
		return $normalized_keyargs;
	}
}
