<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms

namespace DynamicShortcodes\Core\Shortcodes\Types;

use DynamicShortcodes\Core\Shortcodes\BaseShortcode;

class Acpt extends BaseShortcode {
	/**
	 * Local loop state variable name used in acpt-loop scopes.
	 * It is intentionally unlikely to conflict with user variables.
	 */
	private const LOOP_STATE_VAR = '*acpt*';

	/**
	 * @param array<string,mixed> $context
	 * @return array<string>
	 */
	public static function get_shortcode_types( $context ) {
		return [
			'acpt',
			'acpt-loop',
			'acpt-row',
		];
	}

	/**
	 * @return mixed
	 */
	public function evaluate() {
		$this->ensure_plugin_dependencies( [ 'acpt' ] );

		switch ( $this->type ) {
			case 'acpt':
				return $this->evaluate_acpt();
			case 'acpt-loop':
				return $this->evaluate_loop();
			case 'acpt-row':
				return $this->evaluate_row();
		}
	}

	/**
	 * @return mixed
	 */
	private function evaluate_acpt() {
		// If a single argument is provided, treat it as loop-context usage.
		if ( $this->get_args_count() === 1 ) {
			$local_env  = $this->unit_interpreter->local_env;
			$loop_state = $local_env->has_var( self::LOOP_STATE_VAR ) ? $local_env->get_var( self::LOOP_STATE_VAR ) : null;

			if ( ! is_array( $loop_state ) || ( $loop_state['context'] ?? null ) === null ) {
				$this->evaluation_error( esc_html__( 'acpt with a single argument can only be used inside an acpt-loop', 'dynamic-shortcodes' ) );
			}
			$this->init_keyargs( [] );
			$field_name = $this->get_arg( 0, 'string' );
			$context    = $loop_state['context'];
			$box_name   = $loop_state['box_name'] ?? null;
			$parent     = $loop_state['parent_field_name'] ?? null;
			$index      = $loop_state['index'] ?? null;
			$block_name = $loop_state['block_name'] ?? null;
			$block_idx  = $loop_state['block_index'] ?? null;
			$block_data = $loop_state['block_data'] ?? null;

			// Check if we're in a Flexible Content block context
			if ( $block_name !== null && $block_data !== null ) {
				// Access field directly from current block data
				if ( isset( $block_data[ $field_name ] ) ) {
					return $block_data[ $field_name ];
				}
				return null;
			} elseif ( $block_name !== null ) {
				// Fallback: try get_acpt_block_child_field (might not work)
				$params = array_merge(
					$context,
					[
						'box_name' => $box_name,
						'parent_field_name' => $parent,
						'field_name' => $field_name,
						'block_name' => $block_name,
						'block_index' => $block_idx,
					]
				);
				return get_acpt_block_child_field( $params );
			} else {
				// Use get_acpt_child_field for regular Repeater
				$params = array_merge(
					$context,
					[
						'box_name' => $box_name,
						'parent_field_name' => $parent,
						'field_name' => $field_name,
						'index' => $index,
					]
				);

				return get_acpt_child_field( $params );
			}
		}

		// Normal behavior: 2 arguments required
		$this->arity_check( 2, 2 );

		$this->init_keyargs(
			[
				'id' => [],
				'box_name' => [],
				'format' => [ 'fmt' ], // enable acpt formatting of field.
				// locations:
				'post' => [],
				'user' => [],
				'term' => [],
				'option' => [],
			],
			[
				[ 'post', 'user', 'term', 'option' ],
				[ 'option', 'id' ],
				[ 'format' ],
			],
		);

		$box_name   = $this->get_arg( 0, 'string' );
		$field_name = $this->get_arg( 1, 'string' );
		$context    = $this->select_id();
		$format     = $this->get_bool_keyarg( 'format', true );

		// Get the field object to determine its type
		$field_object = get_acpt_meta_field_object( $box_name, $field_name );

		if ( ! $field_object ) {
			return null;
		}

		$field_type = $field_object->type ?? null;

		$params = array_merge(
			$context,
			[
				'box_name' => $box_name,
				'field_name' => $field_name,
				'return' => 'raw', // returns post IDs and attachments instead of objects
			]
		);

		$value = get_acpt_field( $params );

		if ( ! $format ) {
			return $value;
		}

		switch ( $field_type ) {
			case 'Address':
				return $value['address'] ?? '';

			case 'AddressMulti':
				if ( is_array( $value ) ) {
					return array_column( $value, 'address' );
				}
				return '';

			case 'Currency':
				return $value['amount'] ?? '';

			case 'Time':
			case 'Date':
			case 'DateTime':
				return $value['value'] ?? '';

			case 'Length':
				return ( $value['length'] ?? '' ) . ' ' . ( $value['unit'] ?? '' );

			case 'File':
				return $value['file'] ?? '';

			case 'Weight':
				return ( $value['weight'] ?? '' ) . ' ' . ( $value['unit'] ?? '' );

			case 'Barcode':
			case 'QRCode':
				return $value['value'] ?? '';

			case 'Url':
				return $value['url'] ?? '';

			case 'Post':
				if ( is_array( $value ) ) {
					return array_column( $value, 'id' );
				}
				return [];

			case 'Phone':
				return ( $value['dial'] ?? '' ) . ( $value['value'] ?? '' );
		}

		return $value;
	}

	/**
	 * @return string
	 */
	private function evaluate_loop() {
		$this->should_not_sanitize();
		$this->arity_check( 3, 3 );
		$this->init_keyargs(
			[
				'id' => [],
				// locations:
				'post' => [],
				'user' => [],
				'term' => [],
				'option' => [],
				'separator' => [ 'sep' ],
			],
			[
				[ 'post', 'user', 'term', 'option' ],
				[ 'option', 'id' ],
			]
		);

		$box_name   = $this->get_arg( 0, 'string' );
		$field_name = $this->get_arg( 1, 'string' );
		$context    = $this->select_id();
		$sep        = $this->get_keyarg_default( 'separator', '', 'string' );

		$is_flexible = false;
		$loop_data   = null;

		$local_env = $this->unit_interpreter->local_env;

		// Check if we're inside a parent loop (nested repeater/flexible) via local loop state
		$parent_loop_state = $local_env->has_var( self::LOOP_STATE_VAR ) ? $local_env->get_var( self::LOOP_STATE_VAR ) : null;
		if ( is_array( $parent_loop_state ) && ( $parent_loop_state['context'] ?? null ) !== null ) {
			$parent_context = $parent_loop_state['context'];
			$parent_field   = $parent_loop_state['parent_field_name'] ?? null;
			$parent_index   = $parent_loop_state['index'] ?? null;
			$parent_block   = $parent_loop_state['block_name'] ?? null;
			$parent_bindex  = $parent_loop_state['block_index'] ?? null;

			// We're nested: check if parent is Flexible or Repeater
			if ( $parent_block !== null ) {
				// Parent is Flexible Content: use get_acpt_block_child_field
				$params = array_merge(
					$parent_context,
					[
						'box_name' => $box_name,
						'parent_field_name' => $parent_field,
						'field_name' => $field_name,
						'block_name' => $parent_block,
						'block_index' => $parent_bindex,
					]
				);

				$loop_data = get_acpt_block_child_field( $params );
			} else {
				// Parent is Repeater: use get_acpt_child_field
				$params = array_merge(
					$parent_context,
					[
						'box_name' => $box_name,
						'parent_field_name' => $parent_field,
						'field_name' => $field_name,
						'index' => $parent_index,
					]
				);

				$loop_data = get_acpt_child_field( $params );
			}

			// Check if the nested field is a Flexible Content
			// Flexible Content has a ['blocks'] key in the structure
			if ( is_array( $loop_data ) && isset( $loop_data['blocks'] ) ) {
				$is_flexible = true;
				$loop_data   = $loop_data['blocks'];
			}
		} else {
			// Top-level: try to get the field data
			$params = array_merge(
				$context,
				[
					'box_name' => $box_name,
					'field_name' => $field_name,
					'return' => 'raw',
				]
			);

			$loop_data = get_acpt_field( $params );

			// Check if the returned data has a ['blocks'] structure (Flexible Content)
			if ( is_array( $loop_data ) && isset( $loop_data['blocks'] ) ) {
				$is_flexible = true;
				$loop_data   = $loop_data['blocks'];
			}
		}

		if ( ! is_array( $loop_data ) || empty( $loop_data ) ) {
			return '';
		}

		$buf   = [];
		$index = 0;

		foreach ( $loop_data as $item ) {
			$local_env->open_scope();
			try {
				$loop_state = [
					'box_name'          => $box_name,
					'parent_field_name' => $field_name,
					'context'           => $context,
				];

				if ( $is_flexible && is_array( $item ) ) {
					// Flexible Content: extract block name and data.
					// Structure: [ 'block_name' => [...fields...] ]
					$block_name                = key( $item );
					$block_data                = $item[ $block_name ] ?? null;
					$loop_state['block_name']  = $block_name;
					$loop_state['block_index'] = $index;
					$loop_state['block_data']  = $block_data;
				} else {
					// Regular Repeater
					$loop_state['index'] = $index;
				}

				$local_env->define_var( self::LOOP_STATE_VAR, $loop_state );

				$buf[] = $this->get_arg_as_string( 2, true );
			} finally {
				$local_env->close_scope();
			}

			++$index;
		}

		return implode( $sep, $buf );
	}

	/**
	 * Evaluate acpt-row shortcode
	 *
	 * Returns information about the current row/block in the loop.
	 * Must be called inside an acpt-loop.
	 *
	 * @return mixed Block name, index, or null
	 */
	private function evaluate_row() {
		$this->arity_check( 1, 1 );
		$this->init_keyargs( [] );

		$local_env  = $this->unit_interpreter->local_env;
		$loop_state = $local_env->has_var( self::LOOP_STATE_VAR ) ? $local_env->get_var( self::LOOP_STATE_VAR ) : null;

		// Check if we're inside any loop
		if ( ! is_array( $loop_state ) || ! isset( $loop_state['context'] ) ) {
			$this->evaluation_error( esc_html__( 'acpt-row can only be used inside an acpt-loop', 'dynamic-shortcodes' ) );
		}

		$arg = $this->get_arg( 0, 'string' );

		switch ( $arg ) {
			case 'block':
				// Return block name for Flexible Content, null for Repeaters
				return $loop_state['block_name'] ?? null;

			case 'index':
				// Return current index (works for both Repeater and Flexible)
				if ( array_key_exists( 'block_index', $loop_state ) ) {
					// We're in a Flexible Content block
					return $loop_state['block_index'];
				}
				if ( array_key_exists( 'index', $loop_state ) ) {
					// We're in a Repeater row
					return $loop_state['index'];
				}
				return null;

			default:
				$this->evaluation_error( esc_html__( 'unrecognized argument', 'dynamic-shortcodes' ) );
		}
	}

	/**
	 * @return array<string,mixed>
	 */
	private function select_id() {
		if ( $this->has_keyarg( 'option' ) ) {
			$this->ensure_all_privileges( esc_html__( 'Reading ACPT option page fields', 'dynamic-shortcodes' ) );
			return [ 'option_page' => $this->get_keyarg( 'option', 'string' ) ];
		} elseif ( $this->has_keyarg( 'user' ) ) {
			if ( $this->has_keyarg( 'id' ) ) {
				$this->ensure_all_privileges( esc_html__( 'Reading User ACPT fields of a specific user', 'dynamic-shortcodes' ) );
				return [ 'user_id' => $this->get_keyarg( 'id', 'numeric' ) ];
			} else {
				return [ 'user_id' => get_current_user_id() ];
			}
		} elseif ( $this->has_keyarg( 'term' ) ) {
			if ( $this->has_keyarg( 'id' ) ) {
				$term_id = $this->get_keyarg( 'id', 'numeric' );
			} else {
				$term_id = get_queried_object_id();
			}
			if ( ! term_exists( $term_id ) ) {
				$this->evaluation_error( esc_html__( 'Cannot find a term to use', 'dynamic-shortcodes' ) );
			}
			return [ 'term_id' => $term_id ];
		} elseif ( $this->has_keyarg( 'id' ) ) {
			$id = $this->get_keyarg( 'id', 'numeric' );
			if ( ! is_post_publicly_viewable( $id ) ) {
				$this->ensure_all_privileges( esc_html__( 'Reading ACPT Field on not public post', 'dynamic-shortcodes' ) );
			}
			return [ 'post_id' => $id ];
		}
		return [ 'post_id' => get_the_ID() ];
	}
}
