<?php
/**
 * @package   Essential_Grid
 * @author    ThemePunch <info@themepunch.com>
 * @link      https://www.essential-grid.com/
 * @copyright 2025 ThemePunch
 */
 
if( !defined( 'ABSPATH') ) exit();

class Essential_Grid_Woocommerce {

	const ARG_REGULAR_PRICE_FROM = 'reg_price_from';
	const ARG_REGULAR_PRICE_TO = 'reg_price_to';
	const ARG_SALE_PRICE_FROM = 'sale_price_from';
	const ARG_SALE_PRICE_TO = 'sale_price_to';
	const ARG_IN_STOCK_ONLY = 'instock_only';
	const ARG_FEATURED_ONLY = 'featured_only';
	
	const META_REGULAR_PRICE = '_regular_price';
	const META_SALE_PRICE = '_sale_price';
	const META_STOCK_STATUS = '_stock_status'; //can be 'instock' or 'outofstock'
	const META_SKU = '_sku'; //can be 'instock' or 'outofstock'
	const META_FEATURED = '_featured'; //can be 'instock' or 'outofstock'
	const META_STOCK = '_stock'; //can be 'instock' or 'outofstock'
	
	const SORTBY_NUMSALES = 'meta_num_total_sales';
	const SORTBY_REGULAR_PRICE = 'meta_num__regular_price';
	const SORTBY_SALE_PRICE = 'meta_num__sale_price';
	const SORTBY_SKU = 'meta__sku';
	const SORTBY_STOCK = 'meta_num_stock';
	
	const LIGHTBOX_MODE = 'woocommerce-gallery';

	/**
	 * Instance of this class.
	 * @var null|object
	 */
	protected static $instance = null;

	/**
	 * Return an instance of this class.
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	private function __construct() {
		global $esg_wc_is_localized;
		$esg_wc_is_localized = false;
		add_action('woocommerce_init', [ $this, 'add_woo_hooks' ] );
	}

	public function add_woo_hooks() {
		// meta_dialog
		add_action('essgrid_meta_dialog_post', [ $this, 'add_handles_to_meta_picker' ] );
		// item skin editor
		add_action('essgrid_item_skin_show_hide_options_html', [ $this, 'item_skin_show_hide_options_html' ] );
		// grid settings
		add_action('essgrid_grid_settings_navigation_available_modules_html', [ $this, 'navigation_available_modules_html' ] );

		// Essential_Grid_Meta
		add_filter('essgrid_get_all_meta_handle', [ $this, 'get_meta_handles' ] );
		add_filter('essgrid_maybe_get_meta_value_by_handle', [ $this, 'get_meta_value_by_handle' ], 10, 3);
		// Essential_Grid_Navigation
		add_filter('essgrid_translate_demo_filter_additional_data', [ $this, 'demo_filter_additional_data' ] );
		add_filter('essgrid_output_cart', [ $this, 'nav_output_cart' ], 10, 2 );
		// Essential_Grid_Item_Element
		add_filter('essgrid_getElementsForDropdown', [ $this, 'get_dropdown_elements' ] );
		add_filter('essgrid_get_existing_elements_other', [ $this, 'get_existing_elements' ] );
		// Essential_Grid
		add_filter('essgrid_prepare_sorting_array_by_post_post', [ $this, 'prepare_sorting_array_by_post_post' ], 10, 3);
		add_filter('essgrid_get_lightbox_additions', [ $this, 'get_lightbox_additions' ], 10, 3);
		// Essential_Grid_Admin
		add_filter('essgrid_get_combined_order_values', [ $this, 'get_order_values' ] );
		// Essential_Grid_Item_Skin
		add_filter('essgrid_get_lightbox_modes', [ $this, 'get_lightbox_modes' ] );
		add_filter('essgrid_get_lightbox_contentgroup_modes', [ $this, 'get_lightbox_contentgroup_modes' ] );
		add_filter('essgrid_output_item_skin_maybe_skip_layer', [ $this, 'output_item_skin_maybe_skip_layer' ], 10, 4);
		add_filter('essgrid_item_skin_process_layer_settings_source', [ $this, 'process_layer_settings_source' ], 10, 4);
		// Woo Add to Cart Updater
		add_filter('woocommerce_add_to_cart_fragments', [ $this, 'add_to_cart_fragment' ] );

	}
	
	/**
	 * true / false if the wpml plugin exists
	 */
	public function is_woo_exists(){
		return class_exists('Woocommerce');
	}

	/**
	 * compare wc current version to given version
	 *
	 * @param string $version
	 * @return bool
	 */
	public function version_check( $version = '1.0' ) {
		global $woocommerce;
		return version_compare($woocommerce->version, $version, '>=' );
	}

	public function get_product( $post_id ) {
		return $this->version_check('3.0') ? wc_get_product( $post_id ) : get_product( $post_id );
	}

	/**
	 * get wc post types
	 */
	public function get_custom_post_types() {
		$arr = [];
		$arr['product'] = esc_attr__('Product', 'essential-grid');
		$arr['product_variation'] = esc_attr__('Product Variation', 'essential-grid');
		
		return apply_filters( 'essgrid_wc_get_custom_post_types', $arr );
	}
	
	/**
	 * get price query
	 */
	private function get_price_query( $priceFrom, $priceTo, $metaTag ) {
		if(empty($priceFrom))
			$priceFrom = 0;
			
		if(empty($priceTo))
			$priceTo = 9999999999;
		
		$query = [
			'key' => $metaTag,
			'value' => [$priceFrom, $priceTo],
			'type' => 'numeric',
			'compare' => 'BETWEEN'
		];
		
		return (apply_filters('essgrid_wc_get_price_query', $query));
	}
	
	/**
	 * get meta query for filtering woocommerce posts. 
	 */
	public function get_meta_query($args){
		$base = new Essential_Grid_Base();
		
		$regPriceFrom = $base->getVar($args, self::ARG_REGULAR_PRICE_FROM);
		$regPriceTo = $base->getVar($args, self::ARG_REGULAR_PRICE_TO);
		
		$salePriceFrom = $base->getVar($args, self::ARG_SALE_PRICE_FROM);
		$salePriceTo = $base->getVar($args, self::ARG_SALE_PRICE_TO);
		
		$inStockOnly = $base->getVar($args, self::ARG_IN_STOCK_ONLY);
		$featuredOnly = $base->getVar($args, self::ARG_FEATURED_ONLY);
		
		$arrQueries = [];
		
		//get regular price array
		if(!empty($regPriceFrom) || !empty($regPriceTo)){
			$arrQueries[] = $this->get_price_query($regPriceFrom, $regPriceTo, self::META_REGULAR_PRICE);
		}
		
		//get sale price array
		if(!empty($salePriceFrom) || !empty($salePriceTo)){
			$arrQueries[] = $this->get_price_query($salePriceFrom, $salePriceTo, self::META_SALE_PRICE);
		}
		
		if($inStockOnly == 'true'){
			$query = ['key' => self::META_STOCK_STATUS, 'value' => 'instock'];
			$arrQueries[] = $query;
		}
		
		if($featuredOnly == 'true'){
			$query = ['key' => self::META_FEATURED, 'value' => 'yes'];
			$arrQueries[] = $query;
		}
		
		$query = [];
		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
		if(!empty($arrQueries)) $query = [ 'meta_query' => $arrQueries ];
			
		return apply_filters( 'essgrid_wc_get_meta_query', $query );
	}
	
	/**
	 * get sortby function including standart wp sortby array
	 */
	public function get_arr_sort_by() {
		return [
			'opt_disabled_1' => esc_attr__('---- WooCommerce Filters ----', 'essential-grid'),
			self::SORTBY_REGULAR_PRICE => esc_attr__('Price', 'essential-grid'),
			self::SORTBY_NUMSALES => esc_attr__('Number Of Sales', 'essential-grid'),
			self::SORTBY_SKU => esc_attr__('SKU', 'essential-grid'),
			self::SORTBY_STOCK => esc_attr__('Stock Quantity', 'essential-grid'),
			'opt_disabled_2' => esc_attr__('---- Regular Filters ----', 'essential-grid'),
		];
	}
	
	/**
	 * check if product is on sale
	 */
	public function check_if_on_sale( $post_id ) {
		$product = $this->get_product( $post_id );
		return $product !== false && $product->is_on_sale();
	}
	
	/**
	 * check if product is on sale
	 */
	public function check_if_is_featured( $post_id ) {
		$product = $this->get_product( $post_id );
		return $product !== false && $product->is_featured();
	}
	
	/**
	 * get sortby function including standart wp sortby array
	 */
	public function get_value_by_meta($post_id, $meta, $separator = ',', $catmax = false){
		$meta_value = '';
		$is_30 = $this->version_check('3.0');
		$product = $this->get_product( $post_id );
		if ( $product !== false ) {
			switch($meta){
				case 'wc_price':
					if ( $product->is_in_stock() ) {
						$meta_value = wc_price($product->get_price());
					} else {
						$meta_value = esc_html__( 'Out of Stock', 'essential-grid' );
					}
				break;
				case 'wc_price_no_cur':
					if ( $product->is_in_stock() ) {
						$meta_value = $product->get_price();
					} else {
						$meta_value = esc_html__( 'Out of Stock', 'essential-grid' );
					}
				break;
				case 'wc_full_price':
					if ( $product->is_in_stock() ) {
						$meta_value = $product->get_price_html();
					} else {
						$meta_value = esc_html__( 'Out of Stock', 'essential-grid' );
					}
				break;
				case 'wc_stock':
					if($is_30){
						$meta_value = $product->get_stock_quantity();
					}else{
						$meta_value = $product->get_total_stock();
					}
				break;
				case 'wc_rating':
					if($is_30){
						$meta_value = wc_get_rating_html( $product->get_average_rating() );
					}else{
						$meta_value = $product->get_rating_html();
					}
				break;
				case 'wc_star_rating':
					if($is_30){
						$cur_rating = wc_get_rating_html( $product->get_average_rating() );
					}else{
						$cur_rating = $product->get_rating_html();
					}
					if($cur_rating !== '')
						$meta_value = '<div class="esg-starring">'.$cur_rating.'</div>';
				break;
				case 'wc_categories':
					if($is_30){
						$categories = wc_get_product_category_list($post_id, $separator);
						
						// new catmax option only available for WC v3.0+
						if($catmax !== false) {
							$categories = explode($separator, $categories);
							$categories = array_slice($categories, 0, $catmax, true);
							$categories = implode($separator, $categories);
						}
						
					}else{
						$categories = $product->get_categories($separator);
					}
					$meta_value = $categories;
				break;
				case 'wc_add_to_cart':
					$meta_value = $product->add_to_cart_url();
				break;
				case 'wc_add_to_cart_button': 
					// get whole button from WooCommerce
					$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
					$ajax_cart_en = get_option( 'woocommerce_enable_ajax_add_to_cart' ) == 'yes';
					$assets_path = str_replace( ['http:', 'https:'], '', WC()->plugin_url() ) . '/assets/';
					$frontend_script_path = $assets_path . 'js/frontend/';
					
					if ( $ajax_cart_en ){
						wp_enqueue_script( 'esg-wc-add-to-cart', $frontend_script_path . 'add-to-cart' . $suffix . '.js', ['jquery'], WC_VERSION, true );
						
						global $esg_wc_is_localized;
						if($esg_wc_is_localized === false){ //load it only one time
							wp_localize_script( 'esg-wc-add-to-cart', 'wc_add_to_cart_params', apply_filters( 'wc_add_to_cart_params', [
								'ajax_url'                => WC()->ajax_url(),
								'wc_ajax_url'             => WC_AJAX::get_endpoint( "%%endpoint%%" ),
								'ajax_loader_url'         => apply_filters( 'woocommerce_ajax_loader_url', $assets_path . 'images/ajax-loader@2x.gif' ),
								'i18n_view_cart'          => esc_attr__( 'View Cart', 'essential-grid' ),
								'cart_url'                => get_permalink( wc_get_page_id( 'cart' ) ),
								'is_cart'                 => is_cart(),
								'cart_redirect_after_add' => get_option( 'woocommerce_cart_redirect_after_add' )
							]) );
							$esg_wc_is_localized = true;
						}
					}

					$product_type = $is_30 ? $product->get_type() : $product->product_type;

					$meta_value = apply_filters(
						'woocommerce_loop_add_to_cart_link',
						sprintf( '<a href="%s" rel="nofollow" data-product_id="%s" data-product_sku="%s" class="button %s %s product_type_%s">%s</a>',
							esc_url( $product->add_to_cart_url() ),
							esc_attr( $post_id ),
							esc_attr( $product->get_sku() ),
							$product->is_purchasable() ? 'add_to_cart_button' : '',
							$product->is_purchasable() && $ajax_cart_en && esc_attr( $product_type ) !="variable" ? 'ajax_add_to_cart' : '',
							esc_attr( $product_type ),
							esc_html( $product->add_to_cart_text() )
						),
						$product
					);
				break;
				default:
					$meta_value = apply_filters('essgrid_woocommerce_meta_content', $meta_value, $meta, $post_id, $product);
				break;
			}
		}
		
		return apply_filters('essgrid_wc_get_value_by_meta', $meta_value, $meta, $post_id, $product);
	}
	
	/**
	 * get sortby function including standart wp sortby array
	 */
	public function get_meta_array() {
		$wc_array = [
			'wc_full_price' => esc_attr__('Full Price', 'essential-grid'),
			'wc_price' => esc_attr__('Single Price', 'essential-grid'),
			'wc_price_no_cur' => esc_attr__('Single Price without currency', 'essential-grid'),
			'wc_stock' => esc_attr__('In Stock', 'essential-grid'),
			'wc_rating' => esc_attr__('Text Rating', 'essential-grid'),
			'wc_star_rating' => esc_attr__('Star Rating', 'essential-grid'),
			'wc_categories' => esc_attr__('Categories', 'essential-grid'),
			'wc_add_to_cart' => esc_attr__('Add to Cart URL', 'essential-grid'),
			'wc_add_to_cart_button' => esc_attr__('Add to Cart Button', 'essential-grid'),
		];
		
		return apply_filters('essgrid_woocommerce_meta_handle', $wc_array);
	}
	
	/**
	 * get all attached images
	 * @since: 1.5.4
	 */
	public function get_image_attachements( $post_id, $url = false, $source = 'full' ) {
		$ret_img = '';
		$product = $this->get_product( $post_id );
		if( $product !== false ) {
			$wc_img = $this->version_check('3.0') ? $product->get_gallery_image_ids() : $product->get_gallery_attachment_ids();
			if($url){
				$images = [];
				foreach ( $wc_img as $img ) { 
					$t_img = wp_get_attachment_image_src($img, $source);
					if ( $t_img !== false ) {
						$images[] = $t_img[0];
					}
				}
				$ret_img = $images;
			} else { 
				//get URL instead of ID
				$ret_img = $wc_img;
			}
		}
		
		return apply_filters('essgrid_wc_get_image_attachements', $ret_img, $post_id, $url);
	}

	/**
	 * update Cart Overview after Ajax call
	 * @since: 2.1.0.2
	 */
	public function add_to_cart_fragment( $fragments ) {
		global $woocommerce;
		
		$fragments['span.ess-cart-content'] = 
			'<span class="ess-cart-content">' .
			sprintf(
				/* translators: 1: cart_contents_count, 2: get_cart_total */
				__( '%1$s items - %2$s', 'essential-grid' ),
				$woocommerce->cart->cart_contents_count,
				$woocommerce->cart->get_cart_total()
			) .
			'</span>';
		
		return $fragments;
	}

	/**
	 * add woocommerce sorting
	 * 
	 * @param array $sorts
	 * @param array $post
	 * @param array $order_by
	 * @return array
	 */
	public function prepare_sorting_array_by_post_post( $sorts, $post, $order_by ) {
		$product = $this->get_product( $post['ID'] );
		if ( empty( $product ) ) {
			return $sorts;
		}

		foreach ($order_by as $order) {
			switch ($order) {
				case 'meta_num_total_sales':
					$sorts['total-sales'] = get_post_meta($post['ID'], 'total_sales', true);
					break;
				case 'meta_num__regular_price':
					$sorts['regular-price'] = $product->get_price();
					break;
				case 'meta__featured':
					$sorts['featured'] = ($product->is_featured()) ? '1' : '0';
					break;
				case 'meta__sku':
					$sorts['sku'] = $product->get_sku();
					break;
				case 'meta_num_stock':
					$sorts['in-stock'] = $product->get_stock_quantity();
					break;
			}
		}
		
		return $sorts;
	}

	/**
	 * add meta handles to data source in skin editor as post meta
	 *
	 * @param array $meta
	 * @return array
	 */
	public function get_meta_handles($meta)
	{
		$wc_meta = $this->get_meta_array();
		foreach($wc_meta as $handle => $name){
			$meta[] = $handle;
		}
		
		return $meta;
	}

	/**
	 * @param string $text
	 * @param int    $post_id
	 * @param string $handle
	 * @return string
	 */
	public function get_meta_value_by_handle( $text, $post_id, $handle ) {
		$wc_text = $this->get_value_by_meta( $post_id, $handle );
		return empty( $wc_text ) ? $text : $wc_text;
	}

	/**
	 * @param array $data
	 * @return array
	 */
	public function demo_filter_additional_data( $data ) {
		$tmp_wc = $this->get_meta_array();
		foreach ($tmp_wc as $handle => $wc_name) {
			$data[ $handle ][ 'name' ] = $wc_name;
		}
		
		return $data;
	}

	/**
	 * @param string $output
	 * @param Essential_Grid_Navigation $nav
	 * @return string
	 */
	public function nav_output_cart( $output, $nav )	{
		if ( ! function_exists( 'wc_cart_totals_subtotal_html' ) || is_null( WC()->cart ) ) {
			return $output;
		}
		
		$spacing = $nav->get_spacing();
		ob_start();
		echo '<div class="esg-cartbutton-wrapper ' . esc_attr( $nav->get_special_class() ) . '"' .  ( $spacing ? esc_attr( $spacing ) : '' ) .  '>' .
				 '<div class="esg-cartbutton">' . 
					 '<a href="' . esc_url( wc_get_cart_url() ) . '">' . 
						 '<i class="eg-icon-basket"></i>' . 
						 '<span class="ess-cart-content">' .
							 esc_html( WC()->cart->get_cart_contents_count() ) .
							 esc_html__( ' items - ', 'essential-grid' );
							echo wc_cart_totals_subtotal_html();
					echo '</span>' .
					 '</a>' .
				 '</div>' . 
			 '</div>';

		$output .= ob_get_clean();

		return $output;
	}

	/**
	 * @param array $data
	 * @return array
	 */
	public function get_dropdown_elements( $data ) {
		$tmp_wc = $this->get_meta_array();
		$data[ 'woocommerce' ] = [];
		foreach ( $tmp_wc as $handle => $name ) {
			$data[ 'woocommerce' ][ $handle ][ 'name' ] = $name;
		}
		
		return $data;
	}
	
	/**
	 * @param array $data
	 * @return array
	 */
	public function get_existing_elements( $data ) {
		$data[ 'show-on-sale' ] = [
			'value' => 'string',
			'type' => 'checkbox',
			'values' => [ 'default' => false ],
			'style' => 'false',
			'unit' => ''
		];
		$data[ 'show-if-featured' ] = [
			'value' => 'string',
			'type' => 'checkbox',
			'values' => [ 'default' => false ],
			'style' => 'false',
			'unit' => ''
		];
		
		return $data;
	}

	/**
	 * add meta handles to meta key picker dialog
	 *
	 * @return void
	 */
	public function add_handles_to_meta_picker()
	{
		$wc_meta = $this->get_meta_array();
		foreach($wc_meta as $handle => $name){
			echo '<tr><td>%'.esc_html($handle).'%</td><td>'.esc_html($name).'</td></tr>';
		}
	}

	/**
	 * @param false|array $skin
	 * @return void
	 */
	public function item_skin_show_hide_options_html( $skin )
	{
		$tooltip_text_on_sale = esc_attr__('Show the Element only if it is on Sale. This is a WooCommerce setting', 'essential-grid');
		$text_on_sale = esc_attr__('Show if Product is on Sale', 'essential-grid');
		$tooltip_text_featured = esc_attr__('Show the Element only if it is featured. This is a WooCommerce setting', 'essential-grid');
		$text_featured = esc_attr__('Show if Product is featured', 'essential-grid');
		
		echo '<div class="div13"></div>'
			. '<label class="eg-tooltip-wrap esg-w-250" title="' . $tooltip_text_on_sale . '">' . $text_on_sale . '</label>'
			. '<input type="checkbox" name="element-show-on-sale" value="true" />'
			. '<div class="div13"></div>'
			. '<label class="eg-tooltip-wrap esg-w-250" title="' . $tooltip_text_featured . '">' . $text_featured . '</label>'
			. '<input type="checkbox" name="element-show-if-featured" value="true" />';
	}

	/**
	 * @param array $data
	 * @param string $lightbox_mode
	 * @param array $post
	 * @return mixed
	 */
	public function get_lightbox_additions( $data, $lightbox_mode, $post ) {
		if ( $lightbox_mode !== self::LIGHTBOX_MODE ) {
			return $data;
		}

		return [
			$this->get_image_attachements( $post['ID'], true ),
			$this->get_image_attachements( $post['ID'], true, 'thumbnail' ),
		];
	}

	/**
	 * @param array $layout
	 * @return void
	 */
	public function navigation_available_modules_html( $layout ) {
		$data_attr = '';
		if ( ! empty( $layout['cart'] ) ) {
			$data_attr = ' data-putin="'.esc_attr(current(array_keys($layout['cart']))).'" data-sort="'.esc_attr($layout['cart'][current(array_keys($layout['cart']))]).'"';
		}
		echo '<div data-navtype="cart" class="eg-navigation-cons-cart eg-navigation-cons" ' .  $data_attr . '>'
			. '<i class="eg-icon-basket"></i>'
			. esc_html__("Cart", 'essential-grid')
			. '</div>';
	}

	/**
	 * @param array $values
	 * @return array
	 */
	public function get_order_values( $values ) {
		return $this->get_arr_sort_by() + $values;
	}
	
	/**
	 * @param array $values
	 * @return array
	 */
	public function get_lightbox_modes( $values ) {
		$values[ self::LIGHTBOX_MODE ] = esc_html__( 'WooCommerce Gallery', 'essential-grid' );
		return $values;
	}
	
	/**
	 * @param array $values
	 * @return array
	 */
	public function get_lightbox_contentgroup_modes( $values ) {
		$values[] = self::LIGHTBOX_MODE;
		return $values;
	}

	/**
	 * Whether a layer should be skipped
	 *
	 * @param bool $result
	 * @param bool|string $demo
	 * @param Essential_Grid_Item_Skin $skin
	 * @param array $layer
	 * @return bool
	 */
	public function output_item_skin_maybe_skip_layer( $result, $demo, $skin, $layer ) {
		$layer_values = $skin->get_layer_values();
		$post = $skin->get_post();
		
		// skip demo modes and custom grids ( set_layer_values called only in output_by_custom )
		if ( $demo !== false || $layer_values !== false ) {
			return $result;
		}

		$show_on_sale = Essential_Grid_Base::getVar( $layer, [ 'settings', 'show-on-sale' ], 'false' );
		if ( $show_on_sale == 'true' && ! $this->check_if_on_sale( $post['ID'] ) ) {
			return true;
		}
		
		$show_if_featured = Essential_Grid_Base::getVar( $layer, [ 'settings', 'show-if-featured' ], 'false' );
		if ( $show_if_featured == 'true' && ! $this->check_if_is_featured( $post['ID'] ) ) {
			return true;
		}

		return $result;
	}

	/**
	 * Process layer settings source
	 *
	 * @param array $data
	 * @param array $layer
	 * @param bool|string $demo
	 * @param Essential_Grid_Item_Skin $skin
	 * @return array
	 */
	public function process_layer_settings_source( $data, $layer, $demo, $skin ) {
		if ( 'woocommerce' != $layer['settings']['source'] ) {
			$data['demo_element_type'] = '';
			return $data;
		}

		$is_post = empty( $skin->get_layer_values() );
		$separator = Essential_Grid_Base::getVar( $layer, [ 'settings', 'source-separate' ], ',' );
		$catmax = Essential_Grid_Base::getVar( $layer, [ 'settings', 'source-catmax' ], '-1' );
		$source_woo = $layer['settings']['source-woocommerce'] ?: '';

		if ( $demo === false || $demo === 'custom' ) {

			if ($is_post) {
				$data['text'] = '';
				$post = $skin->get_post();
				if ( !empty( $post['ID'] ) ) {
					$catmax = ( ( $catmax = intval( $catmax ) )  > 0 ) ? $catmax : false;
					$data['text'] = $this->get_value_by_meta( $post['ID'], $source_woo, $separator, $catmax );
				}
			} else {
				$data['text'] = $skin->get_custom_element_value($source_woo, $separator);
			}

			if ( $source_woo == 'wc_categories' ) {
				$data['do_limit'] = false;
				$data['do_display'] = false;
				$data['do_full'] = true;
				$data['is_woo_cats'] = true;
			} elseif ( $source_woo == 'wc_add_to_cart_button' ) {
				$data['do_limit'] = false;
				$data['is_woo_button'] = true;
			}
			
		} else {
			
			$wc_meta = $this->get_meta_array();
			$data['text'] = $wc_meta[ $source_woo ] ?? $data['text'];
			
		}

		if ( isset( $data['demo_element_type'] ) ) {
			$data['demo_element_type'] = str_replace('%s', $source_woo, $data['demo_element_type']);
		}
		
		return $data;
	}

}
