<?php
/**
 * User Routes for the REST API.
 *
 * @package Etch
 * @gplv2
 */

declare(strict_types=1);
namespace Etch\RestApi\Routes;

use WP_REST_Response;
use WP_Error;
use WP_REST_Request;

/**
 * UserRoutes
 *
 * This class defines REST API endpoints for retrieving user information.
 *
 * @package Etch\RestApi\Routes
 */
class UserRoutes extends BaseRoute {

	/**
	 * Get the routes for the user.
	 *
	 * @return array<array{
	 *     route: string,
	 *     methods: string,
	 *     callback: callable,
	 *     permission_callback: callable
	 * }> The routes for the user.
	 */
	protected function get_routes(): array {
		return array(
			array(
				'route' => '/user',
				'methods' => 'GET',
				'callback' => array( $this, 'get_user' ),
				'permission_callback' => '__return_true',
			),
		);
	}

	/**
	 * Get the current user information.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @phpstan-param WP_REST_Request<array<string, mixed>> $request
	 * @return WP_REST_Response|WP_Error The user information response or an error.
	 */
	public function get_user( WP_REST_Request $request ): WP_REST_Response|WP_Error {
		$current_user = wp_get_current_user();

		$user_info = array(
			'loggedIn' => is_user_logged_in(),
			'userId' => get_current_user_id(),
			'userLogin' => $current_user->user_login ?? '',
			'email' => $current_user->user_email ?? '',
			'displayName' => $current_user->display_name ?? '',
			'userRole' => $current_user->roles[0] ?? '',
			'userRoles' => $current_user->roles,
		);

		$response = new WP_REST_Response( $user_info, 200 );
		return $response;
	}
}
