<?php

namespace Etn_Pro\Utils;

class Plugin_Installer {
    
    use \Etn_Pro\Traits\Singleton;
        
    public static function make_eventin_ready(){
        
        $basename = 'wp-event-solution/eventin.php';
        $is_plugin_installed 	= self::get_installed_plugin_data( $basename );
        $plugin_data 			= self::get_plugin_data( 'wp-event-solution', $basename );

        if( $is_plugin_installed ) {
            // upgrade plugin - attempt for once
            if( isset( $plugin_data->version ) && $is_plugin_installed['Version'] != $plugin_data->version ) {
                $result = self::upgrade_or_install_plugin( $basename );
                if (is_wp_error($result)) {
                    return false;
                }
            }

            // activate plugin
            if ( is_plugin_active( $basename ) ) {
                return true;
            } else {
                $result = activate_plugin( self::safe_path( WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $basename ), '', false, true );
                if (is_wp_error($result)) {
                    return false;
                }
                return true;
            }
        } else {
            // install & activate plugin
            $download_link = isset( $plugin_data->download_link ) ? $plugin_data->download_link : "";
            if( empty($download_link) ) {
                return false;
            }
            
            $result = self::upgrade_or_install_plugin( $download_link, false );
            if (is_wp_error($result)) {
                return false;
            }
            
            // Check if installation was successful and activate
            if ($result) {
                $installed_plugin = self::get_installed_plugin_data( $basename );
                if ($installed_plugin) {
                    $activate_result = activate_plugin( self::safe_path( WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $basename ), '', false, true );
                    if (is_wp_error($activate_result)) {
                        return false;
                    }
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Check if Eventin free plugin is properly installed and activated
     *
     * @return bool
     */
    public static function is_eventin_ready() {
        $basename = 'wp-event-solution/eventin.php';
        
        // Check if plugin is installed
        $is_plugin_installed = self::get_installed_plugin_data( $basename );
        if (!$is_plugin_installed) {
            return false;
        }
        
        // Check if plugin is active
        if (!is_plugin_active( $basename )) {
            return false;
        }
        
        // Check if the plugin has been loaded
        if (!did_action('eventin/after_load')) {
            return false;
        }
        
        return true;
    }

    /**
     * Get installation status with detailed information
     *
     * @return array
     */
    public static function get_installation_status() {
        $basename = 'wp-event-solution/eventin.php';
        $status = [
            'installed' => false,
            'activated' => false,
            'loaded' => false,
            'ready' => false,
            'version' => null,
            'plugin_data' => null
        ];
        
        // Check if plugin is installed
        $is_plugin_installed = self::get_installed_plugin_data( $basename );
        $status['installed'] = !empty($is_plugin_installed);
        
        if ($status['installed']) {
            $status['plugin_data'] = $is_plugin_installed;
            $status['version'] = isset($is_plugin_installed['Version']) ? $is_plugin_installed['Version'] : null;
            
            // Check if plugin is active
            $status['activated'] = is_plugin_active( $basename );
            
            // Check if the plugin has been loaded
            $status['loaded'] = did_action('eventin/after_load');
            
            // Overall ready status
            $status['ready'] = $status['installed'] && $status['activated'] && $status['loaded'];
        }
        
        return $status;
    }
    
    private static function get_plugin_data( $slug = '', $basename = '' ){
        if( empty( $slug ) ) {
            return false;
        }
        $installed_plugin = false;
        if( $basename ) {
            $installed_plugin = self::get_installed_plugin_data( $basename );
        }

        if( $installed_plugin ) {
            return $installed_plugin;
        }

        $args = array(
            'slug' => $slug,
            'fields' => array(
                'version' => false,
            ),
        );

        $response = wp_remote_post(
            'http://api.wordpress.org/plugins/info/1.0/',
            array(
                'body' => array(
                    'action' => 'plugin_information',
                    'request' => serialize((object) $args),
                ),
                'timeout' => 30,
            )
        );

        if ( is_wp_error( $response ) ) {
            return false;
        } else {
            $response = unserialize( wp_remote_retrieve_body( $response ) );

            if( $response ) {
                return $response;
            } else {
                return false;
            }
        }
    }

    private static function get_installed_plugin_data( $basename = '' ) {
        if( empty( $basename ) ) {
            return false;
        }
        if( ! function_exists( 'get_plugins' ) ) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        return isset( $plugins[ $basename ] ) ? $plugins[ $basename ] : false;
    }

    private static function upgrade_or_install_plugin( $basename = '', $upgrade = true ){
        if( empty( $basename ) ) {
            return false;
        }
        include_once ABSPATH . 'wp-admin/includes/file.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        include_once ABSPATH . 'wp-admin/includes/class-automatic-upgrader-skin.php';

        $skin = new \Automatic_Upgrader_Skin;
        $upgrader = new \Plugin_Upgrader( $skin );
        
        if( $upgrade == true ) {
            $result = $upgrader->upgrade( $basename );
        } else {
            $result = $upgrader->install( $basename );
        }
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return $skin->result;
    }
    
    public static function safe_path( $path ) {
        $path = str_replace( ['//', '\\\\'], ['/', '\\'], $path );
        return str_replace( ['/', '\\'], DIRECTORY_SEPARATOR, $path );
    }
}