<?php

namespace ExternalImporter\application;

defined('\ABSPATH') || exit;

use ExternalImporter\application\admin\WooConfig;
use ExternalImporter\application\components\Scheduler;
use ExternalImporter\application\components\scrap\ScrapFactory;
use ExternalImporter\application\helpers\WooHelper;

/**
 * GalleryScheduler class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class GalleryScheduler extends Scheduler
{
    const CRON_TAG = 'ei_gallery_download';
    const GALLERY_META = '_ei_gallery';
    const PRODUCT_LIMIT = 20;

    public static function getCronTag()
    {
        return self::CRON_TAG;
    }

    public static function initAction()
    {
        self::initSchedule();
        parent::initAction();

        add_action('woocommerce_before_single_product', [__CLASS__, 'maybeShowGalleryAsyncNotice']);
    }

    public static function run(): void
    {
        $lock_key = self::CRON_TAG . '_lock';

        if (get_transient($lock_key))
        {
            return;
        }

        set_transient($lock_key, true, 15 * MINUTE_IN_SECONDS);

        global $wpdb;

        $limit = (int) apply_filters('cegg_gallery_products_limit', self::PRODUCT_LIMIT);

        $sql = $wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s ORDER BY meta_id ASC LIMIT %d",
            self::GALLERY_META,
            $limit
        );

        $results = $wpdb->get_results($sql, ARRAY_A);
        if (empty($results))
        {
            self::clearScheduleEvent();
            delete_transient($lock_key);
            return;
        }

        @set_time_limit(600);

        foreach ($results as $row)
        {
            self::downloadGalleryImages((int) $row['post_id']);
        }

        delete_transient($lock_key);
    }

    public static function downloadGalleryImages($product_id)
    {
        if (!WooHelper::isWooInstalled())
        {
            return;
        }

        if (!$product = \wc_get_product($product_id))
        {
            return;
        }

        $images = \get_post_meta($product_id, GalleryScheduler::GALLERY_META, true);

        ScrapFactory::init();

        if ($media_ids = WooHelper::uploadMedias($images, $product->get_id(), $product->get_title()))
        {
            $product->set_gallery_image_ids($media_ids);
        }

        $product->save();

        \delete_post_meta($product_id, GalleryScheduler::GALLERY_META);
    }

    public static function addGalleryTask($product_id, array $images)
    {
        \update_post_meta($product_id, GalleryScheduler::GALLERY_META, $images);
        GalleryScheduler::addScheduleEvent('five_min');
    }

    public static function initSchedule()
    {
        \add_filter('cron_schedules', array(__CLASS__, 'addSchedule'));
    }

    public static function addSchedule($schedules)
    {
        $schedules['five_min'] = array(
            'interval' => 60 * 5,
            'display' => __('Every 5 minutes'),
        );

        return $schedules;
    }

    public static function maybeShowGalleryAsyncNotice()
    {
        if (is_admin())
        {
            return;
        }

        $product_id = get_the_ID();
        if (! $product_id || ! current_user_can('edit_product', $product_id))
        {
            return;
        }

        if (WooConfig::getInstance()->option('import_gallery') !== 'enabled')
        {
            return;
        }

        $images = get_post_meta($product_id, GalleryScheduler::GALLERY_META, true);
        if (empty($images) || !is_array($images))
        {
            return;
        }

        $count   = count($images);
        $message = sprintf(
            _n(
                'Gallery images are loading asynchronously in the background. %s image is currently queued for this product.',
                'Gallery images are loading asynchronously in the background. %s images are currently queued for this product.',
                $count,
                'external-importer'
            ),
            number_format_i18n($count)
        );

        wc_print_notice(esc_html($message), 'notice');

        if (!\wp_next_scheduled(GalleryScheduler::getCronTag()))
        {
            GalleryScheduler::addScheduleEvent('five_min');
        }
    }
}
