<?php

namespace ExternalImporter\application;

defined('\ABSPATH') || exit;

use ExternalImporter\application\Plugin;
use ExternalImporter\application\admin\LicConfig;
use ExternalImporter\application\admin\ParserConfig;
use ExternalImporter\application\SyncScheduler;
use ExternalImporter\application\GalleryScheduler;
use ExternalImporter\application\AutoimportSheduler;
use ExternalImporter\application\helpers\TextHelper;

use function ExternalImporter\prnx;

/**
 * Installer class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class Installer
{

    private static $instance = null;

    public static function getInstance()
    {

        if (self::$instance == null)
            self::$instance = new self;

        return self::$instance;
    }

    private function __construct()
    {
        if (!empty($GLOBALS['pagenow']) && $GLOBALS['pagenow'] == 'plugins.php')
        {
            \add_action('admin_init', array($this, 'requirements'), 0);
        }

        \add_action('admin_init', array($this, 'upgrade'));
        \add_action('admin_init', array($this, 'redirect_after_activation'));
    }

    static public function dbVesrion()
    {
        return Plugin::db_version;
    }

    public static function activate()
    {
        if (!\current_user_can('activate_plugins'))
            return;

        self::requirements();
        \add_option(Plugin::slug . '_do_activation_redirect', true);
        \add_option(Plugin::slug . '_first_activation_date', time());
        self::upgradeTables();

        SyncScheduler::maybeAddScheduleEvent();
        AutoimportSheduler::maybeAddScheduleEvent();
    }

    public static function deactivate()
    {
        SyncScheduler::clearScheduleEvent();
        GalleryScheduler::clearScheduleEvent();
        AutoimportSheduler::clearScheduleEvent();
    }

    public static function requirements()
    {
        $php_min_version = '7.4';
        $extensions = array(
            'simplexml',
            'mbstring',
        );
        $plugins = array();

        $errors = array();

        global $wp_version;
        if (version_compare(Plugin::wp_requires, $wp_version, '>'))
            $errors[] = sprintf('You are using WordPress %s. <em>%s</em> requires at least <strong>WordPress %s</strong>.', $wp_version, Plugin::name, Plugin::wp_requires);

        $php_current_version = phpversion();
        if (version_compare($php_min_version, $php_current_version, '>'))
            $errors[] = sprintf('PHP is installed on your server %s. <em>%s</em> requires at least <strong>PHP %s</strong>.', $php_current_version, Plugin::name, $php_min_version);

        foreach ($extensions as $extension)
        {
            if (!extension_loaded($extension))
                $errors[] = sprintf('Requires PHP extension <strong>%s</strong>.', $extension);
        }
        foreach ($plugins as $plugin_id => $plugin)
        {
            if (!\is_plugin_active($plugin_id) || \version_compare($plugin['version'], self::getPluginVersion($plugin_id), '>'))
                $errors[] = sprintf('<em>%s</em> requires <strong>%s %s+</strong> to be installed and active.', Plugin::name, $plugin['name'], $plugin['version']);
        }

        if (!$errors)
            return;
        unset($_GET['activate']);
        \deactivate_plugins(\plugin_basename(\ExternalImporter\PLUGIN_FILE));
        $e = sprintf('<div class="error"><p>%1$s</p><p><em>%2$s</em> ' . 'cannot be installed!' . '</p></div>', join('</p><p>', $errors), Plugin::name);
        \wp_die($e);
    }

    public static function uninstall()
    {
        global $wpdb;
        if (!current_user_can('activate_plugins'))
            return;

        \delete_option(Plugin::slug . '_db_version');
        \delete_option(Plugin::getShortSlug() . '_sys_status');
        \delete_option(Plugin::getShortSlug() . '_sys_deadline');

        if (Plugin::isEnvato())
            \delete_option(Plugin::slug . '_env_install');
        if (Plugin::isPro())
            \delete_option(LicConfig::getInstance()->option_name());
    }

    public static function upgrade()
    {
        $db_version = get_option(Plugin::slug . '_db_version');
        if ($db_version >= self::dbVesrion())
            return;
        self::upgradeTables();

        if ($db_version < 27)
        {
            self::upgrade_v27();
        }

        if ($db_version < 28)
        {
            self::upgrade_v28();
        }

        \update_option(Plugin::slug . '_db_version', self::dbVesrion());
    }

    private static function upgradeTables()
    {
        $models = array('TaskModel', 'QueryModel', 'LogModel', 'AutoimportModel', 'AutoimportItemModel');
        $sql = '';
        foreach ($models as $model)
        {
            $m = "\\ExternalImporter\\application\\models\\" . $model;
            $sql .= $m::model()->getDump();
            $sql .= "\r\n";
        }
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    private static function upgrade_v27()
    {
        if (!Plugin::isFree())
            SystemScheduler::addScheduleEvent('weekly', time() + rand(259200, 604800));
    }

    private static function upgrade_v28()
    {
        // One-time migration:
        //  - *_domains => routing_rules
        //  - proxycrawl_domains => crawlbase routing rules
        //  - proxycrawl_token   => crawlbase_token

        $providers    = ParserConfig::get_scraping_providers();
        $provider_ids = array_keys($providers);

        // Read existing settings
        $settings = get_option('external-importer-settings-parser', array());
        if (!is_array($settings))
        {
            $settings = array();
        }

        $routing_rules = array();
        if (!empty($settings['routing_rules']) && is_array($settings['routing_rules']))
        {
            $routing_rules = $settings['routing_rules'];
        }

        foreach ($provider_ids as $provider_id)
        {
            // Normal key: <provider>_domains
            $domain_keys = array($provider_id . '_domains');

            // Historical name: proxycrawl_domains should be treated as crawlbase_domains
            if ($provider_id === 'crawlbase')
            {
                $domain_keys[] = 'proxycrawl_domains';
            }

            foreach ($domain_keys as $option_key)
            {
                if (!isset($settings[$option_key]))
                {
                    continue;
                }

                $option_value = $settings[$option_key];

                if (!is_string($option_value) || $option_value === '')
                {
                    // Nothing useful here; clean up and move on
                    unset($settings[$option_key]);
                    continue;
                }

                // Convert comma-separated list into array
                $domains = TextHelper::commaListArray($option_value);
                if (empty($domains) || !is_array($domains))
                {
                    unset($settings[$option_key]);
                    continue;
                }

                foreach ($domains as $domain)
                {
                    $domain = trim($domain);
                    if ($domain === '')
                    {
                        continue;
                    }

                    $routing_rules[] = array(
                        'pattern'  => $domain,
                        'provider' => $provider_id,
                        'params'   => '',
                    );
                }

                // Remove old *_domains / proxycrawl_domains option
                unset($settings[$option_key]);
            }
        }

        if (!empty($routing_rules))
        {
            $routing_rules = ParserConfig::formatRoutingRules($routing_rules);
        }

        // Save new routing rules back into settings
        $settings['routing_rules'] = $routing_rules;

        // Migrate proxycrawl_token -> crawlbase_token (only if crawlbase_token not set or empty)
        $crawlbase_token  = isset($settings['crawlbase_token']) ? $settings['crawlbase_token'] : '';
        $proxycrawl_token = isset($settings['proxycrawl_token']) ? $settings['proxycrawl_token'] : '';

        if (($crawlbase_token === '' || $crawlbase_token === null) && $proxycrawl_token !== '')
        {
            $settings['crawlbase_token'] = $proxycrawl_token;
        }

        // Remove old proxycrawl_token key to avoid confusion
        if (isset($settings['proxycrawl_token']))
        {
            unset($settings['proxycrawl_token']);
        }

        update_option('external-importer-settings-parser', $settings);
    }

    public function redirect_after_activation()
    {
        if (\get_option(Plugin::slug . '_do_activation_redirect', false))
        {
            \delete_option(Plugin::slug . '_do_activation_redirect');
            \wp_redirect(\get_admin_url(\get_current_blog_id(), 'admin.php?page=' . Plugin::slug()));
        }
    }

    public static function getPluginVersion($plugin_file)
    {
        $data = \get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_file);
        if (isset($data['Version']))
            return $data['Version'];
        else
            return 0;
    }
}
