<?php

namespace ExternalImporter\application;

defined('\ABSPATH') || exit;

use ExternalImporter\application\components\Scheduler;
use ExternalImporter\application\components\WooImporter;
use ExternalImporter\application\admin\SyncConfig;
use ExternalImporter\application\admin\SyncMetabox;
use ExternalImporter\application\components\Synchronizer;
use ExternalImporter\application\components\Throttler;

use function ExternalImporter\prnx;

/**
 * SyncScheduler class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class SyncScheduler extends Scheduler
{
    const CRON_TAG = 'ei_sync_products';
    const PRODUCT_LIMIT = 2;
    const SLEEP_MIN = 2;
    const SLEEP_MAX = 6;

    public static function getCronTag()
    {
        return self::CRON_TAG;
    }

    public static function initAction()
    {
        self::initSchedule();
        parent::initAction();
    }

    public static function getProductLimit()
    {
        return \apply_filters('ei_sync_product_limit', self::PRODUCT_LIMIT);
    }

    public static function run()
    {
        global $wpdb;

        @set_time_limit(595);
        $time      = time();
        $ttl       = round((float) SyncConfig::getInstance()->option('cache_duration') * 86400);
        $throttled = Throttler::getThrottledDomains();

        $meta_table  = $wpdb->postmeta;
        $posts_table = $wpdb->posts;

        // Build JOIN part
        $join  = " JOIN {$posts_table} AS post ON last_update.post_id = post.ID";

        if ($throttled)
        {
            $join .= " JOIN {$meta_table} AS domain ON last_update.post_id = domain.post_id";
        }

        // Exclude products with sync disabled via LEFT JOIN + IS NULL
        $join .= $wpdb->prepare(
            " LEFT JOIN {$meta_table} AS disabled_meta
              ON last_update.post_id = disabled_meta.post_id
             AND disabled_meta.meta_key = %s
             AND disabled_meta.meta_value = %s",
            SyncMetabox::META_SYNC_DISABLED,
            '1'
        );

        // Build WHERE part
        $where = $wpdb->prepare(
            " WHERE %d - last_update.meta_value > %d
                 AND last_update.meta_key = %s",
            $time,
            $ttl,
            WooImporter::META_LAST_UPDATE
        );
        $where .= " AND post.post_type = 'product'";

        if ($throttled)
        {
            $where .= $wpdb->prepare(" AND domain.meta_key = %s", WooImporter::META_PRODUCT_DOMAIN);
        }

        if (SyncConfig::getInstance()->option('published_only'))
        {
            $where .= " AND post.post_status = 'publish'";
        }

        // Only pick rows where no "disabled" meta exists
        $where .= " AND disabled_meta.post_id IS NULL";

        /**
         * Include only specified product IDs (kept in SQL to limit rows early)
         * Usage: apply_filters('ei_enable_sync_product_ids', [1,2,3]);
         */
        if ($enable_sync_ids = \apply_filters('ei_enable_sync_product_ids', []))
        {
            $enable_sync_ids = array_map('absint', $enable_sync_ids);
            $enable_sync_ids = array_filter($enable_sync_ids);

            if (!empty($enable_sync_ids))
            {
                $where .= " AND post.ID IN (" . join(',', $enable_sync_ids) . ")";
            }
        }

        if ($throttled)
        {
            $throttled_sql = array_map(function ($v)
            {
                return "'" . \esc_sql($v) . "'";
            }, $throttled);

            $where .= ' AND domain.meta_value NOT IN (' . implode(',', $throttled_sql) . ')';
        }

        // Final SQL
        $sql = "SELECT last_update.post_id
            FROM {$meta_table} AS last_update
            {$join}
            {$where}";

        $sql .= $wpdb->prepare(" ORDER BY last_update.meta_value ASC LIMIT %d", self::getProductLimit());

        $product_ids = $wpdb->get_col($sql);

        if (!$product_ids)
        {
            return;
        }

        $product_ids = array_map('absint', $product_ids);
        $product_ids = array_unique($product_ids);

        /**
         * Filter: exclude specified product IDs
         * Usage: apply_filters('ei_disable_sync_product_ids', [4,5,6]);
         */
        if ($disable_sync_product_ids = \apply_filters('ei_disable_sync_product_ids', []))
        {
            $disable_sync_product_ids = array_map('absint', (array) $disable_sync_product_ids);
            $disable_sync_product_ids = array_filter($disable_sync_product_ids);

            if (!empty($disable_sync_product_ids))
            {
                $product_ids = array_diff($product_ids, $disable_sync_product_ids);
            }
        }

        /**
         * Final hook to let devs adjust the list of IDs cron will process.
         *
         * Usage: apply_filters('ei_sync_cron_product_ids', $product_ids);
         */
        $product_ids = \apply_filters('ei_sync_cron_product_ids', $product_ids);

        if (empty($product_ids))
        {
            return;
        }

        // Randomize order to spread load
        shuffle($product_ids);

        foreach ($product_ids as $product_id)
        {
            Synchronizer::maybeUpdateProduct($product_id);
            sleep(rand(self::SLEEP_MIN, self::SLEEP_MAX));
        }
    }

    public static function initSchedule()
    {
        \add_filter('cron_schedules', array(__CLASS__, 'addSchedule'));
    }

    public static function addSchedule($schedules)
    {
        $schedules['ten_min'] = array(
            'interval' => 60 * 10,
            'display' => __('Every 10 minutes'),
        );
        return $schedules;
    }

    public static function maybeAddScheduleEvent()
    {
        if (!(float) SyncConfig::getInstance()->option('cache_duration'))
            return;

        if (SyncConfig::getInstance()->option('update_mode') != 'cron')
            return;

        self::initSchedule();
        self::addScheduleEvent('ten_min');
    }
}
