<?php

namespace ExternalImporter\application\admin;

use ExternalImporter\application\components\WooImporter;
use ExternalImporter\application\components\Synchronizer;
use ExternalImporter\application\FrontendViewer;
use ExternalImporter\application\helpers\WooHelper;

defined('\ABSPATH') || exit;

/**
 * SyncListColumn class file
 *
 * Integrates External Importer sync information into the WooCommerce Products list table:
 *
 * Only External Importer products are affected; all other products use standard WooCommerce behavior.
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class SyncListColumn
{
    public function __construct()
    {
        if (! (float) SyncConfig::getInstance()->option('cache_duration'))
        {
            return;
        }

        // Adjust column widths in the Products list table.
        \add_action('admin_head', array($this, 'adminColumnStyles'));

        // Replace WooCommerce admin "Stock" column HTML for External Importer products.
        \add_filter('woocommerce_admin_stock_html', array($this, 'filterAdminStockHtml'), 10, 2);

        // Append sync details to the default "Price" column.
        \add_action('manage_product_posts_custom_column', array($this, 'renderPriceColumnMeta'), 20, 2);

        // Add "Source product" row action under the product Name column.
        \add_filter('post_row_actions', array($this, 'addSourceProductRowAction'), 10, 2);

        // Add "Sync now" row action under the product Name column.
        \add_filter('post_row_actions', array($this, 'addSyncNowRowAction'), 11, 2);

        // AJAX handler for row "Sync now".
        \add_action('wp_ajax_ei_sync_now_row', array($this, 'ajaxSyncNowRow'));

        // Inline JS to handle "Sync now" clicks and update the row.
        \add_action('admin_footer-edit.php', array($this, 'printInlineJs'));

        \add_filter('woocommerce_currency_symbol', array(FrontendViewer::class, 'changeCurrencySymbol'), 10, 2);
    }

    /**
     * Output admin CSS to adjust the width of Stock and Price columns
     * (and optionally nudge a couple of others to make room).
     *
     * Runs only on the Products list screen.
     */
    public function adminColumnStyles()
    {
        $screen = \get_current_screen();

        if (! $screen || $screen->id !== 'edit-product')
        {
            return;
        }
?>
        <style>
            /* Stock column */
            .wp-list-table.widefat.fixed.striped.posts th.column-is_in_stock,
            .wp-list-table.widefat.fixed.striped.posts td.column-is_in_stock {
                width: 110px;
                max-width: 130px;
            }

            /* Price column */
            .wp-list-table.widefat.fixed.striped.posts th.column-price,
            .wp-list-table.widefat.fixed.striped.posts td.column-price {
                width: 180px;
                max-width: 230px;
            }

            /* Slightly shrink some other columns to make room  */
            .wp-list-table.widefat.fixed.striped.posts th.column-sku,
            .wp-list-table.widefat.fixed.striped.posts td.column-sku {
                width: 80px;
            }

            .wp-list-table.widefat.fixed.striped.posts th.column-product_cat,
            .wp-list-table.widefat.fixed.striped.posts td.column-product_cat {
                width: 10%;
            }
        </style>
    <?php
    }

    /**
     * Override WooCommerce admin "Stock" column HTML for External Importer products.
     *
     * For EI products we replace the default Woo stock badge with:
     *
     *   <mark class="instock">In stock</mark>
     *   <mark class="outofstock">Out of stock</mark>
     *   <mark class="onbackorder">On backorder</mark>
     *
     * For non-EI products, the original WooCommerce HTML is preserved.
     *
     * @param string          $stock_html Existing stock HTML from WooCommerce.
     * @param \WC_Product|int $product    Product object (normally) or product ID.
     *
     * @return string
     */
    public function filterAdminStockHtml($stock_html, $product)
    {
        if ($product instanceof \WC_Product)
        {
            $post_id = $product->get_id();
        }
        elseif (\is_numeric($product))
        {
            $post_id = (int) $product;
        }
        else
        {
            return $stock_html;
        }

        if (! $post_id)
        {
            return $stock_html;
        }

        // Only adjust for External Importer products.
        if (! WooHelper::isEiProduct($post_id))
        {
            return $stock_html;
        }

        $status = SyncStatus::get($post_id);

        return $this->buildStockHtmlFromStatus($status) ?: $stock_html;
    }

    /**
     * Append EI sync meta under the default Price column content for EI products.
     *
     * Example:
     *
     *  $15.00
     *  Synced: 28 minutes ago
     *  ❙❙ Auto-sync off
     *  ⚠ Error
     *
     * @param string $column  Column key.
     * @param int    $post_id Product ID.
     *
     * @return void
     */
    public function renderPriceColumnMeta($column, $post_id)
    {
        if ($column !== 'price')
        {
            return;
        }

        $post_id = (int) $post_id;
        if (! $post_id)
        {
            return;
        }

        // Only for External Importer products.
        if (! WooHelper::isEiProduct($post_id))
        {
            return;
        }

        if (! SyncConfig::getInstance()->option('sync_price'))
        {
            return;
        }

        $status = SyncStatus::get($post_id);

        $html = $this->buildPriceMetaHtmlFromStatus($status);

        if ($html)
        {
            echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        }
    }

    /**
     * Add "Source product" link to the row actions under the product Name column.
     *
     * @param array    $actions Existing row actions.
     * @param \WP_Post $post    Post object.
     *
     * @return array
     */
    public function addSourceProductRowAction($actions, $post)
    {
        // Only for products.
        if (! $post instanceof \WP_Post || $post->post_type !== 'product')
        {
            return $actions;
        }

        // Only for External Importer products.
        if (! WooHelper::isEiProduct($post->ID))
        {
            return $actions;
        }

        // Get EI product meta to retrieve source link.
        $product = WooImporter::getProductMeta($post->ID);

        if (! $product || empty($product->link))
        {
            return $actions;
        }

        $actions['ei_source_product'] =
            '<a href="' . \esc_url($product->link) . '" target="_blank" rel="noopener noreferrer">'
            . \esc_html__('Source product', 'external-importer')
            . ' <span aria-hidden="true">↗</span>'
            . '</a>';

        return $actions;
    }

    /**
     * Add "Sync now" AJAX action link under the product Name column.
     *
     * @param array    $actions Existing row actions.
     * @param \WP_Post $post    Post object.
     *
     * @return array
     */
    public function addSyncNowRowAction($actions, $post)
    {
        // Only for products.
        if (! $post instanceof \WP_Post || $post->post_type !== 'product')
        {
            return $actions;
        }

        // Only for External Importer products.
        if (! WooHelper::isEiProduct($post->ID))
        {
            return $actions;
        }

        $nonce = \wp_create_nonce('ei_sync_now_row');

        $actions['ei_sync_now'] =
            '<a href="#"'
            . ' class="ei-sync-now"'
            . ' data-product-id="' . (int) $post->ID . '"'
            . ' data-nonce="' . \esc_attr($nonce) . '">'
            . \esc_html__('Sync now', 'external-importer')
            . '</a>';

        return $actions;
    }

    /**
     * AJAX handler for "Sync now" row action.
     *
     * Returns JSON:
     *  {
     *    success: true,
     *    data: {
     *      product_id: 123,
     *      stock_html: "...",
     *      price_html: "...",      // EI sync meta HTML
     *      wc_price_html: "..."    // WooCommerce price HTML
     *    }
     *  }
     */
    public function ajaxSyncNowRow()
    {
        \check_ajax_referer('ei_sync_now_row');

        // Allow longer-running sync
        @set_time_limit(300);

        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;

        if (! $product_id)
        {
            \wp_send_json_error(array('message' => __('Invalid product ID.', 'external-importer')));
        }

        if (! \current_user_can('edit_post', $product_id))
        {
            \wp_send_json_error(array('message' => __('You are not allowed to edit this product.', 'external-importer')));
        }

        if (! WooHelper::isEiProduct($product_id))
        {
            \wp_send_json_error(array('message' => __('Not an External Importer product.', 'external-importer')));
        }

        // Perform the sync (may take a while).
        Synchronizer::updateProduct($product_id);

        // Rebuild status & HTML fragments.
        $status     = SyncStatus::get($product_id);
        $stock_html = $this->buildStockHtmlFromStatus($status);
        $price_html = $this->buildPriceMetaHtmlFromStatus($status);

        // Get updated WooCommerce price HTML.
        $wc_price_html = '';
        if (\function_exists('wc_get_product'))
        {
            $wc_product = \wc_get_product($product_id);
            if ($wc_product)
            {
                $wc_price_html = $wc_product->get_price_html();
            }
        }

        \wp_send_json_success(array(
            'product_id'    => $product_id,
            'stock_html'    => $stock_html,
            'price_html'    => $price_html,     // EI sync meta
            'wc_price_html' => $wc_price_html,  // WC price HTML
        ));
    }

    /**
     * Print inline JS in the Products list footer to handle "Sync now" clicks.
     *
     * Runs only on edit.php?post_type=product
     */
    public function printInlineJs()
    {
        $screen = \get_current_screen();

        if (! $screen || $screen->id !== 'edit-product')
        {
            return;
        }
    ?>
        <script>
            jQuery(document).ready(function($) {
                $(document).on('click', 'a.ei-sync-now', function(e) {
                    e.preventDefault();

                    var $link = $(this);
                    var productId = $link.data('product-id');
                    var nonce = $link.data('nonce');

                    if (!productId || !nonce) {
                        return;
                    }

                    if ($link.data('ei-syncing') === 1) {
                        // Prevent double-clicks.
                        return;
                    }

                    var originalText = $link.text();
                    $link.data('ei-syncing', 1);
                    $link.text('<?php echo \esc_js(__('Syncing...', 'external-importer')); ?>');

                    $.ajax({
                            type: 'POST',
                            url: ajaxurl,
                            dataType: 'json',
                            timeout: 60000,
                            data: {
                                action: 'ei_sync_now_row',
                                product_id: productId,
                                _ajax_nonce: nonce
                            }
                        })
                        .done(function(response) {
                            if (!response || !response.success || !response.data) {
                                var msg = (response && response.data && response.data.message) ?
                                    response.data.message :
                                    '<?php echo \esc_js(__('Unable to sync product.', 'external-importer')); ?>';
                                alert(msg);
                                return;
                            }

                            var data = response.data;
                            var $row = $('#post-' + data.product_id);

                            if ($row.length) {
                                // Update Stock column (HTML from server already escaped).
                                if (typeof data.stock_html === 'string') {
                                    $row.find('td.column-is_in_stock').html(data.stock_html);
                                }

                                var $priceCell = $row.find('td.column-price');

                                // If we have fresh WooCommerce price HTML, replace the whole cell with new price + sync meta.
                                if (typeof data.wc_price_html === 'string' && data.wc_price_html.length > 0) {
                                    var newHtml = data.wc_price_html;

                                    if (typeof data.price_html === 'string' && data.price_html.length > 0) {
                                        newHtml += data.price_html;
                                    }

                                    $priceCell.html(newHtml);
                                } else if (typeof data.price_html === 'string') {
                                    // Fallback: only update our EI sync meta.
                                    $priceCell.find('.ei-sync-meta').remove();
                                    if (data.price_html.length > 0) {
                                        $priceCell.append(data.price_html);
                                    }
                                }
                            }
                        })
                        .fail(function(jqXHR, textStatus) {
                            var msg;
                            if (textStatus === 'timeout') {
                                msg = '<?php echo \esc_js(__('The sync request timed out. The product may still be syncing in the background.', 'external-importer')); ?>';
                            } else {
                                msg = '<?php echo \esc_js(__('Unable to sync product.', 'external-importer')); ?>';
                            }
                            alert(msg);
                        })
                        .always(function() {
                            $link.data('ei-syncing', 0);
                            $link.text(originalText);
                        });
                });
            });
        </script>
<?php
    }

    /**
     * Build the Stock column HTML from a SyncStatus::get() result.
     *
     * @param array $status
     * @return string
     */
    protected function buildStockHtmlFromStatus(array $status)
    {
        if (empty($status['availability']))
        {
            return '';
        }

        $availability_label = $status['availability_label'] ?: __('Unknown', 'external-importer');
        $availability_class = '';

        if ($status['availability'] === SyncStatus::AVAIL_IN_STOCK)
        {
            $availability_class = 'instock';
        }
        elseif ($status['availability'] === SyncStatus::AVAIL_BACKORDER)
        {
            $availability_class = 'onbackorder';
        }
        elseif ($status['availability'] === SyncStatus::AVAIL_OUT_STOCK)
        {
            $availability_class = 'outofstock';
        }

        if ($availability_class)
        {
            return '<mark class="' . \esc_attr($availability_class) . '">'
                . \esc_html($availability_label)
                . '</mark>';
        }

        return '<mark>' . \esc_html($availability_label) . '</mark>';
    }

    /**
     * Build the Price column EI sync meta HTML from a SyncStatus::get() result.
     *
     * Wrapped in a .ei-sync-meta container so JS can replace it easily.
     *
     * @param array $status
     * @return string
     */
    protected function buildPriceMetaHtmlFromStatus(array $status)
    {
        // If we have literally no sync info, don't output anything.
        if (
            empty($status['last_synced_ts'])
            && empty($status['sync_disabled'])
            && (empty($status['has_error']) || empty($status['status_message']))
        )
        {
            return '';
        }

        $html  = '<div class="ei-sync-meta" style="margin-top:3px;">';

        // 1) Last synced
        if (! empty($status['last_synced_ts']) && ! empty($status['last_synced_rel']))
        {
            $html .= '<div class="ei-sync-row-last-synced">';
            $html .= '<span style="font-size:11px; color:#555;">'
                . \esc_html__('Synced:', 'external-importer')
                . ' </span>';
            $html .= '<span style="font-size:11px;"'
                .       ' title="' . \esc_attr($status['last_synced_full']) . '">'
                .   \esc_html($status['last_synced_rel']) .
                '</span>';
            $html .= '</div>';
        }

        // 2) Auto-sync status (only show when disabled)
        if (! empty($status['sync_disabled']))
        {
            $html .= '<div class="ei-sync-row-auto" style="margin-top:1px;">';
            $html .= '<span style="font-size:11px; color:#b32d2e;">'
                . '<span style="font-size:10px;">&#10073;&#10073;</span> '
                . \esc_html__('Auto-sync off', 'external-importer')
                . '</span>';
            $html .= '</div>';
        }

        // 3) Throttled indicator
        if (! empty($status['throttled']))
        {
            $html .= '<div class="ei-sync-row-throttled" style="margin-top:1px;">';
            $html .= '<span style="font-size:11px; color:#d38b00;">'
                . '&#9888; ' . esc_html__('Currently throttled', 'external-importer')
                . '</span>';
            $html .= '</div>';
        }

        // 4) Tiny error indicator (if last sync ended in error)
        if (! empty($status['has_error']) && ! empty($status['status_message']))
        {
            $html .= '<div class="ei-sync-row-error"'
                .     ' style="margin-top:1px; max-width:160px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;">'
                . '<span style="font-size:11px; color:#d63638;"'
                .       ' title="' . \esc_attr($status['status_message']) . '">'
                .   '&#9888; ' . \esc_html__('Error', 'external-importer')
                . '</span>'
                . '</div>';
        }

        $html .= '</div>';

        return $html;
    }
}
