<?php

namespace ExternalImporter\application\admin;

defined('\ABSPATH') || exit;

use ExternalImporter\application\helpers\WooHelper;
use ExternalImporter\application\components\WooImporter;
use ExternalImporter\application\helpers\InputHelper;
use ExternalImporter\application\components\Synchronizer;
use ExternalImporter\application\LinkProcessor;

/**
 * SyncMetabox class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class SyncMetabox
{
    const META_SYNC_DISABLED = '_ei_sync_disabled';

    public function __construct()
    {
        \add_action('add_meta_boxes', array($this, 'addMetabox'));
        \add_action('wp_ajax_ei_update_product', array($this, 'ajaxUpdateProduct'));
        \add_action('wp_ajax_ei_toggle_sync', array($this, 'ajaxToggleSync'));
    }

    public function addMetabox($post_type)
    {
        if ($post_type !== 'product')
            return;

        global $post;

        if (!WooHelper::isEiProduct($post->ID))
            return;

        \add_meta_box(
            'ei_stat_metabox',
            __('External Importer Sync', 'external-importer'),
            array($this, 'renderMetabox'),
            $post_type,
            'side',
            'default'
        );
    }

    public static function renderMetabox($post)
    {
        echo '<div id="ei_update_metabox_response">';
        echo self::metaboxHtml($post->ID);
        echo '</div>';

        $product       = WooImporter::getProductMeta($post->ID);
        $sync_disabled = (int) \get_post_meta($post->ID, self::META_SYNC_DISABLED, true);
?>

        <!-- per-product sync on/off toggle -->
        <div style="padding-top: 10px;">
            <?php \wp_nonce_field('ei_toggle_sync', 'ei_toggle_sync_nonce'); ?>
            <label for="ei_disable_sync"
                title="<?php echo esc_attr__('Disable automatic synchronization for this product', 'external-importer'); ?>">
                <input type="checkbox"
                    id="ei_disable_sync"
                    <?php checked($sync_disabled, 1); ?> />
                <?php echo esc_html__('Disable auto-sync', 'external-importer'); ?>
            </label>
        </div>

        <div style="padding-top:10px;">
            <?php wp_nonce_field('ei_update_product', 'ei_nonce'); ?>

            <div style="
                display:flex;
                justify-content:space-between;
                align-items:center;
                gap:10px;
                flex-wrap:wrap;
            ">
                <div>
                    <div>
                        <a title="<?php echo esc_attr(__('View product', 'external-importer')); ?>"
                            target="_blank"
                            rel="noopener"
                            href="<?php echo esc_url($product->link); ?>">
                            <?php echo esc_html(__('Direct link', 'external-importer')); ?>
                            <span aria-hidden="true"
                                style="font-size:11px; margin-left:3px; display:inline-block; line-height:1;">↗</span>
                        </a>
                    </div>
                    <?php $affiliate_url = LinkProcessor::generateAffiliateUrl($product->link); ?>
                    <?php if ($affiliate_url !== $product->link) : ?>
                        <div style="margin-top:8px;">
                            <a title="<?php echo esc_attr(__('Affiliate link', 'external-importer')); ?>"
                                target="_blank"
                                rel="noopener"
                                href="<?php echo esc_url($affiliate_url); ?>">
                                <?php echo esc_html(__('Affiliate link', 'external-importer')); ?>
                                <span aria-hidden="true"
                                    style="font-size:11px; margin-left:3px; display:inline-block; line-height:1;">↗</span>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>

                <button type="button"
                    id="ei_update_product"
                    class="button button-primary button-large"
                    style="margin-left:auto;">
                    <?php echo esc_html(__('Sync now', 'external-importer')); ?>
                </button>
            </div>
        </div>

        <script>
            jQuery(document).ready(function($) {
                $('#ei_update_product').click(function(e) {
                    e.preventDefault();
                    var this_btn = $(this);
                    this_btn.attr('disabled', true);

                    var nonce = jQuery("#ei_nonce").val();
                    $.ajax({
                        type: "POST",
                        url: ajaxurl,
                        data: {
                            action: 'ei_update_product',
                            ei_nonce: nonce,
                            product_id: <?php echo (int) $post->ID; ?>
                        },
                        success: function(data) {
                            $("#ei_update_metabox_response").html(data);
                            this_btn.attr('disabled', false);
                            location.reload();
                        },
                        error: function() {
                            location.reload();
                        }
                    });
                    return false;
                });

                // toggle automatic sync on/off via AJAX
                $('#ei_disable_sync').on('change', function() {
                    var checkbox = $(this);
                    var checked = checkbox.is(':checked') ? 1 : 0;
                    var nonce = $('#ei_toggle_sync_nonce').val();

                    checkbox.prop('disabled', true);

                    $.ajax({
                        type: "POST",
                        url: ajaxurl,
                        data: {
                            action: 'ei_toggle_sync',
                            ei_toggle_sync_nonce: nonce,
                            product_id: <?php echo (int) $post->ID; ?>,
                            sync_disabled: checked
                        },
                        success: function(response) {
                            checkbox.prop('disabled', false);

                            if (!response || !response.success) {
                                // revert checkbox state on error
                                checkbox.prop('checked', !checked);

                                var msg = (response && response.data && response.data.message) ?
                                    response.data.message :
                                    '<?php echo \esc_js(__('Unable to save sync setting.', 'external-importer')); ?>';

                                alert(msg);
                            } else {
                                // Dynamically show/hide the "Automatic sync is disabled" notice
                                var container = $('#ei_update_metabox_response');
                                var notice = container.find('#ei-sync-automatic');

                                if (response.data.sync_disabled == 1) {
                                    if (!notice.length) {
                                        container.prepend(
                                            '<div id="ei-sync-automatic" style="margin-bottom:8px; padding:6px 8px; border-left:3px solid #d63638; background:#fef4f4; color:#a00;"><strong><?php echo \esc_js(__('Automatic sync is disabled for this product.', 'external-importer')); ?></strong></div>'
                                        );
                                    }
                                } else {
                                    if (notice.length) {
                                        notice.remove();
                                    }
                                }
                            }
                        },
                        error: function() {
                            checkbox.prop('disabled', false);
                            checkbox.prop('checked', !checked);
                            alert('<?php echo \esc_js(__('Unable to save sync setting.', 'external-importer')); ?>');
                        }
                    });
                });

            });
        </script>
<?php
    }

    public static function metaboxHtml($product_id)
    {
        $status = SyncStatus::get($product_id);

        $res  = '';
        $res .= '<div class="ei-sync-summary" style="font-size:12px; line-height:1.5;">';

        // Automatic sync disabled notice
        if ($status['sync_disabled'])
        {
            $res .= '<div id="ei-sync-automatic" style="
                    margin-bottom:8px;
                    padding:6px 8px;
                    border-left:3px solid #d63638;
                    background:#fef4f4;
                    color:#a00;
                ">
                    <strong>' . __('Automatic sync is disabled for this product.', 'external-importer') . '</strong>
                </div>';
        }

        // If never synced, exit early (keep wrapper so we still show disabled notice).
        if (empty($status['last_synced_ts']))
        {
            $res .= '</div>'; // close .ei-sync-summary
            return $res;
        }

        // Throttling notice
        if ($status['throttled'])
        {
            $res .= '<div style="
                    margin-top:6px;
                    margin-bottom:8px;
                    padding:4px 6px;
                    border-left:3px solid #f0ad4e;
                    background:#fff8e5;
                    color:#8a6d3b;
                 ">
                    ' . __('Currently throttled', 'external-importer') . '
                 </div>';
        }

        // Last update row
        $last_update_text = $status['last_synced_rel'];
        $last_update_full = $status['last_synced_full'];

        $res .= '<div style="margin:3px 0;margin-top:10px;">
            <span class="ei-sync-label" style="color:#555;">' . __('Last synced:', 'external-importer') . '</span>
            <span id="ei-last-updated"
                  style="float:right;"
                  title="' . \esc_attr($last_update_full) . '">'
            . \esc_html($last_update_text) .
            '</span>
            <div style="clear:both;"></div>
         </div>';

        // HTTP status row
        $res .= '<div style="margin:3px 0;">
                <span class="ei-sync-label" style="color:#555;">' . __('Sync Status:', 'external-importer') . '</span>
                <span id="ei-status" style="float:right;">';

        if ($status['status'] === SyncStatus::STATUS_ERROR)
        {
            $res .= '<span style="color:#d63638;">&#9679; ' . __('Error', 'external-importer') . '</span>';
            if (!empty($status['status_code']) && $status['status_code'] > 300)
            {
                $res .= '<span style="display:block; color:#555;">' .
                    __('Code:', 'external-importer') . ' <strong>' . \esc_html($status['status_code']) . '</strong>
                    </span>';
            }
        }
        elseif ($status['status'] === SyncStatus::STATUS_SUCCESS)
        {
            $res .= '<span style="color:#198754;">&#9679; ' . __('Success', 'external-importer') . '</span>';
        }
        else
        {
            // STATUS_NEVER - should not normally happen here because we early-return above,
            // but we handle it gracefully just in case.
            $res .= '<span style="color:#555;">' . __('Never synced', 'external-importer') . '</span>';
        }

        $res .= '   </span>
                <div style="clear:both;"></div>
             </div>';

        // Availability row
        $is_out_of_stock = ($status['availability'] === SyncStatus::AVAIL_OUT_STOCK);

        $res .= '<div style="margin:3px 0;">
                <span class="ei-sync-label" style="color:#555;">' . __('Availability:', 'external-importer') . '</span>
                <span style="float:right;">';

        if ($status['availability'] === SyncStatus::AVAIL_BACKORDER)
        {
            $res .= '<span style="color:#eaa600;">&#9679; ' . \esc_html($status['availability_label']) . '</span>';
        }
        elseif ($status['availability'] === SyncStatus::AVAIL_IN_STOCK)
        {
            $res .= '<span style="color:#7ad03a;">&#9679; ' . \esc_html($status['availability_label']) . '</span>';
        }
        elseif ($status['availability'] === SyncStatus::AVAIL_OUT_STOCK)
        {
            $res .= '<span style="color:#a44;">&#9679; ' . \esc_html($status['availability_label']) . '</span>';
        }
        else
        {
            $res .= '<span style="color:#555;">' . \esc_html($status['availability_label'] ?: __('Unknown', 'external-importer')) . '</span>';
        }

        $res .= '   </span>
                <div style="clear:both;"></div>
             </div>';

        // Last in stock row – only when currently out of stock
        if ($is_out_of_stock && !empty($status['last_in_stock_ts']))
        {
            $last_in_stock_text = $status['last_in_stock_rel'];
            $full_datetime      = $status['last_in_stock_full'];

            $res .= '<div style="margin:3px 0;">
                        <span class="ei-sync-label" style="color:#555;">' . __('Last in stock:', 'external-importer') . '</span>
                        <span id="ei-last-in-stock"
                              style="float:right; color:#555;"
                              title="' . \esc_attr($full_datetime) . '">'
                . \esc_html($last_in_stock_text) .
                '</span>
                        <div style="clear:both;"></div>
                     </div>';
        }

        // Last error message
        if ($status['has_error'] && !empty($status['status_message']))
        {
            $res .= '<div style="margin-top:6px; color:#b32d2e;">
                    <em>' . \esc_html($status['status_message']) . '</em>
                 </div>';
        }

        $res .= '</div>'; // close .ei-sync-summary

        return $res;
    }

    public function ajaxUpdateProduct()
    {
        if (!isset($_POST['ei_nonce']) || !\wp_verify_nonce($_POST['ei_nonce'], 'ei_update_product'))
            \wp_die('Invalid nonce');

        if (!\current_user_can('edit_posts'))
            \wp_die('You don\'t have access to this page.');

        if (!$product_id = (int) InputHelper::post('product_id'))
            \wp_die('Invalid product ID');

        Synchronizer::updateProduct($product_id);

        echo self::metaboxHtml($product_id);
        \wp_die();
    }

    /**
     * AJAX handler to enable/disable automatic sync for a product.
     */
    public function ajaxToggleSync()
    {
        if (!isset($_POST['ei_toggle_sync_nonce']) || !\wp_verify_nonce($_POST['ei_toggle_sync_nonce'], 'ei_toggle_sync'))
        {
            \wp_send_json_error(array('message' => __('Invalid nonce', 'external-importer')));
        }

        if (!\current_user_can('edit_posts'))
        {
            \wp_send_json_error(array('message' => __('You don\'t have access to this page.', 'external-importer')));
        }

        if (!$product_id = (int) InputHelper::post('product_id'))
        {
            \wp_send_json_error(array('message' => __('Invalid product ID', 'external-importer')));
        }

        $sync_disabled = (int) InputHelper::post('sync_disabled') ? 1 : 0;

        if ($sync_disabled)
        {
            \update_post_meta($product_id, self::META_SYNC_DISABLED, 1);
        }
        else
        {
            \delete_post_meta($product_id, self::META_SYNC_DISABLED);
        }

        \wp_send_json_success(array(
            'product_id'    => $product_id,
            'sync_disabled' => $sync_disabled,
        ));
    }
}
