<?php

namespace ExternalImporter\application\admin;

defined('\ABSPATH') || exit;

use ExternalImporter\application\helpers\WooHelper;
use ExternalImporter\application\components\WooImporter;
use ExternalImporter\application\components\Throttler;

/**
 * SyncStatus helper
 *
 * Centralised place to compute sync-related status for a product.
 * Used by:
 *  - SyncMetabox (single product screen)
 *  - Sync list column (All Products table)
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class SyncStatus
{
    const STATUS_NEVER   = 'never';
    const STATUS_SUCCESS = 'success';
    const STATUS_ERROR   = 'error';

    const AVAIL_IN_STOCK   = 'in_stock';
    const AVAIL_BACKORDER  = 'backorder';
    const AVAIL_OUT_STOCK  = 'out_of_stock';
    const AVAIL_UNKNOWN    = 'unknown';

    /**
     * Get normalized sync status info for a product.
     *
     * @param int $product_id
     *
     * @return array {
     *   @type int         $product_id
     *   @type bool        $is_ei_product
     *   @type bool        $sync_disabled
     *   @type int|null    $last_synced_ts       GMT timestamp or null
     *   @type string|null $last_synced_rel      "9 minutes ago"
     *   @type string|null $last_synced_full     localized full datetime
     *   @type string      $status               success|error|never
     *   @type int|null    $status_code
     *   @type string|null $status_message
     *   @type bool        $has_error
     *   @type string      $availability         in_stock|backorder|out_of_stock|unknown
     *   @type string      $availability_label   translated label
     *   @type int|null    $last_in_stock_ts     GMT timestamp or null
     *   @type string|null $last_in_stock_rel    "2 hours ago" or null
     *   @type string|null $last_in_stock_full   localized full datetime or null
     *   @type bool        $throttled
     * }
     */
    public static function get($product_id)
    {
        $product_id = (int) $product_id;

        $result = array(
            'product_id'          => $product_id,
            'is_ei_product'       => false,
            'sync_disabled'       => false,

            'last_synced_ts'      => null,
            'last_synced_rel'     => null,
            'last_synced_full'    => null,

            'status'              => self::STATUS_NEVER,
            'status_code'         => null,
            'status_message'      => null,
            'has_error'           => false,

            'availability'        => self::AVAIL_UNKNOWN,
            'availability_label'  => '',

            'last_in_stock_ts'    => null,
            'last_in_stock_rel'   => null,
            'last_in_stock_full'  => null,

            'throttled'           => false,
        );

        if (! $product_id)
        {
            return $result;
        }

        // Basic sanity: make sure post exists.
        $post_status = get_post_status($product_id);
        if (! $post_status)
        {
            return $result;
        }

        // Is this an External Importer product?
        $result['is_ei_product'] = WooHelper::isEiProduct($product_id);

        // Per-product auto-sync disabled flag.
        $sync_disabled = (int) \get_post_meta($product_id, SyncMetabox::META_SYNC_DISABLED, true);
        $result['sync_disabled'] = ($sync_disabled === 1);

        // Last update timestamp (GMT or unix timestamp, depending on WooImporter implementation).
        $last_update = WooImporter::getLastUpdateMeta($product_id);
        if ($last_update)
        {
            $last_update_ts = (int) $last_update;

            $result['last_synced_ts'] = $last_update_ts;

            // Relative time: "9 minutes ago"
            $diff = \human_time_diff($last_update_ts, time());
            $result['last_synced_rel']  = sprintf(__('%s ago', 'external-importer'), $diff);

            // Full formatted date/time, used for tooltips etc.
            $result['last_synced_full'] = WooHelper::dateFormatFromGmt($last_update_ts);
        }

        // Product info & meta (may be null if never synced).
        $info    = WooImporter::getProductInfoMeta($product_id);
        $product = WooImporter::getProductMeta($product_id);

        // HTTP / sync status
        if (! $last_update)
        {
            $result['status'] = self::STATUS_NEVER;
        }
        elseif ($info && ! empty($info['last_error']))
        {
            $result['status']         = self::STATUS_ERROR;
            $result['has_error']      = true;
            $result['status_message'] = $info['last_error'];

            if (! empty($info['last_error_code']) && $info['last_error_code'] > 0)
            {
                $result['status_code'] = (int) $info['last_error_code'];
            }
        }
        else
        {
            $result['status'] = self::STATUS_SUCCESS;
        }

        // Availability
        if ($product)
        {
            if ($product->availability === 'Backorder')
            {
                $result['availability']       = self::AVAIL_BACKORDER;
                $result['availability_label'] = __('On backorder', 'external-importer');
            }
            elseif ($product->inStock)
            {
                $result['availability']       = self::AVAIL_IN_STOCK;
                $result['availability_label'] = __('In stock', 'external-importer');
            }
            else
            {
                $result['availability']       = self::AVAIL_OUT_STOCK;
                $result['availability_label'] = __('Out of stock', 'external-importer');
            }
        }

        // Last in stock (only meaningful when we have info)
        if ($info && ! empty($info['last_in_stock']))
        {
            $last_in_stock_ts = \strtotime($info['last_in_stock']);
            if ($last_in_stock_ts)
            {
                $result['last_in_stock_ts']   = $last_in_stock_ts;

                $diff_last_in_stock           = \human_time_diff($last_in_stock_ts, time());
                $result['last_in_stock_rel']  = sprintf(__('%s ago', 'external-importer'), $diff_last_in_stock);
                $result['last_in_stock_full'] = WooHelper::dateFormatFromGmt($info['last_in_stock']);
            }
        }

        // Throttling
        if ($product && ! empty($product->domain))
        {
            $result['throttled'] = Throttler::isThrottled($product->domain);
        }

        return $result;
    }
}
