<?php

namespace ExternalImporter\application\libs\pextractor\parser\parsers;

defined('\ABSPATH') || exit;

use ExternalImporter\application\libs\pextractor\parser\Product;
use ExternalImporter\application\libs\pextractor\parser\Variation;
use ExternalImporter\application\libs\pextractor\ExtractorHelper;

use function ExternalImporter\prnx;

/**
 * EtsycomAdvanced class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class EtsycomAdvanced extends AdvancedParser
{

    public function parseLinks()
    {
        $locale = $this->extractLocale();
        $urls   = $this->getRawCatalogUrls();

        return array_map(fn(string $url) => $this->applyLocale($url, $locale), $urls);
    }

    private function getRawCatalogUrls(): array
    {
        // Try JSON source
        $jsonUrls = $this->parseLinksJson();
        if (!empty($jsonUrls))
        {
            return $jsonUrls;
        }

        $paths = [
            "//a[contains(@class,'listing-link')]/@href",
        ];

        return $this->xpathArray($paths);
    }

    private function extractLocale(): ?string
    {
        $url    = $this->getUrl();
        $parsed = parse_url($url);

        // 1) Check 'locale' query parameter
        if (!empty($parsed['query']))
        {
            parse_str($parsed['query'], $params);
            if (!empty($params['locale']))
            {
                return $params['locale'];
            }
        }

        // 2) Fallback: detect two-letter locale in path, e.g. '/it/'
        if (!empty($parsed['path']) && preg_match('~^/([a-z]{2})(?=/|$)~i', $parsed['path'], $matches))
        {
            return strtolower($matches[1]);
        }

        return null;
    }

    private function applyLocale(string $url, ?string $locale): string
    {
        $base = strtok($url, '?');

        if ($locale === null)
        {
            return $base;
        }

        return $base . '?locale=' . urlencode($locale);
    }

    public function parseLinksJson()
    {
        $json = $this->xpathScalar(".//script[@type='application/ld+json']");

        if (!$items = json_decode($json, true))
            return array();

        if (!isset($items['itemListElement']))
            return array();

        $urls = array();
        foreach ($items['itemListElement'] as $item)
        {
            if (isset($item['url']))
                $urls[] = $item['url'];
        }

        return $urls;
    }

    public function parsePagination()
    {
        $xpaths = [
            ".//nav[@data-clg-id='WtPagination']//a[not(contains(@href, 'reviews?'))]/@href",
            ".//nav[@class='search-pagination']//a/@href",
        ];

        $urls = $this->xpathArray($xpaths);

        // Get current page number from the current URL
        $currentUrl = $this->getUrl();
        $currentPage = null;
        if (preg_match('/[?&]page=(\d+)\b/', $currentUrl, $matches))
        {
            $currentPage = (int)$matches[1];
        }

        $urls = array_filter($urls, function ($url) use ($currentPage)
        {
            // Must be a paginated URL with &page= and not page=1
            if (!preg_match('/[?&]page=(\d+)\b/', $url, $matches))
            {
                return false;
            }

            $page = (int)$matches[1];

            // Exclude page 1 and the current page
            return $page !== 1 && $page !== $currentPage;
        });

        return array_values(array_unique($urls));
    }

    public function parseDescription()
    {
        $paths = array(
            ".//p[@class='wt-text-body-01 wt-break-word']",
        );

        $d = $this->xpathScalar($paths, true);
        $d = html_entity_decode($d);

        return $d;
    }

    public function parsePrice()
    {
        $paths = array(
            ".//div[@data-buy-box-region='price']//p[@class='wt-text-title-03 wt-mr-xs-1']/span[2]",
            ".//div[@data-buy-box-region='price']//p[@class='wt-text-title-03 wt-mr-xs-1']",
            ".//div[@class='wt-mb-xs-3']//p[@class='wt-text-title-03 wt-mr-xs-2']",
            ".//*[@class='text-largest strong override-listing-price']",
            ".//p[@class='wt-text-title-03 wt-mr-xs-1']/span",
            ".//div[@data-selector='price-only']//p[@class='wt-text-title-larger wt-mr-xs-1 wt-text-slime ']",
        );

        return $this->xpathScalar($paths);
    }

    public function parseOldPrice()
    {
        $paths = array(
            ".//p[contains(@class, 'wt-text-gray')]//span[@class='wt-text-strikethrough']",
            ".//div[@class='wt-mb-xs-3']//p[contains(@class, 'wt-text-strikethrough')]",
            ".//meta[@property='product:price:amount']/@content",

        );

        return $this->xpathScalar($paths);
    }

    public function parseImage()
    {
        if ($images = $this->parseImages())
            return reset($images);
    }

    public function parseImages()
    {
        $paths = array(
            ".//div[contains(@class, 'image-carousel-container')]//*/@data-src",
            ".//div[contains(@class, 'image-carousel-container')]//*/@data-src-zoom-image",
            ".//div[contains(@data-selector, 'main-carousel')]//*/@data-zoom-src",
            ".//div[contains(@class, 'image-carousel-container')]//img/@src",
        );

        return $this->xpathArray($paths);
    }

    public function getReviewsXpath()
    {
        return array(
            array(
                'review' => ".//div[@id='same-listing-reviews-panel' or @id='reviews']//*[contains(@id, 'review-preview-toggle')]",
                'rating' => ".//div[@id='same-listing-reviews-panel' or @id='reviews']//span[@class='wt-display-inline-block wt-mr-xs-1']//span[last()]/@data-rating",
                'author' => ".//div[@id='same-listing-reviews-panel' or @id='reviews']//a[@class='wt-text-link wt-mr-xs-1']",
                'date' => ".//div[@id='same-listing-reviews-panel'or @id='reviews']//p[@class='wt-text-caption wt-text-gray']/text()",
            ),
        );
    }

    public function parseCurrencyCode()
    {
        if ($this->parsePrice())
        {
            if (preg_match('/"locale_currency_code":"([A-Z]+?)"/', $this->html, $matches))
                return $matches[1];
        }
    }

    public function getFeaturesXpath()
    {
        return array(
            array(
                'name-value' => ".//p[@id='legacy-materials-product-details']",
                'separator' => ":",
            ),
            array(
                'name-value' => ".//div[@id='product_details_content_toggle']//div[@class='wt-ml-xs-1']",
                'separator' => ":",
            ),
        );
    }

    public function afterParseFix(Product $product)
    {
        if (strstr($product->title, 'item is unavailable'))
        {
            $product->availability = 'OutOfStock';
            $product->price = 0;
            $product->oldPrice = 0;
        }

        foreach ($product->reviews as $i => $r)
        {
            if ($r['rating'])
                $product->reviews[$i]['rating']++;
        }

        if ($product->variations)
        {
            $product->features = [];
        }

        return $product;
    }

    public function parseVariations(): array
    {
        $variations = [];
        $groups = [];

        for ($i = 0;; $i++)
        {
            $labelXpath = ".//label[@id='label-variation-selector-{$i}']";
            $rawName    = $this->xpathScalar($labelXpath);
            if (! $rawName)
            {
                break;
            }
            $groupName = trim(sanitize_text_field($rawName));
            if ($groupName === '')
            {
                continue;
            }
            $optionsXpath = ".//select[@id='variation-selector-{$i}']//option[string-length(normalize-space(@value)) > 0]";
            $rawOptions   = $this->xpathArray($optionsXpath);
            if (empty($rawOptions))
            {
                continue;
            }
            $opts = [];
            foreach ($rawOptions as $rawOption)
            {
                $rawOption = trim(sanitize_text_field($rawOption));
                if ($rawOption === '')
                {
                    continue;
                }

                $parsed = $this->parseVariationAndPrice($rawOption);

                $value = $parsed['variation'];
                $price = $parsed['price'];
                $availability = $parsed['availability'];

                $opts[] = [
                    'value' => $value,
                    'price' => $price,
                    'availability' => $availability,
                ];
            }
            if (! empty($opts))
            {
                $groups[] = [
                    'name'    => $groupName,
                    'options' => $opts,
                    'availability' => $availability,
                ];
            }
        }

        if (empty($groups))
        {
            return [];
        }

        $combinations = [[]];
        foreach ($groups as $group)
        {
            $newCombinations = [];
            foreach ($combinations as $combo)
            {
                foreach ($group['options'] as $opt)
                {
                    $newCombo = $combo;
                    $newCombo[] = [
                        'name'  => $group['name'],
                        'value' => $opt['value'],
                        'price' => $opt['price'],
                        'availability' => $opt['availability'],
                    ];
                    $newCombinations[] = $newCombo;
                }
            }
            $combinations = $newCombinations;
        }

        foreach ($combinations as $combo)
        {
            $variation = new Variation();
            $totalPrice = 0;
            foreach ($combo as $item)
            {
                $variation->attributes[] = [
                    'name'  => $item['name'],
                    'value' => $item['value']
                ];
                $totalPrice += $item['price'];
            }
            $variation->price = $totalPrice;
            $variation->oldPrice = 0;
            $variation->availability = $item['availability'];
            $variations[] = $variation;
        }

        return $variations;
    }

    protected function parseVariationAndPrice($input)
    {
        $result = [];
        $pattern = '/^\s*(.*?)\s*\((.*?)\)\s*$/';

        if (preg_match($pattern, $input, $matches))
        {
            $result = [
                'variation' => $matches[1],
                'price'     => (float) ExtractorHelper::parsePriceAmount($matches[2]),
            ];
        }
        else
        {
            $result = [
                'variation' => $input,
                'price'     => $this->parsePrice(),
            ];
        }

        if (preg_match('/\[.+?\]/', $input))
        {
            $result['variation'] = trim(preg_replace('/\[.+?\]/', '', $result['variation']));
            $result['availability'] = 'OutOfStock';
        }
        else
        {
            $result['availability'] = 'InStock';
        }

        return $result;
    }
}
