<?php

namespace ExternalImporter\application\libs\pextractor\parser\parsers;

defined('\ABSPATH') || exit;

use ExternalImporter\application\libs\pextractor\parser\Product;
use ExternalImporter\application\libs\pextractor\ExtractorHelper;

use function ExternalImporter\prnx;

/**
 * TikivnAdvanced class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class TikivnAdvanced extends AdvancedParser
{

    private $_oldPrice;

    public function parseLinks()
    {

        if ($urls = $this->_getFilteredLinks())
            return $urls;

        if (preg_match_all('~"Product","url":"(.+?)",~', $this->html, $matches))
            $urls = $matches[1];

        if (!$urls)
        {
            $path = array(
                ".//div[contains(@class, 'product-item')]//a/@href",
                ".//*[@class='search-a-product-item']",
                ".//p[@class='title']/a/@href",
                ".//a[@class='product-item']/@href",
            );

            $urls = $this->xpathArray($path);
        }
        foreach ($urls as $i => $url)
        {
            $urls[$i] = strtok($url, '?');
        }

        return $urls;
    }

    public function _getFilteredLinks()
    {
        $query = parse_url($this->getUrl(), PHP_URL_QUERY);
        parse_str($query, $arr);

        if (!$path = parse_url($this->getUrl(), PHP_URL_PATH))
            return array();

        if (!preg_match('~/c(\d+)~', $path, $matches))
            return array();

        $category_id = $matches[1];

        $url = 'https://tiki.vn/api/personalish/v1/blocks/listings?limit=300&include=advertisement&aggregations=1&category=' . urlencode($category_id);
        $url = \add_query_arg($arr, $url);

        $response = $this->getRemoteJson($url);

        if (!$response || !isset($response['data']))
            return array();

        $urls = array();
        foreach ($response['data'] as $d)
        {
            $urls[] = $d['url_key'] . '.html';
        }

        return $urls;
    }

    public function parsePagination()
    {
        $path = array(
            ".//div[@class='list-pager']//li/a/@href",
        );

        $urls = $this->xpathArray($path);
        if (!$urls)
            $urls = array(\add_query_arg('page', 2, $this->getUrl()));

        return $urls;
    }

    public function parseDescription()
    {
        $path = array(
            ".//div[@class='content']//div[contains(@class, 'ToggleContent__Wrapper-sc')]",
            ".//div[@class='summary']//div[@class='group border-top']",
        );
        return $this->xpathScalar($path, true);
    }

    public function parsePrice()
    {
        $prices = $this->_getTikiVariationPrices();
        if ($prices)
        {
            $price = $prices['sale_price'];
            $this->_oldPrice = $prices['old_price'];
            return $price;
        }

        if (preg_match('/,"price":(\d+),"/', $this->html, $matches))
            return $matches[1];
    }

    public function parseOldPrice()
    {
        if ($this->_oldPrice)
        {
            return $this->_oldPrice;
        }

        if (preg_match('/,"list_price":(\d+),"/', $this->html, $matches))
            return $matches[1];

        $paths = array(
            ".//div[@class='summary']//p[@class='original-price']",
        );

        if ($price = $this->xpathScalar($paths))
            return $price;
    }

    public function parseImage()
    {
        if ($images = $this->parseImages())
            return reset($images);
    }

    public function parseImages()
    {
        $images = array();
        $paths = array(
            ".//div[@class='review-images']//img/@src",
            ".//div[@class='group-images']//div/@src",
            ".//div[@class='review-images']//picture/@srcset",
            ".//span[@class='slider']//img/@src",
        );
        $results = $this->xpathArray($paths);
        foreach ($results as $img)
        {
            $img = str_replace('/w80/', '/w390/', $img);
            $img = str_replace('/w64/', '/w390/', $img);
            $img = str_replace('/100x100/', '/w390/', $img);
            $img = str_replace('/200x280/', '/750x750/', $img);
            $images[] = $img;
        }
        return $images;
    }

    public function getFeaturesXpath()
    {
        return array(
            array(
                'name' => ".//div[contains(@class, 'ProductDescription__Wrapper')]//table//tr/td[1]",
                'value' => ".//div[contains(@class, 'ProductDescription__Wrapper')]//table//tr/td[2]",
            ),
            array(
                'name' => ".//div[contains(@class, 'style__Wrapper')]//table//tr/td[1]",
                'value' => ".//div[contains(@class, 'style__Wrapper')]//table//tr/td[2]",
            ),
        );
    }

    public function parseReviews()
    {
        if (!preg_match('~-p(\d+)\.html~', $this->getUrl(), $matches))
            return array();

        $url = 'https://tiki.vn/api/v2/reviews?product_id=' . urldecode($matches[1]) . '&limit=20&sort=score|desc,id|desc,stars|all&include=comments&page=1';
        $response = $this->getRemoteJson($url);

        if (!$response || !isset($response['data']))
            return array();

        $results = array();
        foreach ($response['data'] as $r)
        {
            $review = array();

            if (empty($r['content']))
                continue;

            $review['review'] .= $r['content'];

            if (isset($r['rating']))
                $review['rating'] = ExtractorHelper::ratingPrepare($r['rating']);

            if (isset($r['created_by']['name']))
                $review['author'] = $r['created_by']['name'];

            if (isset($r['created_at']))
                $review['date'] = $r['created_at'];

            $results[] = $review;
        }
        return $results;
    }

    public function parseCurrencyCode()
    {
        return 'VND';
    }

    public function parseInStock()
    {
        if ($this->xpathScalar(".//p[@class='product-status discontinued']"))
            return false;
    }

    public function afterParseFix(Product $product)
    {
        if ($product->categoryPath)
            array_shift($product->categoryPath);

        return $product;
    }

    protected function _getTikiVariationPrices()
    {
        $productUrl = $this->getUrl();

        $jsonData = $this->xpathScalar(".//script[@id='__NEXT_DATA__']");

        if (!$jsonData)
        {
            return false;
        }

        // 1) Parse the URL to get the spid
        $query = parse_url($productUrl, PHP_URL_QUERY);
        parse_str($query, $params);
        $spid = $params['spid'] ?? null;

        // 2) Decode the JSON
        $data = json_decode($jsonData, true);
        if (json_last_error() !== JSON_ERROR_NONE)
        {
            return false;
        }

        // 3) Navigate into initialState → productv2 → productData → response → data
        if (empty($data['props']['initialState']['productv2']['productData']['response']['data']))
        {
            return false;
        }

        $prod = $data['props']['initialState']['productv2']['productData']['response']['data'];

        // 4) If there are variations, find the one matching the URL’s spid
        if (!empty($prod['configurable_products']) && $spid !== null)
        {
            foreach ($prod['configurable_products'] as $variation)
            {
                if ((string)($variation['id'] ?? '') === $spid)
                {
                    return [
                        'sale_price' => (int)($variation['price'] ?? $prod['price']),
                        'old_price'  => (int)($variation['original_price'] ?? $prod['original_price']),
                    ];
                }
            }
        }

        // 5) Fallback to the master product’s prices
        return [
            'sale_price' => (int)($prod['price']          ?? 0),
            'old_price'  => (int)($prod['original_price'] ?? $prod['list_price'] ?? 0),
        ];
    }
}
