var externalImporter = angular.module("externalImporter", [
  "ngSanitize",
  "ngAnimate",
]);
externalImporter.controller(
  "ExternalImporterController",
  function ($scope, extractorService, $timeout) {
    const MAX_PRODUCT_ARRAY_LENGTH = 1000;
    const MAX_LOG_ARRAY_LENGTH = 200;

    $scope.settings = {};
    $scope.products = [];
    $scope.log = [];
    $scope.stat = {};
    $scope.stat.success = $scope.stat.errors = $scope.stat.new = 0;
    $scope.importStat = {};
    $scope.importStat.success = $scope.importStat.errors = 0;
    $scope.listingProcessor = {};
    $scope.listingProcessor.params = {};
    $scope.listingProcessor.params.salt = rnd();
    $scope.listingProcessor.params.url = "";
    $scope.listingProcessor.params.max_count = 200;
    $scope.listingProcessor.params.automatic_pagination = true;
    $scope.productProcessor = {};
    $scope.productProcessor.params = {};
    $scope.productProcessor.params.salt = rnd();
    $scope.productProcessor.params.urls = "";
    $scope.loading = false;
    $scope.settings.timeout = 1;
    $scope.in_progress = false;
    $scope.in_waiting = false;
    $scope.importQueue = [];
    $scope.importParams = {};
    $scope.importParams.applay_ai = true;
    $scope.import_in_progress_count = 0;
    $scope.importSettings = {};
    $scope.importSettings.threads = 3;
    $scope.debug = "";
    $scope.usedParsers = [];
    $scope.usedProvider = "";
    $scope.currentProduct = null;

    $scope.$watchCollection("importQueue", function (newValue, oldValue) {
      if (!newValue.length) return;
      if (oldValue.length > newValue.length) return;
      $scope.maybeImport();
    });

    $scope.maybeImport = function () {
      if ($scope.import_in_progress_count >= $scope.importSettings.threads)
        return;
      var threads =
        $scope.importSettings.threads - $scope.import_in_progress_count;
      if (threads > $scope.importQueue.length)
        threads = $scope.importQueue.length;

      for (var i = 0; i < threads; i++) {
        var product_id = $scope.importQueue[0];
        $scope.importQueue.splice(0, 1);
        $scope.productImport(product_id);
      }
    };

    $scope.maybeCleanup = function () {
      if ($scope.import_in_progress_count.length || $scope.importQueue.length)
        return;
      for (const product of $scope.products) {
        if (product) return;
      }
      $scope.products = [];
      $scope.checkAll = false;
    };

    $scope.startImport = function (processor) {
      $scope.loading = true;

      var params = { [processor]: $scope[processor].params };
      loadRemoteData(params);
    };

    $scope.stopImport = function (processor) {
      $scope.loading = false;
    };

    $scope.restartImport = function (processor) {
      $scope[processor].params.salt = rnd();
      $scope.deleteAllProducts();
      $scope.products = [];
      $scope.debug = "";
      $scope.usedParsers = [];
      $scope.usedProvider = "";
      $scope.startImport(processor);
    };

    $scope.addToImportQueue = function (product_id) {
      if ($scope.importQueue.indexOf(product_id) > -1) return;
      $scope.products[product_id]._import_in_queue = true;
      $scope.importQueue.push(product_id);
      $scope.products[product_id]._selected = false;
    };

    $scope.deleteProduct = function (product_id) {
      var product = $scope.products[product_id];
      if (
        product &&
        !product._import_in_queue &&
        !product._import_status &&
        !product._import_in_progress
      )
        delete $scope.products[product_id];
      $scope.maybeCleanup();
    };

    $scope.deleteAllProducts = function () {
      $scope.maybeCleanup();
      if (!$scope.products.length) return;

      $scope.products.forEach(function (product, product_id) {
        $scope.deleteProduct(product_id);
      });
    };

    $scope.productImport = function (product_id) {
      $scope.import_in_progress_count++;
      $scope.products[product_id]._import_in_queue = false;
      $scope.products[product_id]._import_in_progress = true;

      var product = $scope.products[product_id];
      product._index = product_id;
      var params = { product: product, params: $scope.importParams };
      loadImportResponse(params);
    };

    $scope.selectedCount = function () {
      var count = 0;
      angular.forEach($scope.products, function (product, key) {
        if (product._selected) count++;
      });
      return count;
    };

    $scope.toggleSeleted = function () {
      angular.forEach($scope.products, function (product, key) {
        if (
          product &&
          !product._import_in_queue &&
          !product._import_status &&
          !product._import_in_progress
        ) {
          product._selected = $scope.checkAll;
        }
      });
    };

    $scope.importSelected = function () {
      angular.forEach($scope.products, function (product, key) {
        if (product && product._selected) {
          $scope.addToImportQueue(key);
          product._selected = false;
        }
      });
      $scope.checkAll = false;
    };

    $scope.importAll = function () {
      angular.forEach($scope.products, function (product, key) {
        if (
          product &&
          !product._import_status &&
          !product._import_in_queue &&
          !product._import_in_progress
        ) {
          $scope.addToImportQueue(key);
        }
      });
    };

    $scope.initAutomaticImport = function () {
      if ($scope.automaticImport) $scope.importAll();
    };

    $scope.rnd = function () {
      return rnd();
    };

    $scope.hasUrls = function (val) {
      var s = (val || "").trim();
      if (!s) return false;
      return /(https?:\/\/|www\.)/i.test(s);
    };

    $scope.hasPrice = function (v) {
      if (v === null || v === undefined) return false;
      if (typeof v === "string" && v.trim() === "") return false;

      var n = Number(v);
      if (!isFinite(n)) return false;

      return n > 0;
    };

    $scope.currentProduct = null;

    $scope.openJson = function (product, index, $event) {
      if ($event) {
        $event.preventDefault();
        $event.stopPropagation();
      }
      $scope.currentProduct = product;

      $scope.$evalAsync(function () {
        var el = document.getElementById("eiJsonModal");
        if (!el) return;

        if (window.bootstrap && window.bootstrap.Modal) {
          var modal = window.bootstrap.Modal.getOrCreateInstance(el, {
            backdrop: true,
            keyboard: true,
          });
          modal.show();
          return;
        }

        if (!document.querySelector(".modal-backdrop")) {
          var backdrop = document.createElement("div");
          backdrop.className = "modal-backdrop fade show";
          document.body.appendChild(backdrop);
        }

        el.classList.add("show");
        el.style.display = "block";
        el.removeAttribute("aria-hidden");
        document.body.classList.add("modal-open");

        var backdropEl = document.querySelector(".modal-backdrop");
        var onBackdropClick = function () {
          close();
        };
        var onShellClick = function (e) {
          if (e.target === el) close();
        };
        var onKeydown = function (e) {
          if (e.key === "Escape") close();
        };

        function close() {
          if (backdropEl)
            backdropEl.removeEventListener("click", onBackdropClick);
          el.removeEventListener("click", onShellClick);
          document.removeEventListener("keydown", onKeydown);

          el.classList.remove("show");
          el.style.display = "";
          el.setAttribute("aria-hidden", "true");
          document.body.classList.remove("modal-open");
          var bd = document.querySelector(".modal-backdrop");
          if (bd) bd.remove();
        }

        if (backdropEl)
          backdropEl.addEventListener("click", onBackdropClick, {
            once: false,
          });
        el.addEventListener("click", onShellClick);
        document.addEventListener("keydown", onKeydown);

        el.querySelectorAll('[data-bs-dismiss="modal"], .btn-close').forEach(
          function (btn) {
            btn.addEventListener("click", close, { once: true });
          }
        );
      });
    };

    function rnd() {
      return Date.now();
    }

    function getTimeout() {
      if ($scope.settings.timeout >= 0) return $scope.settings.timeout * 1000;

      var min = 1;
      var min = 10;
      if ($scope.settings.timeout == -1) max = 5;
      if ($scope.settings.timeout == -2) max = 10;

      return (Math.floor(Math.random() * (max - min + 1)) + min) * 1000;
    }

    function isDublicateProduct(newProduct) {
      for (const product of $scope.products) {
        if (product && newProduct.link == product.link) return true;
      }
      return false;
    }

    function applyRemoteData(newData) {
      if (newData.products) {
        for (const newProduct of newData.products) {
          if (!isDublicateProduct(newProduct)) $scope.products.push(newProduct);
        }

        var defined_size = $scope.products.filter(function (value) {
          return value !== undefined;
        }).length;
        var undefined_size = $scope.products.length - defined_size;
        $scope.products.splice(MAX_PRODUCT_ARRAY_LENGTH + undefined_size);
      }

      if (newData.log) {
        newData.log.reverse();
        $scope.log = newData.log.concat($scope.log);
        $scope.log.splice(MAX_LOG_ARRAY_LENGTH);
      }

      if (newData.stat) $scope.stat = newData.stat;

      if (newData.debug) $scope.debug = newData.debug;
      if (newData.parsers) $scope.usedParsers = newData.parsers;
      if (newData.provider) $scope.usedProvider = newData.provider;

      if ($scope.automaticImport) $scope.importAll();
    }

    function loadRemoteData(params) {
      $scope.in_progress = true;
      $scope.in_waiting = false;
      $scope.errorMessage = null;

      extractorService
        .getProducts(params)
        .then(function (response) {
          var data =
            response && response.data !== undefined ? response.data : response;

          applyRemoteData(data);

          if ($scope.loading && data.cmd === "next") {
            // we are going to request again after a timeout
            $scope.in_progress = false;
            $scope.in_waiting = true;

            $timeout(function () {
              loadRemoteData(params);
            }, getTimeout());
          } else {
            // finished
            $scope.loading = false;
            $scope.in_progress = false;
            $scope.in_waiting = false;
          }
        })
        .catch(function (error) {
          $scope.loading = false;
          $scope.in_progress = false;
          $scope.in_waiting = false;

          // always have an array
          $scope.log = $scope.log || [];

          var error_mess = [];

          // 1) if service provided a log array (e.g. from backend or normalizeHttpError)
          if (error && Array.isArray(error.log) && error.log.length) {
            error_mess = error.log;
          } else if (error && error.data && Array.isArray(error.data.log)) {
            // or old shape: error.data.log
            error_mess = error.data.log;
          } else {
            // 2) fallback: build a single log entry
            var msg;

            if (error && error.friendlyMessage) {
              msg = error.friendlyMessage;
            } else if (error && error.status === -1) {
              msg = "The request timed out.";
            } else if (error && error.status) {
              msg =
                "Request failed (" +
                error.status +
                (error.statusText ? " " + error.statusText : "") +
                ").";
            } else {
              msg = "An unexpected error occurred while loading data.";
            }

            error_mess.push({
              message: msg,
              type: "error",
            });
          }

          // prepend new messages to existing log
          $scope.log = error_mess.concat($scope.log);
        });
    }

    function productImportHandler(data) {
      $scope.import_in_progress_count--;

      var product_id = data.index;
      var product = $scope.products[product_id];

      if (!product) {
        return;
      }

      product._import_in_progress = false;
      product._import_status = data.status || "error";

      var t = 2000;

      if (data.status === "success") {
        $scope.importStat.success++;
      } else {
        product._import_message =
          "Error: " + (data.message || "Import failed.");
        $scope.importStat.errors++;
        t = 10000;
      }

      $scope.maybeImport();

      $timeout(function () {
        delete $scope.products[product_id];
        $scope.maybeCleanup();
      }, t);
    }

    function loadImportResponse(params) {
      extractorService
        .importProduct(params)
        .then(function (data) {
          // success payload from backend
          productImportHandler(data);
        })
        .catch(function (err) {
          var product_id = params.product._index;

          var message;

          if (err && err.friendlyMessage) {
            // from normalizeHttpError (timeout, 403, 500, etc.)
            message = err.friendlyMessage;
          } else if (err && err.status === -1) {
            message = "The import request timed out.";
          } else if (err && err.status) {
            message =
              "Import failed (" +
              err.status +
              (err.statusText ? " " + err.statusText : "") +
              ").";
          } else {
            message = "Import failed due to an unexpected error.";
          }

          productImportHandler({
            index: product_id,
            status: "error",
            message: message,
            _rawError: err,
          });
        });
    }
  }
);

externalImporter.directive("onEnter", function () {
  var linkFn = function (scope, element, attrs) {
    element.on("keypress", function (event) {
      if (event.which === 13) {
        scope.$apply(function () {
          scope.$eval(attrs.onEnter);
        });
        event.preventDefault();
      }
    });
  };
  return {
    link: linkFn,
  };
});

externalImporter.directive("imageloaded", [
  function () {
    "use strict";
    return {
      restrict: "A",
      link: function (scope, element, attrs) {
        var cssClass = attrs.loadedclass;

        element.on("load", function (e) {
          angular.element(element).addClass(cssClass);
        });
      },
    };
  },
]);

externalImporter.directive("repeatDone", [
  function () {
    return {
      restrict: "A",
      link: function (scope, element, iAttrs) {
        var parentScope = element.parent().scope();
        if (scope.$last) {
          parentScope.$last = true;
        }
      },
    };
  },
]);

externalImporter.directive("ngConfirmClick", function () {
  return {
    priority: -1,
    restrict: "A",
    link: function (scope, element, attrs) {
      element.on("click", function (e) {
        var message = attrs.ngConfirmClick;
        if (message && !confirm(message)) {
          e.stopImmediatePropagation();
          e.preventDefault();
        }
      });
    },
  };
});

externalImporter.directive("selectOnClick", function () {
  return {
    restrict: "A",
    link: function (scope, element, attrs) {
      element.on("click", function () {
        this.select();
      });
    },
  };
});

externalImporter.directive("convertToNumber", function () {
  return {
    require: "ngModel",
    link: function (scope, element, attrs, ngModel) {
      ngModel.$parsers.push(function (val) {
        return val != null ? parseInt(val, 10) : null;
      });
      ngModel.$formatters.push(function (val) {
        return val != null ? "" + val : null;
      });
    },
  };
});

externalImporter.filter("intlCurrency", function () {
  return function (value, code, locale, minFrac, maxFrac) {
    if (value == null || code == null) return value;
    try {
      return new Intl.NumberFormat(locale || undefined, {
        style: "currency",
        currency: code,
        currencyDisplay: "symbol",
        minimumFractionDigits: minFrac == null ? 2 : minFrac,
        maximumFractionDigits: maxFrac == null ? 2 : maxFrac,
      }).format(+value);
    } catch (e) {
      var n = isNaN(+value) ? value : (+value).toFixed(2);
      return code + " " + n;
    }
  };
});

externalImporter.directive("fillToBottom", function ($window, $timeout) {
  return {
    restrict: "A",
    link: function (scope, el, attrs) {
      var node = el[0];
      var minH = parseInt(attrs.fillToBottomMin, 10) || 900; // default 900px

      function resize() {
        var vh =
          (window.visualViewport && window.visualViewport.height) ||
          window.innerHeight;
        var top = node.getBoundingClientRect().top;
        var remaining = vh - top;

        var h = Math.max(remaining, minH);
        node.style.height = h + "px";
        node.style.overflowY = "auto";
        node.style.overflowX = "hidden";
      }

      $timeout(resize, 0);
      angular.element($window).on("resize", resize);
      if (window.visualViewport) {
        window.visualViewport.addEventListener("resize", resize);
        window.visualViewport.addEventListener("scroll", resize);
      }

      var ro = "ResizeObserver" in window ? new ResizeObserver(resize) : null;
      if (ro) ro.observe(document.body);

      scope.$on("$destroy", function () {
        angular.element($window).off("resize", resize);
        if (window.visualViewport) {
          window.visualViewport.removeEventListener("resize", resize);
          window.visualViewport.removeEventListener("scroll", resize);
        }
        if (ro) ro.disconnect();
      });
    },
  };
});
