<?php
/**
 * Compatibility - WooPayments plugin.
 *
 * Tested upto: 9.7.0
 *
 * @since 12.5.0
 * @link https://woocommerce.com/products/woopayments/
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'FGF_WooCommerce_Payments_Compatibility' ) ) {

	/**
	 * Class.
	 *
	 * @since 12.5.0
	 */
	class FGF_WooCommerce_Payments_Compatibility extends FGF_Compatibility {

		/**
		 * Class Constructor.
		 *
		 * @since 12.5.0
		 */
		public function __construct() {
			$this->id = 'woocommerce_payments';

			parent::__construct();
		}

		/**
		 * Is plugin enabled?.
		 *
		 * @since 12.5.0
		 * @return bool
		 * */
		public function is_plugin_enabled() {
			return function_exists( 'WC_Payments_Multi_Currency' ) && WC_Payments_Features::is_customer_multi_currency_enabled();
		}

		/**
		 * Actions.
		 * 
		 * @since 12.5.0
		 */
		public function actions() {
			// Convert the price based on default currency. 
			add_filter('fgf_order_item_gift_product_original_price', array( $this, 'convert_price_to_default_currency' ), 10, 1);
			// Convert the price based on default currency. 
			add_filter('fgf_rule_total_price', array( $this, 'convert_price_to_default_currency' ), 11, 1);
		}

		/**
		 * Frontend Action.
		 * 
		 * @since 12.5.0
		 */
		public function frontend_action() {
			// Convert the price based on current currency. 
			add_filter('fgf_convert_price_to_current_currency', array( $this, 'convert_price_to_current_currency' ), 10, 1);
		}

		/**
		 * Convert the price based on current currency.
		 *
		 * @since 12.5.0
		 * @param int|float $price Price.
		 * @return int|float
		 */
		public static function convert_price_to_current_currency( $price ) {
			return self::convert_price($price);
		}

		/**
		 * Convert the price based on default currency.
		 *
		 * @since 12.5.0
		 * @param int|float $price Price.
		 * @return int|float
		 */
		public static function convert_price_to_default_currency( $price ) {
			return self::convert_price($price, true);
		}

		/**
		 * Convert the price based on current currency.
		 *
		 * @since 12.5.0
		 * @param int|float $price Price.
		 * @param bool      $convert_to_default_currency Whether to convert the price to the default currency or not.
		 * @return int|float
		 */
		public static function convert_price( $price, $convert_to_default_currency = false ) {
			if ( ! $price ) {
				return $price;
			}

			if ( $convert_to_default_currency ) {
				$from_currency = WC_Payments_Multi_Currency()->get_selected_currency()->get_code();
				$to_currency   = WC_Payments_Multi_Currency()->get_default_currency()->get_code();

				return WC_Payments_Multi_Currency()->get_raw_conversion( $price, $to_currency, $from_currency );
			}

			return WC_Payments_Multi_Currency()->get_price( $price, 'exchange_rate' );
		}
	}

}
