<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Shortcodes
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

add_shortcode( 'post_date', 'genesis_post_date_shortcode' );
/**
 * Produces the date of post publication.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is empty string),
 *   format (date format, default is value in date_format option field),
 *   label (text following 'before' output, but before date).
 *
 * Output passes through `genesis_post_date_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Output for `post_date` shortcode.
 */
function genesis_post_date_shortcode( $atts ) {

	$defaults = [
		'after'          => '',
		'before'         => '',
		'format'         => get_option( 'date_format' ),
		'label'          => '',
		'relative_depth' => 2,
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_date' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'label' ] );

	if ( 'relative' === $atts['format'] ) {
		$display  = genesis_human_time_diff( get_the_time( 'U' ), current_time( 'timestamp' ), $atts['relative_depth'] ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested -- safe to compare WP to WP timestamps, see https://make.wordpress.org/core/2019/09/23/date-time-improvements-wp-5-3/#comment-37319.
		$display .= ' ' . __( 'ago', 'genesis' );
	} else {
		$display = get_the_time( $atts['format'] );
	}

	$output = sprintf( '<time %s>', genesis_attr( 'entry-time' ) ) . $atts['before'] . $atts['label'] . $display . $atts['after'] . '</time>';

	return apply_filters( 'genesis_post_date_shortcode', $output, $atts );

}

add_shortcode( 'post_time', 'genesis_post_time_shortcode' );
/**
 * Produces the time of post publication.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is empty string),
 *   format (date format, default is value in date_format option field),
 *   label (text following 'before' output, but before date).
 *
 * Output passes through `genesis_post_time_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Output for `post_time` shortcode`.
 */
function genesis_post_time_shortcode( $atts ) {

	$defaults = [
		'after'  => '',
		'before' => '',
		'format' => get_option( 'time_format' ),
		'label'  => '',
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_time' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'label' ] );

	$output = sprintf( '<time %s>', genesis_attr( 'entry-time' ) ) . $atts['before'] . $atts['label'] . get_the_time( $atts['format'] ) . $atts['after'] . '</time>';

	return apply_filters( 'genesis_post_time_shortcode', $output, $atts );

}

add_shortcode( 'post_modified_date', 'genesis_post_modified_date_shortcode' );
/**
 * Produce the post last modified date.
 *
 * Supported shortcode attributes are:
 *  * after (output after date, default is empty string),
 *  * before (output before date, default is empty string),
 *  * format (date format, default is value in date_format option field),
 *  * label (text following 'before' output, but before date).
 *
 * Output passes through `genesis_post_modified_date_shortcode` filter before returning.
 *
 * @since 2.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Output for `post_modified_date` shortcode.
 */
function genesis_post_modified_date_shortcode( $atts ) {

	$defaults = [
		'after'          => '',
		'before'         => '',
		'format'         => get_option( 'date_format' ),
		'label'          => '',
		'relative_depth' => 2,
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_modified_date' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'label' ] );

	if ( 'relative' === $atts['format'] ) {
		$display  = genesis_human_time_diff( get_the_modified_time( 'U' ), current_time( 'timestamp' ), $atts['relative_depth'] ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested -- safe to compare WP to WP timestamps, see https://make.wordpress.org/core/2019/09/23/date-time-improvements-wp-5-3/#comment-37319.
		$display .= ' ' . __( 'ago', 'genesis' );
	} else {
		$display = get_the_modified_time( $atts['format'] );
	}

	$output = sprintf( '<time %s>', genesis_attr( 'entry-modified-time' ) ) . $atts['before'] . $atts['label'] . $display . $atts['after'] . '</time>';

	/**
	 * Change the output of the post_modified_date shortcode.
	 *
	 * @since 2.1.0
	 *
	 * @param string $output Markup containing post last modification date.
	 * @param array  $atts {
	 *     Shortcode attributes after merging with default values.
	 *
	 *     @type string $after Output after date.
	 *     @type string $before Output before date.
	 *     @type string $format Date format, could be 'relative'.
	 *     @type string $label Text following 'before' output, but before date.
	 * }
	 */
	return apply_filters( 'genesis_post_modified_date_shortcode', $output, $atts );

}

add_shortcode( 'post_modified_time', 'genesis_post_modified_time_shortcode' );
/**
 * Produce the post last modified time.
 *
 * Supported shortcode attributes are:
 *  * after (output after time, default is empty string),
 *  * before (output before time, default is empty string),
 *  * format (date format, default is value in date_format option field),
 *  * label (text following 'before' output, but before time).
 *
 * Output passes through `genesis_post_modified_time_shortcode` filter before returning.
 *
 * @since 2.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Output for `post_modified_time` shortcode.
 */
function genesis_post_modified_time_shortcode( $atts ) {

	$defaults = [
		'after'  => '',
		'before' => '',
		'format' => get_option( 'time_format' ),
		'label'  => '',
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_modified_time' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'label' ] );

	$output = sprintf( '<time %s>', genesis_attr( 'entry-modified-time' ) ) . $atts['before'] . $atts['label'] . get_the_modified_time( $atts['format'] ) . $atts['after'] . '</time>';

	/**
	 * Change the output of the post_modified_time shortcode.
	 *
	 * @since 2.1.0
	 *
	 * @param string $output Markup containing post last modification time.
	 * @param array  $atts {
	 *     Shortcode attributes after merging with default values.
	 *
	 *     @type string $after Output after time.
	 *     @type string $before Output before time.
	 *     @type string $format Date format, could be 'relative'.
	 *     @type string $label Text following 'before' output, but before time.
	 * }
	 */
	return apply_filters( 'genesis_post_modified_time_shortcode', $output, $atts );

}

add_shortcode( 'post_author', 'genesis_post_author_shortcode' );
/**
 * Produces the author of the post (unlinked display name).
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is empty string).
 *
 * Output passes through `genesis_post_author_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Return empty string if post type does not support `author`, or has no author assigned.
 *                Otherwise, output for `post_author` shortcode.
 */
function genesis_post_author_shortcode( $atts ) {

	if ( ! post_type_supports( get_post_type(), 'author' ) ) {
		return '';
	}

	$author = get_the_author();

	if ( ! $author ) {
		return '';
	}

	$defaults = [
		'after'  => '',
		'before' => '',
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_author' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after' ] );

	$output  = sprintf( '<span %s>', genesis_attr( 'entry-author' ) );
	$output .= $atts['before'];
	$output .= sprintf( '<span %s>', genesis_attr( 'entry-author-name' ) ) . esc_html( $author ) . '</span>';
	$output .= $atts['after'];
	$output .= '</span>';

	return apply_filters( 'genesis_post_author_shortcode', $output, $atts );

}

add_shortcode( 'post_author_link', 'genesis_post_author_link_shortcode' );
/**
 * Produces the author of the post (link to author URL).
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is empty string).
 *
 * Output passes through `genesis_post_author_link_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Return empty string if post type does not support `author` or post has no author assigned.
 *                Return `genesis_post_author_shortcode()` if author has no URL.
 *                Otherwise, output for `post_author_link` shortcode.
 */
function genesis_post_author_link_shortcode( $atts ) {

	if ( ! post_type_supports( get_post_type(), 'author' ) ) {
		return '';
	}

	$url = get_the_author_meta( 'url' );

	// If no URL, use post author shortcode function.
	if ( ! $url ) {
		return genesis_post_author_shortcode( $atts );
	}

	$author = get_the_author();

	if ( ! $author ) {
		return '';
	}

	$defaults = [
		'after'  => '',
		'before' => '',
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_author_link' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after' ] );

	$output  = sprintf( '<span %s>', genesis_attr( 'entry-author' ) );
	$output .= $atts['before'];
	$output .= sprintf( '<a href="%s" %s>', $url, genesis_attr( 'entry-author-link' ) );
	$output .= sprintf( '<span %s>', genesis_attr( 'entry-author-name' ) );
	$output .= esc_html( $author );
	$output .= '</span></a>' . $atts['after'] . '</span>';

	return apply_filters( 'genesis_post_author_link_shortcode', $output, $atts );

}

add_shortcode( 'post_author_posts_link', 'genesis_post_author_posts_link_shortcode' );
/**
 * Produces the author of the post (link to author archive).
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is empty string).
 *
 * Output passes through `genesis_post_author_posts_link_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Return empty string if post type does not support `author` or post has no author assigned.
 *                Otherwise, output for `post_author_posts_link` shortcode.
 */
function genesis_post_author_posts_link_shortcode( $atts ) {

	if ( ! post_type_supports( get_post_type(), 'author' ) ) {
		return '';
	}

	$author = get_the_author();

	if ( ! $author ) {
		return '';
	}

	$defaults = [
		'after'  => '',
		'before' => '',
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_author_posts_link' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after' ] );

	$url = get_author_posts_url( get_the_author_meta( 'ID' ) );

	$output  = sprintf( '<span %s>', genesis_attr( 'entry-author' ) );
	$output .= $atts['before'];
	$output .= sprintf( '<a href="%s" %s>', $url, genesis_attr( 'entry-author-link' ) );
	$output .= sprintf( '<span %s>', genesis_attr( 'entry-author-name' ) );
	$output .= esc_html( $author );
	$output .= '</span></a>' . $atts['after'] . '</span>';

	return apply_filters( 'genesis_post_author_posts_link_shortcode', $output, $atts );

}

add_shortcode( 'post_comments', 'genesis_post_comments_shortcode' );
/**
 * Produces the link to the current post comments.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is empty string),
 *   hide_if_off (hide link if comments are off, default is 'enabled' (true)),
 *   more (text when there is more than 1 comment, use % character as placeholder
 *     for actual number, default is '% Comments')
 *   one (text when there is exactly one comment, default is '1 Comment'),
 *   zero (text when there are no comments, default is 'Leave a Comment').
 *
 * Output passes through `genesis_post_comments_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Return empty string if post does not support `comments`, or `hide_if_off` is enabled and
 *                comments are closed or disabled in Genesis theme settings.
 *                Otherwise, output for `post_comments` shortcode.
 */
function genesis_post_comments_shortcode( $atts ) {

	if ( ! post_type_supports( get_post_type(), 'comments' ) ) {
		return '';
	}

	$defaults = [
		'after'       => '',
		'before'      => '',
		'hide_if_off' => 'enabled',
		'more'        => __( '% Comments', 'genesis' ),
		'one'         => __( '1 Comment', 'genesis' ),
		'zero'        => __( 'Leave a Comment', 'genesis' ),
	];
	$atts     = shortcode_atts( $defaults, $atts, 'post_comments' );
	$atts     = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'zero', 'one', 'more' ] );

	if ( 'enabled' === $atts['hide_if_off'] && ( ! genesis_get_option( 'comments_posts' ) || ! comments_open() ) ) {
		return '';
	}

	// Darn you, WordPress!
	ob_start();
	comments_number( $atts['zero'], $atts['one'], $atts['more'] );
	$comments = ob_get_clean();

	$comments = sprintf( '<a href="%s">%s</a>', get_comments_link(), $comments );

	$output = genesis_markup(
		[
			'open'    => '<span class="entry-comments-link">',
			'close'   => '</span>',
			'content' => $atts['before'] . $comments . $atts['after'],
			'context' => 'comments-shortcode',
			'echo'    => false,
		]
	);

	return apply_filters( 'genesis_post_comments_shortcode', $output, $atts );

}

add_shortcode( 'post_tags', 'genesis_post_tags_shortcode' );
/**
 * Produces the tag links list.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is 'Tagged With: '),
 *   sep (separator string between tags, default is ', ').
 *
 * Output passes through `genesis_post_tags_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Return empty string if the `post_tag` taxonomy is not associated with the current post type
 *                or if the post has no tags. Otherwise, output for `post_tags` shortcode.
 */
function genesis_post_tags_shortcode( $atts ) {

	if ( ! is_object_in_taxonomy( get_post_type(), 'post_tag' ) ) {
		return '';
	}

	$defaults = [
		'after'  => '',
		'before' => __( 'Tagged With: ', 'genesis' ),
		'sep'    => ', ',
	];
	$atts     = shortcode_atts( $defaults, $atts, 'post_tags' );
	$atts     = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'sep' ] );

	$tags = get_the_tag_list( $atts['before'], trim( $atts['sep'] ) . ' ', $atts['after'] );

	// Do nothing if no tags.
	if ( ! $tags ) {
		return '';
	}

	$output = sprintf( '<span %s>', genesis_attr( 'entry-tags' ) ) . $tags . '</span>';

	return apply_filters( 'genesis_post_tags_shortcode', $output, $atts );

}

add_shortcode( 'post_categories', 'genesis_post_categories_shortcode' );
/**
 * Produces the category links list.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is 'Tagged With: '),
 *   sep (separator string between tags, default is ', ').
 *
 * Output passes through 'genesis_post_categories_shortcode' filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Return empty string if the `category` taxonomy is not associated with the current post type
 *                or if the post has no categories. Otherwise, output for `post_categories` shortcode.
 */
function genesis_post_categories_shortcode( $atts ) {

	if ( ! is_object_in_taxonomy( get_post_type(), 'category' ) ) {
		return '';
	}

	$defaults = [
		'sep'    => ', ',
		'before' => __( 'Filed Under: ', 'genesis' ),
		'after'  => '',
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_categories' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'sep' ] );

	$cats = get_the_category_list( trim( $atts['sep'] ) . ' ' );

	// Do nothing if there are no categories.
	if ( ! $cats ) {
		return '';
	}

	$output = sprintf( '<span %s>', genesis_attr( 'entry-categories' ) ) . $atts['before'] . $cats . $atts['after'] . '</span>';

	return apply_filters( 'genesis_post_categories_shortcode', $output, $atts );

}

add_shortcode( 'post_terms', 'genesis_post_terms_shortcode' );
/**
 * Produces the linked post taxonomy terms list.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is 'Filed Under: '),
 *   sep (separator string between tags, default is ', '),
 *    taxonomy (name of the taxonomy, default is 'category').
 *
 * Output passes through `genesis_post_terms_shortcode` filter before returning.
 *
 * @since 1.6.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Output for `post_terms` shortcode, or empty string on failure to retrieve terms.
 */
function genesis_post_terms_shortcode( $atts ) {

	$defaults = [
		'after'    => '',
		'before'   => __( 'Filed Under: ', 'genesis' ),
		'sep'      => ', ',
		'taxonomy' => 'category',
	];

	/**
	 * Post terms shortcode defaults.
	 *
	 * Allows the default args in the post terms shortcode function to be filtered.
	 *
	 * @since 2.0.0
	 *
	 * @param array $defaults The default args array.
	 */
	$defaults = apply_filters( 'genesis_post_terms_shortcode_defaults', $defaults );

	$atts = shortcode_atts( $defaults, $atts, 'post_terms' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'sep' ] );

	$terms = get_the_term_list( get_the_ID(), $atts['taxonomy'], $atts['before'], trim( $atts['sep'] ) . ' ', $atts['after'] );

	if ( is_wp_error( $terms ) ) {
			return '';
	}

	if ( empty( $terms ) ) {
			return '';
	}

	$output = sprintf( '<span %s>', genesis_attr( 'entry-terms' ) ) . $terms . '</span>';

	return apply_filters( 'genesis_post_terms_shortcode', $output, $terms, $atts );

}

add_shortcode( 'post_edit', 'genesis_post_edit_shortcode' );
/**
 * Produces the edit post link for logged in users.
 *
 * Supported shortcode attributes are:
 *   after (output after link, default is empty string),
 *   before (output before link, default is 'Tagged With: '),
 *   link (link text, default is '(Edit)').
 *
 * Output passes through `genesis_post_edit_shortcode` filter before returning.
 *
 * @since 1.1.0
 *
 * @param array|string $atts Shortcode attributes. Empty string if no attributes.
 * @return string Output for `post_edit` shortcode, or empty string if `genesis_edit_post_link` filter returns `false`.
 */
function genesis_post_edit_shortcode( $atts ) {

	if ( ! apply_filters( 'genesis_edit_post_link', true ) ) {
		return '';
	}

	$defaults = [
		'after'  => '',
		'before' => '',
		'link'   => __( '(Edit)', 'genesis' ),
	];

	$atts = shortcode_atts( $defaults, $atts, 'post_edit' );
	$atts = genesis_sanitize_shortcode_attributes( $atts, [ 'before', 'after', 'link' ] );

	// Darn you, WordPress!
	ob_start();
	edit_post_link( $atts['link'], $atts['before'], $atts['after'] );
	$edit = ob_get_clean();

	$output = $edit;

	return apply_filters( 'genesis_post_edit_shortcode', $output, $atts );

}
