<?php

/**
 * MonsterInsights PPC Tracking Google Provider class.
 *
 * @access public
 * @since 1.0.0
 *
 * @package MonsterInsights_PPC_Tracking
 * @subpackage Ad_Providers
 * @author  David Paternina
 */
class MonsterInsights_PPC_Tracking_Google extends MonsterInsights_PPC_Tracking_Provider {

    /**
     * Google Ads conversion id
     *
     * @var string
     */
    private $conversion_id;

    /**
     * Google Ads conversion label
     *
     * @var string
     */
    private $conversion_label;

    public function __construct() {
        parent::__construct();

        $this->conversion_id = $this->get_conversion_id();
        $this->conversion_label = $this->get_conversion_label();
    }

    /**
     * @inheritDoc
     */
    public function get_provider_id() {
        return 'google';
    }

    /**
     * @inheritDoc
     */
    protected function add_frontend_hooks() {
        add_action( 'monsterinsights_frontend_tracking_gtag_after_pageview', array( $this, 'add_conversion_id_to_gtag_tracking' ) );
    }

	/**
	 * Not in use for Google
	 */
	protected function init_server_handler() {
		return null;
	}

    /**
     * @return void
     */
    public function add_conversion_id_to_gtag_tracking() {
        $aw_id = esc_attr( $this->conversion_id );

	    $data_for_google = [];

        if ( ! empty( $aw_id ) ) {
	        $enable_enhanced_conversions = boolval( monsterinsights_get_option( 'ads_google_enhanced_conversions' ) );

	        if ( !empty($enable_enhanced_conversions) ) {
		        $data_for_google['allow_enhanced_conversions'] = true;
	        }

            echo "__gtagTracker( 'config', '" . $aw_id . "', " . json_encode( $data_for_google ) . " );\n";
        }
    }

    /**
     * @inheritDoc
     */
    public function maybe_print_conversion_code( $conversion_data, $customer_info = [] ) {
		//  Conversions require both conversion id and label
        if ( empty( $this->conversion_id ) || empty( $this->conversion_label ) ) {
            return false;
        }

        $data_for_google = [
            'send_to'           => $this->get_conversion_id_and_label(),
            'value'             => $conversion_data['order_total'],
            'currency'          => $conversion_data['currency'],
            'transaction_id'    => $conversion_data['order_id'],
        ];

		$this->maybe_print_enhanced_conversion_data( $customer_info );

        // wp_json_encode not used due to supported WP version.
        echo stripslashes( "\n__gtagTracker( 'event', 'conversion', " . json_encode( $data_for_google ) . " );\n" );
        return true;
    }

	/**
	 * Maybe print enhanced conversion data
	 * @param $customer_info
	 *
	 * @return void
	 */
	private function maybe_print_enhanced_conversion_data( $customer_info ) {
		if ( empty( $customer_info ) || empty( $customer_info['email'] ) ) {
			return;
		}

		$user_data = $this->get_array_with_hashed_values( $customer_info );

		if ( !empty( $customer_info['address'] ) ) {
			$user_data['address'] = $this->get_array_with_hashed_values( $customer_info['address'] );
		}

		// wp_json_encode not used due to supported WP version.
		echo stripslashes( "\n__gtagTracker( 'set', 'user_data', " . json_encode( $user_data ) . " );\n" );
	}

	/**
	 * Get array with hashed values for Google Enhanced Conversions
	 * @param $data
	 *
	 * @return array
	 */
	private function get_array_with_hashed_values( $data )
	{
		$hashable_keys = ['email', 'phone_number', 'first_name', 'last_name'];

		$result = [];

		foreach ( $data as $key => $value ) {
			if ( in_array( $key, $hashable_keys ) ) {
				$result[ 'sha256_' . $key ] = $this->hash( $value );
			} else {
				$result[ $key ] = $value;
			}
		}

		return $result;
	}

    /**
     * Get concatenated conversion id and label
     *
     * @return string
     */
    private function get_conversion_id_and_label() {
        return $this->conversion_id . '/' . $this->conversion_label;
    }

    /**
     * Get conversion id from settings
     *
     * @return string
     */
    private function get_conversion_id() {
        $conversion_id = monsterinsights_get_option( 'ads_google_conversion_id' );
        return apply_filters( 'monsterinsights_ppc_tracking_google_conversion_id', $conversion_id );
    }

    /**
     * Get conversion label from settings
     *
     * @return string
     */
    private function get_conversion_label() {
        return apply_filters( 'monsterInsights_ppc_tracking_google_label_id', monsterinsights_get_option( 'ads_google_conversion_label' ) );
    }

	/**
	 * @inheritDoc
	 */
	public function get_tracking_id()
	{
		$this->get_conversion_id();
	}

	/**
	 * Not in use for Google
	 */
	public function get_api_token()
	{
		return null;
	}

    /**
     * @inheritdoc
     */
    public function is_active() {
        $is_active = !empty( $this->get_conversion_id() );

        return apply_filters(
            "monsterinsights_ppc_tracking_provider_active_{$this->get_provider_id()}",
            $is_active
        );
    }
}
