<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MonsterInsights PPC tracking Server base class.
 *
 * @since 1.2.0
 * @package MonsterInsights_PPC_Tracking
 * @subpackage Ad_Providers/Server
 * @author David Paternina
 */
abstract class MonsterInsights_PPC_Server
{
	/**
	 * The event names that the server can handle
	 * @var array[]
	 */
	protected $event_names = [];

	/**
	 * The provider id
	 * @var MonsterInsights_PPC_Tracking_Provider
	 */
	protected $provider;

	/**
	 * @param MonsterInsights_PPC_Tracking_Provider $provider
	 */
	public function __construct($provider)
	{
		$this->provider = $provider;
	}

	/**
	 * Send an event to the server
	 *
	 * @param string $event_name
	 * @param array $data
	 * @param MonsterInsights_PPC_Tracking_Ecommerce_Tracking $provider
	 *
	 * @return array|WP_Error
	 */
	public function send_event($event_name, $data, $provider)
	{
		if ( !$this->has_event( $event_name ) ){
			return null;
		}

		do_action("monsterinsights_ppc_server_before_send_event_{$this->provider->get_provider_id()}", $event_name, $data);

		// For clarity, server events are named event_[event_name].
		$event_function_name = 'event_' . $event_name;
		$result = call_user_func([ $this, $event_function_name ], $data, $provider);

		do_action("monsterinsights_ppc_server_after_send_event_{$this->provider->get_provider_id()}", $event_name, $result);

		return $result;
	}

	/**
	 * Process event data in the format for the API, optional.
	 *
	 * @param array $data The event data.
	 *
	 * @return array
	 */
	public function prepare_event( $data )
	{
		return $data;
	}

	/*
	 * Get available events for the provider
	 * @return mixed
	 */
	public function has_event( $event_name )
	{
		$events = $this->get_events();
		return !empty( $events[ $event_name ] ) && method_exists($this, "event_{$event_name}");
	}

	/**
	 * Get available events for the provider
	 * @return mixed|null
	 */
	public function get_events()
	{
		return apply_filters("monsterinsights_ppc_server_events_{$this->provider->get_provider_id()}", $this->event_names);
	}

	/**
	 * Get the event name to use for a specific event, by the event key.
	 * This adds a filter to allow customizing events for different steps of the funnel.
	 *
	 * @param string $event_key The event key.
	 *
	 * @return string
	 */
	public function get_event_name( $event_key ) {
		$event_name = '';
		if ( isset( $this->event_names[ $event_key ] ) ) {
			$event_name = $this->event_names[ $event_key ];
		}

		return apply_filters( "monsterinsights_ppc_server_event_name_{$this->provider->get_provider_id()}", $event_name, $event_key );
	}

	/**
	 * Get the extra order data for the provider.
	 *
	 * @return array
	 */
	public function get_extra_order_data()
	{
		return [];
	}

	/**
	 *
	 * Send a request to the API server.
	 *
	 * @param mixed $data The data to send in the body.
	 * @param string $method The method to use for the request. Default 'POST'.
	 * @return array|WP_Error
	 */
	protected function send_request($data, $method = 'POST')
	{
		$data = apply_filters( "monsterinsights_ppc_server_event_data_{$this->provider->get_provider_id()}", $this->prepare_event( $data ) );

		$url     = $this->get_api_url();
		$headers = $this->get_headers();

		// Send the request.
		$response = wp_remote_request(
			$url,
			array(
				'headers'  => $headers,
				'method'   => $method,
				'body'     => $data,
			)
		);

		return $response;
	}

	/**
	 * Get headers for the server call
	 * @return array
	 */
	protected function get_headers()
	{
		return [];
	}

	/**
	 * Hash a value with sha256.
	 *
	 * @param string $value The value to hash,
	 *
	 * @return string
	 */
	protected function hash( $value ) {
		return $this->provider->hash( $value );
	}

	/**
	 * Get the user agent.
	 *
	 * @return string
	 */
	protected function get_user_agent()
	{
		return isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : ''; // @codingStandardsIgnoreLine
	}

	/**
	 * Get the api url for the server
	 *
	 * @return string
	 */
	abstract protected function get_api_url();
}