<?php
/**
 * Log Site Notes events from settings.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class MonsterInsights_Site_Notes_Important_Events_Log_Events {
	/**
	 * Holds instance of self object.
	 */
	private static $instance;

	/**
	 * Create singleton instance of the class.
	 *
	 * @return self
	 */
	public static function get_instance() {
		if ( ! self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor of the class.
	 */
	private function __construct() {
		add_action( 'activated_plugin', array( $this, 'plugin_activated' ) );
		add_action( '_core_updated_successfully', array( $this, 'core_updated' ) );
		add_filter( 'site_status_test_result', array( $this, 'site_health_result' ), 10, 1 );
	}

	/**
	 * Log site note for plugin activated.
	 */
	public function plugin_activated( $plugin ) {
		$all_plugins = get_plugins();

		if ( ! isset( $all_plugins[ $plugin ] ) ) {
			return;
		}

		$log_note = monsterinsights_get_option( 'site_notes_log_plugin_activated', false );

		if ( true != $log_note ) {
			return;
		}

		if ( ! $this->is_user_licensed() ) {
			return;
		}

		$plugin_name = $all_plugins[ $plugin ]['Name'];

		monsterinsights_add_site_note(
			array(
				/* translators: Plugin name */
				'description' => sprintf( __( '%s plugin activated.', 'monsterinsights-site-notes-important-events' ), $plugin_name ),
			)
		);
	}

	/**
	 * Log site note for WordPress version updated
	 */
	public function core_updated( $wp_version ) {
		$log_note = monsterinsights_get_option( 'site_notes_log_wordpress_version_updated', false );

		if ( true != $log_note ) {
			return;
		}

		if ( ! $this->is_user_licensed() ) {
			return;
		}

		monsterinsights_add_site_note( array(
			/* translators: WordPress version */
			'description' => sprintf( __( 'WordPress version updated to %s', 'monsterinsights-site-notes-important-events'), $wp_version ),
		) );
	}

	/**
	 * Log Site Health notice.
	 *
	 * @return array
	 */
	public function site_health_result( $test_result ) {
		// If the value is falsy.
		if ( ! $test_result ) {
			return $test_result;
		}

		// We are logging only critical items.
		if ( 'critical' != $test_result['status'] ) {
			return $test_result;
		}

		$log_note = monsterinsights_get_option( 'site_notes_log_new_site_health_notice', false );

		if ( true != $log_note ) {
			return $test_result;
		}

		if ( ! $this->is_user_licensed() ) {
			return $test_result;
		}

		$option_key = 'site_notes_loged_' . $test_result['test'];

		$logged = get_option( $option_key );

		// If we already logged the same issue.
		if ( $logged ) {
			return $test_result;
		}

		monsterinsights_add_site_note( array(
			'description' => $test_result['label'],
		) );

		add_option( $option_key, 'true', '', 'no' );

		return $test_result;
	}

	/**
	 * Check the user is agency licensed.
	 */
	private function is_user_licensed() {
		return MonsterInsights()->license->license_can( 'agency' );
	}
}
