<?php

/**
 * Plugin Name:       MonsterInsights - PPC Tracking Addon
 * Plugin URI:        https://www.monsterinsights.com
 * Description:       Easily track conversions and website events with Google Ads, Bing Ads, and Meta.
 * Author:            MonsterInsights Team
 * Author URI:        https://www.monsterinsights.com
 * Version:           1.2.0
 * Requires at least: 4.8.0
 * Requires PHP:      5.5
 * Text Domain:       monsterinsights-ppc-tracking
 * Domain Path:       languages
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main plugin class.
 *
 * @since 1.0.0
 *
 * @package MonsterInsights_PPC_Tracking_Premium
 * @author  David Paternina
 */
class MonsterInsights_PPC_Tracking_Premium
{
    /**
     * Holds the class object.
     *
     * @since 1.0.0
     *
     * @var object
     */
    public static $instance;

    /**
     * Plugin version, used for cache-busting of style and script file references.
     *
     * @since 1.0.0
     *
     * @var string
     */
    public $version = '1.2.0';

    /**
     * The name of the plugin.
     *
     * @since 1.0.0
     *
     * @var string
     */
    public $plugin_name = 'MonsterInsights PPC Tracking';

    /**
     * Unique plugin slug identifier.
     *
     * @since 1.0.0
     *
     * @var string
     */
    public $plugin_slug = 'monsterinsights-ppc-tracking';

    /**
     * Plugin file.
     *
     * @since 1.0.0
     *
     * @var string
     */
    public $file;

    public function __construct()
    {
        $this->file = __FILE__;

        if (!$this->check_compatibility()) {
            return;
        }

        // Define Addon Constant
        if (!defined('MONSTERINSIGHTS_PPC_TRACKING_VERSION')) {
            define('MONSTERINSIGHTS_PPC_TRACKING_VERSION', $this->version);
        }

        // Load the plugin textdomain.
        add_action('plugins_loaded', array($this, 'load_plugin_textdomain'));

        // Load the updater
        add_action('monsterinsights_updater', array($this, 'updater'));

        // Load the plugin.
        add_action('monsterinsights_load_plugins', array($this, 'init'), 99);

        if (!defined('MONSTERINSIGHTS_PRO_VERSION')) {
            // Make sure plugin is listed in Auto-update Disabled view
            add_filter('auto_update_plugin', array($this, 'disable_auto_update'), 10, 2);

            // Display call-to-action to get Pro in order to enable auto-update
            add_filter('plugin_auto_update_setting_html', array($this, 'modify_autoupdater_setting_html'), 11, 2);
        }
    }

    /**
     * Check compatibility with PHP and WP, and display notices if necessary
     *
     * @return bool
     * @since 1.0.0
     */
    private function check_compatibility()
    {
        if (defined('MONSTERINSIGHTS_FORCE_ACTIVATION') && MONSTERINSIGHTS_FORCE_ACTIVATION) {
            return true;
        }

        require_once plugin_dir_path(__FILE__) . 'includes/class-monsterinsights-ppc-tracking-compatibility-check.php';
        $compatibility = MonsterInsights_PPC_Tracking_Premium_Compatibility_Check::get_instance();
        $compatibility->maybe_display_notice();

        return $compatibility->is_php_compatible() && $compatibility->is_wp_compatible();
    }

    /**
     * Loads the plugin textdomain for translation.
     *
     * @since 1.0.0
     */
    public function load_plugin_textdomain()
    {
        load_plugin_textdomain($this->plugin_slug, false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }

    /**
     * Loads the plugin into WordPress.
     *
     * @since 1.0.0
     */
    public function init()
    {

        if (!defined('MONSTERINSIGHTS_PRO_VERSION')) {
            // admin notice, MI not installed
            add_action('admin_notices', array(self::$instance, 'requires_monsterinsights'));

            return;
        }

        if (version_compare(MONSTERINSIGHTS_VERSION, '8.10.0', '<')) {
            // MonsterInsights version not supported
            add_action('admin_notices', array(self::$instance, 'requires_monsterinsights_version'));

            return;
        }

        if ( class_exists('MonsterInsights_Ads') ) {
            // When MonsterInsights ads and ppc both activate
            add_action('admin_notices', array(self::$instance, 'monsterinsights_ads_and_ppc_both_activate'));

            return;
        }

        $this->require_deprecation_files();
        $this->require_tracker();
    }

	/**
	 * @return void
	 */
    private function require_deprecation_files()
    {
        //  Deprecated hooks
        require_once plugin_dir_path(__FILE__) . 'includes/deprecated/hooks.php';
    }

    /**
     * Load integration classes
     *
     * @return void
     */
    private function require_tracker()
    {
        //  Load main tracking class
        require_once plugin_dir_path(__FILE__) . 'includes/class-monsterinsights-ppc-tracking.php';
    }

    /**
     * Initializes the addon updater.
     *
     * @param string $key The user license key.
     *
     * @since 1.0.0
     *
     */
    public function updater($key)
    {
        $args = array(
            'plugin_name' => $this->plugin_name,
            'plugin_slug' => $this->plugin_slug,
            'plugin_path' => plugin_basename(__FILE__),
            'plugin_url'  => trailingslashit(WP_PLUGIN_URL) . $this->plugin_slug,
            'remote_url'  => 'https://www.monsterinsights.com/',
            'version'     => $this->version,
            'key'         => $key
        );

        new MonsterInsights_Updater($args);
    }

    /**
     * Display MonsterInsights Pro CTA on Plugins -> autoupdater setting column
     *
     * @param string $html
     * @param string $plugin_file
     *
     * @return string
     */
    public function modify_autoupdater_setting_html($html, $plugin_file)
    {
        if (
            plugin_basename(__FILE__) === $plugin_file &&
            // If main plugin (free) happens to be enabled and already takes care of this, then bail
            !apply_filters("monsterinsights_is_autoupdate_setting_html_filtered_" . $plugin_file, false)
        ) {
            $html = sprintf(
                '<a href="%s">%s</a>',
                'https://www.monsterinsights.com/docs/go-lite-pro/?utm_source=liteplugin&utm_medium=plugins-autoupdate&utm_campaign=upgrade-to-autoupdate&utm_content=monsterinsights-ppc-tracking',
                __('Enable the MonsterInsights PRO plugin to manage auto-updates', 'monsterinsights-ppc-tracking')
            );
        }

        return $html;
    }

    /**
     * Disable auto-update.
     *
     * @param $update
     * @param $item
     *
     * @return bool
     */
    public function disable_auto_update($update, $item)
    {
        // If this is multisite and is not on the main site, return early.
        if (is_multisite() && !is_main_site()) {
            return $update;
        }

        if (isset($item->id) && plugin_basename(__FILE__) === $item->id) {
            return false;
        }

        return $update;
    }

    /**
     * Output a nag notice if the user does not have MI installed
     *
     * @access public
     * @return    void
     * @since 1.0.0
     *
     */
    public function requires_monsterinsights()
    {
?>
        <div class="error">
            <p><?php esc_html_e('Please install MonsterInsights Pro to use the MonsterInsights PPC Tracking addon', 'monsterinsights-ppc-tracking'); ?></p>
        </div>
    <?php
    }

    /**
     * Output a nag notice if the user does not have MI version installed
     *
     * @access public
     * @return    void
     * @since 1.0.0
     *
     */
    public function requires_monsterinsights_version()
    {
    ?>
        <div class="error">
            <p><?php esc_html_e('Please install or update MonsterInsights Pro with version 8.10 or newer to use the MonsterInsights PPC Tracking addon', 'monsterinsights-ppc-tracking'); ?></p>
        </div>
<?php
    }

    /**
     * Perform plugin install tasks:
     *
     * - Migrate old gtag options
     *
     * @return void
     */
    public static function install()
    {

        if (!defined('MONSTERINSIGHTS_PRO_VERSION')) {
            return;
        }

        //  Check for old addon config
        $old_gtag_conversion_id = monsterinsights_get_option('gtag_ads_conversion_id', null);

        if (!empty($old_gtag_conversion_id)) {
            monsterinsights_update_option('ads_google_conversion_id', $old_gtag_conversion_id);
        }

        $old_gtag_conversion_label = monsterinsights_get_option('gtag_ads_conversion_label', null);

        if (!empty($old_gtag_conversion_label)) {
            monsterinsights_update_option('ads_google_conversion_label', $old_gtag_conversion_label);
        }
    }

    /**
     * Returns the singleton instance of the class.
     *
     * @return object The MonsterInsights_PPC_Tracking_Premium object.
     * @since 1.0.0
     *
     */
    public static function get_instance()
    {
        if (!isset(self::$instance) && !(self::$instance instanceof MonsterInsights_PPC_Tracking_Premium)) {
            self::$instance = new MonsterInsights_PPC_Tracking_Premium();
        }

        return self::$instance;
    }

    /**
     * Output a notice if user activated both Ads and PPC.
     *
     * @return void
     */
    public function monsterinsights_ads_and_ppc_both_activate() {
        ?>
        <div class="notice notice-warning">
            <p><?php esc_html_e('You have activated both MonsterInsights Ads and MonsterInsights PPC Tracking addon, Please deactivate MonsterInsights Ads addon. The settings from the prior plugin have been automatically carried over.', 'monsterinsights-ppc-tracking'); ?></p>
        </div>
        <?php
    }
}

register_activation_hook(__FILE__, array('MonsterInsights_PPC_Tracking_Premium', 'install'));

// Load the main plugin class.
$MonsterInsights_PPC_Tracking_premium = MonsterInsights_PPC_Tracking_Premium::get_instance();
