<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MonsterInsights PPC Tracking Meta Server class.
 *
 * @since 1.2.0
 * @package MonsterInsights_PPC_Tracking
 * @subpackage Ad_Providers/Server
 * @author David Paternina
 */
class MonsterInsights_PPC_Meta_Server extends MonsterInsights_PPC_Server
{

	/**
	 * @inheritdoc
	 */
	protected $event_names = [
		'add_to_cart'       => 'AddToCart',
		'view_content'      => 'ViewContent',
		'begin_checkout'    => 'InitiateCheckout',
		'purchase'          => 'Purchase',
	];

	/**
	 * @inheritdoc
	 */
	public function prepare_event( $data )
	{
		$data['action_source'] = 'website';

		$data = array(
			'data'         => array( wp_json_encode( $data ) ),
			'access_token' => $this->provider->get_api_token(),
		);

		$test_event_code = apply_filters( 'monsterinsights_ppc_meta_test_event_code', false );
		if ( false !== $test_event_code ) {
			$data['test_event_code'] = $test_event_code;
		}

		return $data;
	}

	/**
	 * Send AddToCart event to Meta API
	 * @param $data
	 *
	 * @return void
	 */
	public function event_add_to_cart( $data )
	{
		$event_data = array(
			'event_name'  => $this->get_event_name( 'add_to_cart' ),
			'event_time'  => time(),
			'custom_data' => [
				'product_name' => $data['name'],
				'content_type' => 'product',
				'currency'     => $data['currency'],
				'value'        => $data['price'],
			],
			'contents'    => [
				[
					'id'         => $data['product_id'],
					'quantity'   => $data['quantity'],
					'item_price' => $data['price'],
				],
			],
			'content_ids' =>  [(string) $data['product_id']],
		);

		$event_data['user_data'] = $this->get_user_data( $data );

		// Send the event to the server.
		$this->send_request( $event_data );
	}

	/**
	 * Send ViewContent event to Meta API
	 * @param $data
	 *
	 * @return void
	 */
	public function event_view_content( $data )
	{
		$event_data = array(
			'event_name'  => $this->get_event_name( 'view_content' ),
			'event_time'  => time(),
			'custom_data' => [
				'product_name' => $data['name'],
				'content_type' => 'product',
				'currency'     => $data['currency'],
				'value'        => $data['price'],
			],
			'contents'    => [
				[
					'id'         => $data['product_id'],
					'item_price' => $data['price'],
				],
			],
			'content_ids' =>  [(string) $data['product_id']],
		);

		$event_data['user_data'] = $this->get_user_data( $data );

		// Send the event to the server.
		$this->send_request( $event_data );
	}

	/**
	 * Send BeginCheckout event to Meta API
	 * @param $data
	 * @param $provider
	 *
	 * @return void
	 */
	public function event_begin_checkout( $data )
	{

		$event_data = array(
			'event_name'  => $this->get_event_name( 'begin_checkout' ),
			'event_time'  => time(),
			'custom_data' => [
				'content_type' => 'product',
				'num_items'    => isset($data['num_items']) ? $data['num_items'] : 1,
				'currency'     => isset($data['currency']) ? $data['currency'] : 'USD',
				'value'        => isset($data['total']) ? $data['total'] : 0,
			],
			'contents'    => [],
			'content_ids' => [],
		);

		if ( isset($data['products']) ) {
			foreach ( $data['products'] as $product ) {
				$event_data['custom_data']['content_ids'][] = $product['id'];
				$event_data['contents'][]                   = [
					'id'       => $product['id'],
					'quantity' => $product['quantity'],
				];
			}
		}

		$event_data['user_data'] = $this->get_user_data( $data );

		$this->send_request( $event_data );
	}

	/**
	 * Send Purchase event to Meta API
	 * @param $data
	 * @param MonsterInsights_PPC_Tracking_Ecommerce_Tracking $ecommerce_provider
	 *
	 * @return void
	 */
	public function event_purchase( $data, $ecommerce_provider )
	{
		$tracked = $ecommerce_provider->already_tracked_server( $data['order_id'], $this->provider->get_provider_id() );

		// If the event for this specific order and pixel was already sent, bail.
		if ( $tracked ) {
			return;
		}

		$event_data = array(
			'event_name'  => $this->get_event_name( 'purchase' ),
			'event_time'  => time(),
			'custom_data' => array(
				'content_type' => 'product',
				'num_items'    => $data['num_items'],
				'currency'     => $data['currency'],
				'value'        => $data['total'],
			),
			'contents'    => array(),
			'content_ids' => array(),
		);

		foreach ( $data['products'] as $product ) {
			$event_data['custom_data']['content_ids'][] = $product['id'];
			$event_data['contents'][]                   = array(
				'id'       => $product['id'],
				'quantity' => $product['quantity'],
			);
		}

		$event_data['user_data'] = $this->get_user_data( $data );

		$event_data['user_data']['fbc'] = $ecommerce_provider->get_order_meta( $data['order_id'], '_monsterinsights_ppc_meta_fbc' );
		$event_data['user_data']['fbp'] = $ecommerce_provider->get_order_meta( $data['order_id'], '_monsterinsights_ppc_meta_fbp' );

		$this->send_request( $event_data );

		$ecommerce_provider->mark_order_tracked_server( $data['order_id'], $this->provider->get_provider_id() );
	}

	/**
	 * Get an array of user data formatted for a server request to FB.
	 *
	 * @param array $data The data from the provider.
	 *
	 * @return array The data processed
	 */
	public function get_user_data( $data ) {
		$data = wp_parse_args(
			$data,
			array(
				'user_ip'    => '',
				'user_agent' => '',
				'user_id'    => '',
				'email'      => '',
				'city'       => '',
				'state'      => '',
				'country'    => '',
				'zip'        => '',
				'first_name' => '',
				'last_name'  => '',
				'phone'      => '',
			)
		);

		$user_data = array(
			'client_ip_address' => $data['user_ip'],
			'client_user_agent' => $data['user_agent'],
			'fbc'               => $this->get_fbc(),
			'fbp'               => $this->get_fbp(),
			'em'                => array(
				$this->hash( $data['email'] ),
			),
			'external_id'       => $this->hash( $data['user_id'] ),
			'ct'                => $this->hash( $data['city'] ),
			'st'                => $this->hash( $data['state'] ),
			'country'           => $this->hash( $data['country'] ),
			'zp'                => $this->hash( $data['zip'] ),
			'fn'                => $this->hash( $data['first_name'] ),
			'ln'                => $this->hash( $data['last_name'] ),
			'ph'                => $this->hash( $data['phone'] ),
		);

		foreach ( $user_data as $key => $value ) {
			if ( ! is_array( $value ) && '' === (string) $value ) {
				unset( $user_data[ $key ] );
			}
		}

		return $user_data;
	}

	/**
	 * Get facebook click id from the browser cookie.
	 *
	 * @return string
	 */
	public function get_fbc() {
		if ( isset( $_COOKIE['_fbc'] ) ) {
			return sanitize_text_field( $_COOKIE['_fbc'] );
		}

		return '';
	}

	/**
	 * Get facebook browser id from the browser cookie.
	 *
	 * @return string
	 */
	public function get_fbp() {
		if ( isset( $_COOKIE['_fbp'] ) ) {
			return sanitize_text_field( $_COOKIE['_fbp'] );
		}

		return '';
	}

	/**
	 * Look for fb-specific cookies and store them with the order for proper attribution when order is marked
	 * as complete.
	 *
	 * @return array
	 */
	public function get_extra_order_data() {
		$extra_data = array();

		$fbc = $this->get_fbc();
		if ( ! empty( $fbc ) ) {
			$extra_data['_monsterinsights_ppc_meta_fbc'] = $fbc;
		}

		$fbp = $this->get_fbp();
		if ( ! empty( $fbp ) ) {
			$extra_data['_monsterinsights_ppc_meta_fbp'] = $fbp;
		}

		return $extra_data;
	}

	/**
	 * @inheritdoc
	 */
	protected function get_api_url()
	{
		return 'https://graph.facebook.com/v20.0/' . $this->provider->get_tracking_id() . '/events';
	}
}