<?php
/**
 * Plugin Name:       MonsterInsights - Site Notes Important Events Addon
 * Plugin URI:        https://www.monsterinsights.com
 * Description:       Designed specifically for enterprise and marketing agencies, our Site Notes Addon lets you easily create, update, and delete site notes programmatically.
 * Author:            MonsterInsights Team
 * Author URI:        https://www.monsterinsights.com
 * Version:           1.0.0
 * Requires at least: 5.6.0
 * Requires PHP:      7.2
 * Text Domain:       monsterinsights-site-notes-important-events
 * Domain Path:       languages
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
	exit;
}

/**
 * Main plugin class.
 *
 * @since 1.0.0
 *
 * @package MonsterInsights_Site_Notes_Important_Events
 */
class MonsterInsights_Site_Notes_Important_Events {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Plugin version, used for cache-busting of style and script file references.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $version = '1.0.0';

	/**
	 * The name of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'MonsterInsights Site Notes Important Events';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'monsterinsights-site-notes-important-events';

	/**
	 * Plugin file.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $file;

	/**
	 * Site note model class.
	 *
	 * @since 1.0.0
	 *
	 * @var MonsterInsights_Site_Notes_Important_Events_Model
	 */
	public $model;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->file = __FILE__;

		// Load the plugin textdomain.
		add_action( 'plugins_loaded', array( $this, 'load_plugin_textdomain' ) );

		// Load the updater
		add_action( 'monsterinsights_updater', array( $this, 'updater' ) );

		// Load the plugin.
		add_action( 'monsterinsights_load_plugins', array( $this, 'init' ), 99 );

		if ( ! defined( 'MONSTERINSIGHTS_PRO_VERSION' ) ) {
			// Make sure plugin is listed in Auto-update Disabled view
			add_filter( 'auto_update_plugin', array( $this, 'disable_auto_update' ), 10, 2 );

			// Display call-to-action to get Pro in order to enable auto-update
			add_filter( 'plugin_auto_update_setting_html', array( $this, 'modify_autoupdater_setting_html' ), 11, 2 );
		}
	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.0
	 */
	public function load_plugin_textdomain() {
		load_plugin_textdomain($this->plugin_slug, false, dirname(plugin_basename(__FILE__)) . '/languages/');
	}

	/**
	 * Initializes the addon updater.
	 *
	 * @param string $key The user license key.
	 *
	 * @since 1.0.0
	 */
	public function updater( $key ) {
		$args = array(
			'plugin_name' => $this->plugin_name,
			'plugin_slug' => $this->plugin_slug,
			'plugin_path' => plugin_basename(__FILE__),
			'plugin_url'  => trailingslashit(WP_PLUGIN_URL) . $this->plugin_slug,
			'remote_url'  => 'https://www.monsterinsights.com/',
			'version'     => $this->version,
			'key'         => $key
		);

		new MonsterInsights_Updater($args);
	}

	/**
	 * Disable auto-update.
	 *
	 * @param $update
	 * @param $item
	 *
	 * @return bool
	 */
	public function disable_auto_update($update, $item) {
		// If this is multisite and is not on the main site, return early.
		if (is_multisite() && !is_main_site()) {
			return $update;
		}

		if (isset($item->id) && plugin_basename(__FILE__) === $item->id) {
			return false;
		}

		return $update;
	}

	/**
	 * Display MonsterInsights Pro CTA on Plugins -> autoupdater setting column
	 *
	 * @param string $html
	 * @param string $plugin_file
	 *
	 * @return string
	 */
	public function modify_autoupdater_setting_html( $html, $plugin_file ) {
		if (
			plugin_basename(__FILE__) === $plugin_file &&
			// If main plugin (free) happens to be enabled and already takes care of this, then bail
			!apply_filters("monsterinsights_is_autoupdate_setting_html_filtered_" . $plugin_file, false)
		) {
			$html = sprintf(
				'<a href="%s">%s</a>',
				'https://www.monsterinsights.com/docs/go-lite-pro/?utm_source=liteplugin&utm_medium=plugins-autoupdate&utm_campaign=upgrade-to-autoupdate&utm_content=monsterinsights-site-notes-important-events',
				__('Enable the MonsterInsights PRO plugin to manage auto-updates', 'monsterinsights-site-notes-important-events')
			);
		}

		return $html;
	}

	/**
	 * Loads the plugin into WordPress.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		if ( ! defined( 'MONSTERINSIGHTS_PRO_VERSION' ) ) {
			// admin notice, MI not installed
			add_action( 'admin_notices', array( $this, 'requires_monsterinsights' ) );

			return;
		}

		if ( version_compare( MONSTERINSIGHTS_VERSION, '8.26', '<' ) ) {
			// MonsterInsights version not supported
			add_action('admin_notices', array($this, 'requires_monsterinsights_version'));

			return;
		}

		// Load all necessary files.
		$this->require_files();

		// Load Uncanny Automator files.
		add_action( 'automator_add_integration', array( $this, 'automator_integration_load_files' ) );
	}

	/**
	 * Output a nag notice if the user does not have MI installed
	 *
	 * @access public
	 * @return void
	 * @since 1.0.0
	 */
	public function requires_monsterinsights() {
		?>
		<div class="error">
			<p><?php esc_html_e('Please install MonsterInsights Pro to use the MonsterInsights Site Notes Important Events addon.', 'monsterinsights-site-notes-important-events'); ?></p>
		</div>
		<?php
	}

	/**
	 * Output a nag notice if the user does not have MI version installed
	 *
	 * @access public
	 * @return void
	 * @since 1.0.0
	 */
	public function requires_monsterinsights_version() {
		?>
		<div class="error">
			<p><?php esc_html_e('Please install or update MonsterInsights Pro with version 8.26 or newer to use the MonsterInsights Site Notes Important Events addon.', 'monsterinsights-site-notes-important-events'); ?></p>
		</div>
		<?php
	}

	/**
	 * Loads all PHP files into scope.
	 *
	 * @since 1.0.0
	 */
	public function require_files() {
		require_once plugin_dir_path(__FILE__) . 'includes/helpers.php';
		require_once plugin_dir_path(__FILE__) . 'includes/model.php';
		require_once plugin_dir_path(__FILE__) . 'includes/log-events.php';

		$this->model = MonsterInsights_Site_Notes_Important_Events_Model::get_instance();
		MonsterInsights_Site_Notes_Important_Events_Log_Events::get_instance();
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @return object The MonsterInsights_Exceptions object.
	 * @since 1.0.0
	 */
	public static function get_instance() {
		if ( ! self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Load necessary files for Uncanny Automator.
	 */
	public function automator_integration_load_files() {
		// If this class doesn't exist, Uncanny Automator plugin needs to be updated.
		if ( ! class_exists( '\Uncanny_Automator\Integration' ) ) {
			return;
		}

		if ( ! MonsterInsights()->license->license_can( 'agency' ) ) {
			return;
		}

		require_once plugin_dir_path(__FILE__) . 'includes/uncanny-automator/action-create-site-notes.php';
		require_once plugin_dir_path(__FILE__) . 'includes/uncanny-automator/integration.php';

		new MonsterInsights_Site_Notes_IE_Automator_Integration();
		new MonsterInsights_Site_Notes_IE_Action_Create_Site_Notes();
	}
}

// Initialization function
function monsterinsights_site_notes_important_events_addon() {
	return MonsterInsights_Site_Notes_Important_Events::get_instance();
}

monsterinsights_site_notes_important_events_addon();
