<?php

namespace GtmEcommerceWooPro\Lib\EventStrategy\Browser;

use GtmEcommerceWoo\Lib\EventStrategy\AbstractEventStrategy;
use GtmEcommerceWoo\Lib\GaEcommerceEntity\Event;
use GtmEcommerceWooPro\Lib\Service\GtmSnippetService;
use GtmEcommerceWooPro\Lib\Type\EventType;
use WC_Order;

class UserDataStrategy extends AbstractEventStrategy {

	protected $eventName = EventType::USER_DATA;

	public function defineActions() {
		return [
			'wp_head' => [[$this, 'wpHead'], GtmSnippetService::PRIORITY_BEFORE_GTM ],
			'rest_api_init' => [[$this, 'restApiInit'], 10],
		];
	}

	public function restApiInit() {
		register_rest_route(
			'gtm-ecommerce-woo/v1',
			'/get-user-data',
			[
				'methods'             => 'POST',
				'callback'            => [ $this, 'endpointGetUserData' ],
				'permission_callback' => [ $this, 'permissionCallback' ],
			]
		);
	}

	public function permissionCallback() {
		return is_user_logged_in();
	}

	public function endpointGetUserData() {
		$user = wp_get_current_user();
		if ( 0 === $user->ID ) {
			return new \WP_REST_Response( [ 'error' => 'unauthorized' ], 401 );
		}

		$address1 = get_user_meta( $user->ID, 'billing_address_1', true );
		$address2 = get_user_meta( $user->ID, 'billing_address_2', true );

		$userData = [
			'email'        => $user->user_email,
			'phone'        => get_user_meta( $user->ID, 'billing_phone', true ),
			'address'      => [
				'first_name' => get_user_meta( $user->ID, 'billing_first_name', true ),
				'last_name'  => get_user_meta( $user->ID, 'billing_last_name', true ),
				'street'     => trim( join( ' ', array_filter( [ $address1, $address2 ] ) ) ),
				'city'       => get_user_meta( $user->ID, 'billing_city', true ),
				'region'      => get_user_meta( $user->ID, 'billing_state', true ),
				'postal_code'   => get_user_meta( $user->ID, 'billing_postcode', true ),
				'country'    => get_user_meta( $user->ID, 'billing_country', true ),
			],
		];

		if ( empty( $userData['phone'] ) || empty( $userData['address']['street'] ) ) {
			$orders = wc_get_orders(
				[
					'customer_id' => $user->ID,
					'limit'       => 1,
					'orderby'     => 'date',
					'order'       => 'DESC',
				]
			);

			if ( ! empty( $orders ) ) {
				$lastOrder = $orders[0];
				if ( empty( $userData['phone'] ) ) {
					$userData['phone'] = $lastOrder->get_billing_phone();
				}
				if ( empty( $userData['address']['first_name'] ) ) {
					$userData['address']['first_name'] = $lastOrder->get_billing_first_name();
				}
				if ( empty( $userData['address']['last_name'] ) ) {
					$userData['address']['last_name'] = $lastOrder->get_billing_last_name();
				}
				if ( empty( $userData['address']['street'] ) ) {
					$userData['address']['street'] = trim( join( ' ', array_filter( [ $lastOrder->get_billing_address_1(), $lastOrder->get_billing_address_2() ] ) ) );
				}
				if ( empty( $userData['address']['city'] ) ) {
					$userData['address']['city'] = $lastOrder->get_billing_city();
				}
				if ( empty( $userData['address']['region'] ) ) {
					$userData['address']['region'] = $lastOrder->get_billing_state();
				}
				if ( empty( $userData['address']['postal_code'] ) ) {
					$userData['address']['postal_code'] = $lastOrder->get_billing_postcode();
				}
				if ( empty( $userData['address']['country'] ) ) {
					$userData['address']['country'] = $lastOrder->get_billing_country();
				}
			}
		}

		$userData = array_filter(
			$userData,
			function ( $value ) {
				if ( is_array( $value ) ) {
					return ! empty( array_filter( $value ) );
				}
				return ! is_null( $value ) && '' !== $value;
			}
		);

		if ( isset( $userData['address'] ) ) {
			$userData['address'] = array_filter(
				$userData['address'],
				function ( $value ) {
					return ! is_null( $value ) && '' !== $value;
				}
			);
		}

		return new \WP_REST_Response( $userData, 200 );
	}

	public function wpHead()
	{
		$this->orderUserData();
		$this->generalUserData();
	}

	private function generalUserData()
	{
		if (!is_user_logged_in()) {
			return;
		}

		echo <<<EOD
<script>
var dataLayer = dataLayer || [];
(function(dataLayer, sessionStorage){
	const STORAGE_KEY = 'gtm_ecommerce_woo_user_data';
	const storedData = sessionStorage.getItem(STORAGE_KEY);

	if (storedData) {
		try {
			const userData = JSON.parse(storedData);
			if (userData && !userData.error) {
				const event = {
					'event': 'user_data',
					'email': userData.email,
					'phone_number': userData.phone,
					'address': userData.address,
					'user_data': userData
				};
				dataLayer.push(event);
			}
		} catch (e) {
			sessionStorage.removeItem(STORAGE_KEY);
		}
	}
})(dataLayer, window.sessionStorage);
</script>
EOD;

		$getUserDataEndpointUrl = sprintf('%s%s', get_rest_url(), 'gtm-ecommerce-woo/v1/get-user-data');
		$nonce = wp_create_nonce('wp_rest');

		$this->wcOutput->script(
			<<<EOD
(function($, dataLayer, sessionStorage){
	const STORAGE_KEY = 'gtm_ecommerce_woo_user_data';
	const storedData = sessionStorage.getItem(STORAGE_KEY);

	if (storedData) {
		return;
	}

	$.ajax({
		type: 'POST',
		url: '{$getUserDataEndpointUrl}',
		beforeSend: function (xhr) {
			xhr.setRequestHeader('X-WP-Nonce', '{$nonce}');
		},
		success: function(response) {
			if (response && response.email) {
				sessionStorage.setItem(STORAGE_KEY, JSON.stringify(response));
				const event = {
					'event': 'user_data',
					'email': response.email,
					'phone_number': response.phone,
					'address': response.address,
					'user_data': response
				};
				dataLayer.push(event);
			}
		}
	});
})(jQuery, dataLayer, window.sessionStorage);
EOD
		);
	}

	private function orderUserData()
	{
		if (false === is_wc_endpoint_url('order-received')) {
			return;
		}

		if (false === isset($_GET['key'])) {
			return;
		}

		$orderKey = sanitize_text_field($_GET['key']);
		$orderId = wc_get_order_id_by_order_key($orderKey);

		if (null === $orderId) {
			return;
		}

		/** WC_Order @var WC_Order|false $order*/
		$order = wc_get_order($orderId);

		if (false === $order) {
			return;
		}

		$address = [
			'first_name'  => $order->get_billing_first_name(),
			'last_name'   => $order->get_billing_last_name(),
			'street'      => join( ' ', array( $order->get_billing_address_1(), $order->get_billing_address_2() ) ),
			'postal_code' => $order->get_billing_postcode(),
			'country'     => $order->get_billing_country(),
			'region'      => $order->get_billing_state(),
			'city'        => $order->get_billing_city(),
		];

		$userData = [
			'email' => $order->get_billing_email(),
			'phone' => $order->get_billing_phone(),
			'address' => $address,
		];

		$event = ( new Event(EventType::USER_DATA) )
			->setExtraProperty('email', $userData['email'])
			->setExtraProperty('phone_number', $userData['phone'])
			->setExtraProperty('address', $userData['address'])
			->setExtraProperty('user_data', array_filter($userData, function($value) {
				return !is_null($value) && '' !== $value;
			}));
		;

		$serializedEvent = json_encode($event);

		echo sprintf(
			"<script>var dataLayer = dataLayer || [];dataLayer.push(%s);</script>\n",
			filter_var($serializedEvent, FILTER_FLAG_STRIP_BACKTICK)
		);
	}

}
