<?php
/**
 * Integration for LeanDash Courses
 *
 * @since 1.0.0
 * @package MemberDash
 */

/**
 * Integration for LeanDash Courses
 *
 * @since 1.0.0
 */
class MS_Addon_Learndash extends MS_Addon {

	/**
	 * The Add-on ID
	 *
	 * @since 1.0.0
	 */
	const ID = 'addon_learndash';

	/**
	 * WPML Translation context.
	 *
	 * @since 1.0.0
	 */
	const CONTEXT = 'Membership';

	/**
	 * Value from WPML: The site default language.
	 *
	 * @var string
	 */
	protected $default_lang = '';

	/**
	 * Value from WPML: Currently selected language.
	 *
	 * @var string
	 */
	protected $current_lang = '';

	/**
	 * Ajax action identifier used in the Membership settings.
	 *
	 * @since 1.0.0
	 */
	const AJAX_ACTION_SAVE_COURSES = 'addon_learndash_save_courses';

	const AJAX_ACTION_SAVE_SETTINGS = 'addon_learndash_save_settings';

	const ENROLL_TO_MEMBERSHIP = 'enroll_to_membership';

	const ENROLL_TO_COURSE = 'enroll_to_course';

	/**
	 * The enroll type.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $enroll_type = self::ENROLL_TO_MEMBERSHIP;


	/**
	 * Checks if the current Add-on is enabled.
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public static function is_active() {
		if (
			! self::learndash_active()
			&& MS_Model_Addon::is_enabled( self::ID )
		) {
			return false;
		}

		return MS_Model_Addon::is_enabled( self::ID );
	}

	/**
	 * Returns the Add-on ID (self::ID).
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function get_id() {
		return self::ID;
	}

	/**
	 * Initializes the Add-on. Always executed.
	 *
	 * @since 1.0.0
	 */
	public function init() {
		static $Init_Done = false;

		if ( $Init_Done ) {
			return;
		}

		$Init_Done = true;

		if ( self::is_active() ) {

			$this->check_requirements();

			// When user purchase membership it will enroll to course.
			$this->init_enroll_to_course();

			// When user purchase course it will enroll to membership.
			$this->init_enroll_to_membership();

			$this->add_ajax_action(
				self::AJAX_ACTION_SAVE_SETTINGS,
				'ms_learndash_ajax_save_settings'
			);

		} else {
			$this->add_action( 'ms_model_addon_enable', 'enable_addon' );
		}
	}

	/**
	 * Initializes enroll to course integration.
	 *
	 * @since 1.1.4
	 *
	 * @return void
	 */
	protected function init_enroll_to_course(): void {
		$this->add_filter(
			'ms_controller_membership_tabs',
			'ms_learndash_tab',
			5, // Make it displays before 3rd-party add-ons.
			3
		);

		$this->add_action(
			'ms_view_membership_edit_render_callback',
			'ms_learndash_tab_view',
			10,
			3
		);

		// Ajax handler that saves a courses ids.
		$this->add_ajax_action(
			self::AJAX_ACTION_SAVE_COURSES,
			'ms_learndash_ajax_save_courses'
		);

		foreach ( $this->get_subscription_valid_statuses() as $status ) {
			$this->add_action(
				'ms_subscription_status-' . $status,
				'enroll_to_course',
				10
			);
		}

		$this->add_action(
			'ms_model_relationship_add_payment_after',
			'enroll_to_course',
			10
		);
	}

	/**
	 * Initializes enroll to membership integration.
	 *
	 * @since 1.1.4
	 *
	 * @return void
	 */
	protected function init_enroll_to_membership(): void {
		$this->add_action(
			'add_meta_boxes',
			'ms_register_meta_box'
		);

		$this->add_filter(
			'learndash_header_tab_menu',
			'ms_learndash_menu',
			5, // Hook early to display before 3rd-party extensions.
			3
		);

		$this->add_action(
			'save_post_' . self::get_learndash_course_post_type(),
			'ms_learndash_membership_save'
		);

		$this->add_action(
			'learndash_update_course_access',
			'ms_enroll_to_membership',
			10,
			4
		);
	}

	/**
	 * Returns a list of valid subscription statuses.
	 *
	 * @since 1.1.1
	 *
	 * @return string[] List of valid subscription statuses.
	 */
	public function get_subscription_valid_statuses(): array {
		return array(
			MS_Model_Relationship::STATUS_ACTIVE,
			MS_Model_Relationship::STATUS_TRIAL,
		);
	}

	/**
	 * Enrolls a user to a LD course when a user purchases a membership.
	 *
	 * @since 1.1.1
	 *
	 * @param MS_Model_Relationship $subscription The relationship object.
	 *
	 * @return void
	 */
	public function enroll_to_course( $subscription ): void {
		if ( ! function_exists( 'ld_update_course_access' ) ) {
			return;
		}

		// Validate subscription status before proceeding.
		$statuses = $this->get_subscription_valid_statuses();
		if ( ! in_array( $subscription->get_status(), $statuses, true ) ) {
			return;
		}

		$membership = $subscription->get_membership();

		// Stop if membership is not valid.
		if ( ! $membership->is_valid() ) {
			return;
		}

		// Fetch courses for the membership and stop if none.
		$courses = self::ms_get_learndash_membership_courses( $membership );
		if ( empty( $courses ) ) {
			return;
		}

		$user_id = $subscription->get_user_id();

		foreach ( $courses as $course ) {
			ld_update_course_access( $user_id, $course );
		}
	}

	/**
	 * Enrolls a user to a course when a user purchases a membership.
	 *
	 * Was used only to hook into `ms_model_relationship_create_ms_relationship_after`.
	 *
	 * @deprecated 1.1.1 This method is not needed anymore. All logic was moved to `enroll_to_course` method.
	 *
	 * @since 1.0.0
	 *
	 * @param MS_Model_Relationship|null $subscription The relationship object.
	 * @param int                        $membership_id The membership ID.
	 * @param int                        $user_id The user ID.
	 * @param int                        $gateway_id The gateway ID.
	 * @param int                        $move_from_id The move from ID.
	 *
	 * @return void
	 */
	public function ms_learndash_enroll_to_course( $subscription, $membership_id, $user_id, $gateway_id, $move_from_id ) {
		_deprecated_function( __METHOD__, '1.1.1', 'enroll_to_course' );

		if ( ! empty( $subscription ) ) {

			$membership = MS_Factory::load( 'MS_Model_Membership', $membership_id );

			if ( $membership->is_valid() ) {
				$courses = self::ms_get_learndash_membership_courses( $membership );

				if ( ! empty( $courses ) ) {
					foreach ( $courses as $course ) {
						ld_update_course_access( $user_id, $course );
					}
				}
			}
		}

	}

	public static function ms_get_learndash_membership_courses( $membership ) {

		return $membership->get_custom_data( 'course_list' );

	}



	public function ms_learndash_ajax_save_settings() {
		$res    = MS_Helper_Membership::MEMBERSHIP_MSG_NOT_UPDATED;
		$fields = array( 'field', 'membership_id' );

		if ( self::validate_required( $fields ) && $this->verify_nonce() ) {
			$values = $_POST['values'];
			$res    = MS_Helper_Membership::MEMBERSHIP_MSG_UPDATED;
		}

		echo $res; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * Ajax handler that saves a membership attribute.
	 *
	 * @since 1.0.0
	 */
	public function ms_learndash_ajax_save_courses() {
		$res    = MS_Helper_Membership::MEMBERSHIP_MSG_NOT_UPDATED;
		$fields = array( 'field', 'membership_id' );

		if ( self::validate_required( $fields ) && $this->verify_nonce() ) {
			$id         = intval( $_POST['membership_id'] );
			$values     = isset( $_POST['values'] ) ? $_POST['values'] : array();
			$membership = MS_Factory::load( 'MS_Model_Membership', $id );

			if ( $membership->is_valid() ) {

				$membership->set_custom_data(
					'course_list',
					$values
				);
				$membership->save();
			}
			$res = MS_Helper_Membership::MEMBERSHIP_MSG_UPDATED;
		}

		echo $res; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * Returns a list of LearnDash courses.
	 *
	 * @since 1.0.0
	 *
	 * @return array<int|string,string>
	 */
	public static function ms_list_of_learndash_courses() {

		$args = [
			'numberposts' => -1,
			'post_type'   => self::get_learndash_course_post_type(),
		];

		$courses = get_posts( $args );
		if ( ! empty( $courses ) ) {
			foreach ( $courses as $course ) {
				$course_list[ $course->ID ] = $course->post_title;
			}
		} else {
			$course_list[''] = __( 'No Courses Available', 'memberdash' );
		}

		return $course_list;
	}

	public function ms_learndash_tab_view( $callback, $tab, $data ) {

		if ( $tab == self::ID ) {
			$view       = MS_Factory::load( 'MS_Addon_Learndash_View_List' );
			$view->data = $data;
			$callback   = array( $view, 'render_tab' );
		}
			return $callback;
	}

	public function ms_learndash_tab( $tabs ) {
		$tabs[ self::ID ] = array(
			'title' => __( 'LearnDash', 'memberdash' ),
		);

		return $tabs;
	}

	/**
	 * Adds a new tab to the LearnDash course edit page.
	 *
	 * @since 1.0.0
	 *
	 * @param array<int,array<string,array<int,string>|string>> $header_data      The header data.
	 * @param string                                            $menu_tab_key     The menu tab key.
	 * @param string                                            $screen_post_type The screen post type.
	 *
	 * @return array<int,array<string,array<int,string>|string>>
	 */
	public function ms_learndash_menu( $header_data, $menu_tab_key, $screen_post_type ) {
		if ( $screen_post_type === 'sfwd-courses' ) {
			$header_data = array_merge(
				$header_data,
				[
					[
						'id'                  => 'ms_learndash_course_membership',
						'name'                => esc_html__( 'Membership', 'memberdash' ),
						'metaboxes'           => [ 'learndash-course-membership' ],
						'showDocumentSidebar' => 'false',
					],
				]
			);
		}

		return $header_data;
	}

	/**
	 * Registers meta box to display membership dropdown in LD course edit page.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function ms_register_meta_box() {
		add_meta_box(
			'learndash-course-membership',
			esc_html__( 'Membership', 'memberdash' ),
			[ $this, 'ms_meta_box_callback' ],
			self::get_learndash_course_post_type(),
			'normal',
			'high'
		);
	}

	/**
	 * Register meta box to display membership dropdown in LD course edit page.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Post $post The post object.
	 *
	 * @return void
	 */
	public function ms_meta_box_callback( $post ) {
		$memberships = MS_Model_Membership::get_memberships();
		$value       = get_post_meta( $post->ID, 'ms_course_membership', true );

		if ( ! empty( $memberships ) ) { ?>
				<p>
					<label for="ms_course_membership">
						<?php esc_html_e( 'Membership', 'memberdash' ); ?>
					</label>
					<select id="ms_course_membership" name="ms_course_membership" >
						<option value="" <?php echo selected( $value, '' ); ?>>
							<?php esc_html_e( 'Select Membership', 'memberdash' ); ?>
						</option>
						<?php
						foreach ( $memberships as $membership ) {
							?>
							<option value="<?php echo esc_attr( $membership->id ); ?>" <?php echo selected( $value, $membership->id ); ?>>
								<?php echo esc_html( $membership->name ); ?>
							</option>
							<?php
						}
						?>
					</select>
				</p>

				<p><small><?php esc_html_e( '* User will enroll to selected membership', 'memberdash' ); ?></small></p>
				<?php

		} else {
			?>
				<p><?php esc_html_e( 'Please add Membership', 'memberdash' ); ?></p>
			<?php
		}
	}

	public function ms_learndash_membership_save( $post_id ) {
		if ( ! isset( $_POST['ms_course_membership'] ) ) {
			return $post_id;
		}
		$membership_id = $_POST['ms_course_membership'];
		update_post_meta( $post_id, 'ms_course_membership', $membership_id );
	}


	/**
	 * Enrolls a user to a membership when a user purchases a course.
	 *
	 * @since 1.0.0
	 *
	 * @param int         $user_id            The user ID.
	 * @param int         $course_id          The course ID.
	 * @param string|null $course_access_list A comma-separated list of user IDs.
	 * @param bool        $remove             True if the user is removed from the course.
	 *
	 * @return void
	 */
	public function ms_enroll_to_membership( $user_id, $course_id, $course_access_list, $remove ) {
		// Skip if user is removed from course.
		if ( $remove ) {
			return;
		}

		// Get the membership ID from the course.
		$membership_id = get_post_meta( $course_id, 'ms_course_membership', true );

		if ( empty( $membership_id ) ) {
			return;
		}

		$member = MS_Factory::load( 'MS_Model_Member', $user_id );

		// Add the membership and use the admin payment gateway.
		$subscription = $member->add_membership( intval( $membership_id ), 'admin' );

		if ( empty( $subscription ) ) {
			return;
		}

		// Activate membership instantly.
		$invoice = $subscription->get_current_invoice();
		$invoice->pay_it();

		$subscription->save();
	}

	/**
	 * Chooses how MD and LD integrate
	 *
	 * @deprecated 1.1.4 There are no more options in this add-on. This method is obsolete now.
	 *
	 * @since 1.0.0
	 */
	public static function enroll_types() {
		_deprecated_function( __METHOD__, '1.1.4' );

		return array(
			self::ENROLL_TO_COURSE     => __( 'When a user subscribes to a Membership they will be automatically enrolled into the selected Course', 'memberdash' ),
			self::ENROLL_TO_MEMBERSHIP => __( 'When a user enrolls in a Course they will be automatically subscribed to the selected Membership', 'memberdash' ),
		);

	}

	/**
	 * Registers the Add-On.
	 *
	 * @since 1.0.0
	 *
	 * @param  array $list The Add-Ons list.
	 *
	 * @return array The updated Add-Ons list.
	 */
	public function register( $list ) {
		$list[ self::ID ] = (object) array(
			'name'        => __( 'LearnDash Integration', 'memberdash' ),
			'description' => __( 'This module integrates MemberDash with LearnDash', 'memberdash' ),
		);

		if ( ! self::learndash_active() ) {
			$list[ self::ID ]->description .= sprintf(
				'<br /><b>%s</b>',
				__( 'Activate LearnDash to use this Modules', 'memberdash' )
			);
		}

		return $list;
	}

	/**
	 * Checks if the LearnDash plugin is active.
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public static function learndash_active() {
		return defined( 'LEARNDASH_VERSION' );
	}

	/**
	 * Function is triggered every time an add-on is enabled.
	 *
	 * We flush the Factory Cache when the LearnDash Add-on is enabled so all strings
	 * are properly registered for translation.
	 *
	 * @since 1.0.0
	 *
	 * @param string $addon The Add-on ID.
	 */
	public function enable_addon( $addon ) {
		if ( self::ID === $addon ) {
			MS_Factory::clear();
		}
	}

	/**
	 * Check if LearnDash is up to date and required modules are active.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	protected function check_requirements(): void {
		if ( ! self::learndash_active() ) {
			mslib3()->ui->admin_message(
				sprintf(
					'<b>%s</b><br>%s',
					__( 'LearnDash not active!', 'memberdash' ),
					__( 'Heads up: Activate the LearnDash plugin to enable the LearnDash integration.', 'memberdash' )
				),
				'err'
			);
			return;
		}

		if ( version_compare( LEARNDASH_VERSION, '4.7.0', 'lt' ) ) {
			mslib3()->ui->admin_message(
				sprintf(
					'<b>%s</b><br>%s',
					__( 'Great, you\'re using LearnDash!', 'memberdash' ),
					__( 'Heads up: Your version of LearnDash is outdated. Please update LearnDash to version <b>4.7.0 or higher</b> to enable the LearnDash integration.', 'memberdash' )
				),
				'err'
			);
			return;
		}
	}

	/**
	 * Returns the LearnDash course post type slug.
	 *
	 * @since 1.1.4
	 *
	 * @return string
	 */
	protected static function get_learndash_course_post_type(): string {
		if ( ! class_exists( 'LDLMS_Post_Types' ) ) {
			return '';
		}

		$slug = learndash_get_post_type_slug( LDLMS_Post_Types::COURSE );

		return MS_Helper_Cast::to_string( $slug );
	}
}
