<?php
/**
 * Banners Controller.
 *
 * @since 1.4.5
 *
 * @package MemberDash
 */

/**
 * Banners Controller class.
 *
 * @since 1.4.5
 */
class MS_Controller_Banners extends MS_Controller {
	/**
	 * Initialize the sale banners.
	 *
	 * Inherited from MS_Controller.
	 *
	 * @since 1.4.5
	 *
	 * @return void
	 */
	public function admin_init() {
		$this->add_action(
			'admin_notices',
			'render_banners'
		);
	}

	/**
	 * Returns all sale banners.
	 *
	 * @since 1.4.5
	 *
	 * @return array<string,array<string,string>>
	 *
	 * @phpstan-return array<string,array{
	 *   id: string,
	 *   name: string,
	 *   description: string,
	 *   link: string,
	 *   query_string: string,
	 *   start_date: string,
	 *   end_date: string
	 * }>
	 */
	protected function get_banners(): array {
		return [
			'bf2024' => [
				'id'           => 'bf2024',
				'name'         => __( 'MemberDash Black Friday Sale', 'memberdash' ),
				'description'  => __( 'MemberDash Black Friday Sale: Save 40% on MemberDash. Elevate your memberships before 12/3/2024.', 'memberdash' ),
				'link'         => 'https://memberdashwp.com/black-friday/',
				'query_string' => 'bf2024',
				'start_date'   => '20241127',
				'end_date'     => '20241205',
			],
		];
	}

	/**
	 * Returns a specific sale banner.
	 *
	 * @since 1.4.5
	 *
	 * @param string $banner_id The banner ID.
	 *
	 * @return array<string,string>
	 *
	 * @phpstan-return array{
	 *    id: string,
	 *    name: string,
	 *    description: string,
	 *    link: string,
	 *    query_string: string,
	 *    start_date: string,
	 *    end_date: string
	 * }
	 */
	protected function get_banner( string $banner_id ): array {
		$banners = $this->get_banners();

		if ( ! isset( $banners[ $banner_id ] ) ) {
			return [
				'id'           => '',
				'name'         => '',
				'description'  => '',
				'link'         => '',
				'query_string' => '',
				'start_date'   => '',
				'end_date'     => '',
			];
		}

		return $banners[ $banner_id ];
	}

	/**
	 * Checks if the current page is a MemberDash settings page.
	 *
	 * @since 1.4.5
	 *
	 * @return bool
	 */
	protected function is_settings_page(): bool {
		// Stop if isn't the admin area.
		if ( ! function_exists( 'get_current_screen' ) ) {
			return false;
		}

		$screen = get_current_screen();

		if ( ! $screen instanceof WP_Screen ) {
			return false;
		}

		$pages = MS_Plugin::instance()->get_controller()->ms_set_notice_pages();

		return in_array( $screen->id, $pages, true );
	}

	/**
	 * Checks if the banner can be displayed.
	 *
	 * @since 1.4.5
	 *
	 * @param string $banner_id The banner ID.
	 *
	 * @return bool
	 */
	protected function can_display( string $banner_id ): bool {
		$banner = $this->get_banner( $banner_id );

		// Stop if the banner is empty.
		if ( empty( $banner['id'] ) ) {
			return false;
		}

		// Check if the query string is present and valid.
		if ( ! empty( $banner['query_string'] ) ) {
			$query_string = MS_Helper_Cast::to_string(
				self::get_request_field( $banner['query_string'], '0', 'GET' )
			);

			if ( '1' === $query_string ) {
				return true;
			}
		}

		// Check if the banner has been dismissed.
		$notices_dismissed = get_user_meta( get_current_user_id(), 'ms_admin_notices_dismissed', true );
		$notices_dismissed = mslib3()->array->get(
			$notices_dismissed
		);

		if (
			isset( $notices_dismissed[ $banner['id'] ] )
			&& $notices_dismissed[ $banner['id'] ] === true
		) {
			return false;
		}

		// Check if the banner is within the start and end date.
		$start = MS_Helper_Cast::to_int( $banner['start_date'] );
		$end   = MS_Helper_Cast::to_int( $banner['end_date'] );
		$now   = MS_Helper_Cast::to_int( gmdate( 'Ymd' ) );

		return $now >= $start && $now <= $end;
	}

	/**
	 * Render the banners.
	 *
	 * @since 1.4.5
	 *
	 * @return void
	 */
	public function render_banners(): void {
		// Stop if it isn't the admin area.
		if ( ! $this->is_settings_page() ) {
			return;
		}

		foreach ( $this->get_banners() as $banner_id => $banner ) {
			if ( ! $this->can_display( $banner_id ) ) {
				continue;
			}

			// Render the banner.
			$view = MS_Factory::create( 'MS_View_Banner' );
			$view->set_data( $banner );
			$view->render();
		}
	}
}
