<?php
/**
 * Gateway: Paypal Express - API Client
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

use StellarWP\Memberdash\StellarWP\Arrays\Arr;

/**
 * PayPal Express gateway API Client class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress_Api_Client extends MS_Gateway_Paypalexpress_Api {
	/**
	 * Returns the headers for a PayPal request.
	 *
	 * @since 1.5.0
	 *
	 * @param string $request_id The request ID to use.
	 *
	 * @return array<string,string>
	 */
	protected function get_request_headers( string $request_id ): array {
		$gateway = $this->get_gateway();

		return [
			'PayPal-Partner-Attribution-Id' => $gateway->get_partner_attribution_id(),
			'PayPal-Request-Id'             => md5( $request_id . $gateway->get_partner_attribution_id() ),
			'Prefer'                        => 'return=representation',
		];
	}

	/**
	 * Get the PayPal Partner JS URL.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_partner_js_url(): string {
		return 'https://paypal.com/webapps/merchantboarding/js/lib/lightbox/partner.js';
	}

	/**
	 * Fetches the JS SDK url.
	 *
	 * We use something like: https://www.paypal.com/sdk/js?client-id=sb&locale=en_US&components=buttons
	 *
	 * @link https://developer.paypal.com/docs/checkout/reference/customize-sdk/#query-parameters
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,string> $query_args Which query args will be added.
	 *
	 * @return string
	 */
	public function get_js_sdk_url( array $query_args = [] ): string {
		$settings = MS_Plugin::get_settings();

		/**
		 * Filters the JS SDK args.
		 *
		 * @since 1.5.0
		 *
		 * @param array<string,string> $args The args to filter.
		 *
		 * @return array<string,string>
		 */
		$args = apply_filters(
			'ms_gateway_paypalexpress_js_sdk_args',
			array_merge(
				[
					'client-id'       => '',
					'merchant-id'     => '',
					'components'      => 'buttons,card-fields',
					'intent'          => 'capture',
					'disable-funding' => 'credit',
					'currency'        => $settings->get_currency(),
				],
				$query_args
			),
		);

		return add_query_arg( $args, 'https://www.paypal.com/sdk/js' );
	}

	/**
	 * Fetches an access token from the PayPal API using the authorization code.
	 *
	 * @since 1.5.0
	 *
	 * @param string $shared_id     The shared ID to use.
	 * @param string $auth_code     The authorization code to use.
	 * @param string $code_verifier The verifier code. Defaults to the transient hash.
	 * @param bool   $is_sandbox    Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function get_access_token_from_authorization_code(
		string $shared_id,
		string $auth_code,
		string $code_verifier = '',
		bool $is_sandbox = false
	): array {
		$auth       = base64_encode( $shared_id ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- We need to encode the shared ID.
		$query_args = [];

		$code_verifier = ! empty( $code_verifier )
			? $code_verifier
			: MS_Gateway_Paypalexpress_Api_Whodat::get_transient_hash();

		$args = [
			'headers' => [
				'Authorization' => sprintf( 'Basic %s', $auth ),
				'Content-Type'  => 'application/x-www-form-urlencoded',
			],
			'body'    => [
				'grant_type'    => 'authorization_code',
				'code'          => $auth_code,
				'code_verifier' => $code_verifier,
			],
		];

		$response = $this->client_post( 'v1/oauth2/token', $query_args, $args, $is_sandbox );

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Retrieves a Client Token from the stored Access Token.
	 *
	 * @link https://developer.paypal.com/docs/business/checkout/advanced-card-payments/
	 *
	 * @since 1.5.0
	 *
	 * @param bool $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed> The client token details response or empty array if there was a problem.
	 */
	public function get_client_token( bool $is_sandbox = false ): array {
		$stored_token = Arr::wrap(
			get_option( 'ms_paypal_express_client_token', [] )
		);
		$valid_until  = MS_Helper_Cast::to_int(
			Arr::get( $stored_token, 'valid_until', 0 )
		);

		// If the token is still valid, return it.
		if (
			! empty( $stored_token )
			&& $valid_until > time()
		) {
			return $stored_token;
		}

		$token = $this->client_post(
			'v1/identity/generate-token',
			[],
			[],
			$is_sandbox
		);

		if ( is_wp_error( $token ) ) {
			return [];
		}

		$expires_in = MS_Helper_Cast::to_int(
			Arr::get( $token, 'expires_in', 0 )
		);

		/*
		 * The token is valid for 1 hour, but we store it until 5 minutes before
		 * it expires to avoid any issues.
		 *
		 * The checkout form is refreshed in case the token is expired, but we
		 * want to avoid any issues with the token being invalid during the
		 * checkout process or the user taking too long to complete the payment.
		 */
		$token['valid_until'] = ( time() + $expires_in ) - 300;

		update_option( 'ms_paypal_express_client_token', $token );

		return $token;
	}

	/**
	 * Deletes the stored PayPal client token.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function delete_client_token(): bool {
		return delete_option( 'ms_paypal_express_client_token' );
	}

	/**
	 * Creates an order in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/orders/v2/#orders_create
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data The data to use.
	 * @param bool                $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function create_order( array $data, bool $is_sandbox = false ): array {
		$body = [
			'intent'         => 'CAPTURE',
			'purchase_units' => [],
			'payment_source' => [
				'paypal' => [
					'experience_context' => [
						'shipping_preference' => 'NO_SHIPPING',
						'user_action'         => 'PAY_NOW',
						'return_url'          => $data['return_url'],
						'cancel_url'          => $data['cancel_url'],
					],
				],
			],
		];

		// Remove the payment source if we are using card fields.
		if ( ! empty( $data['use_card_fields'] ) ) {
			unset( $body['payment_source']['paypal'] );

			$body['payment_source']['card'] = [
				'experience_context' => [
					'shipping_preference' => 'NO_SHIPPING',
					'return_url'          => $data['return_url'],
					'cancel_url'          => $data['cancel_url'],
				],
			];
		}

		$purchase_units = [
			'reference_id'        => $data['reference_id'],
			'description'         => $data['description'],
			'invoice_id'          => $data['invoice_id'],
			'amount'              => [
				'currency_code' => $data['currency_code'],
				'value'         => $data['amount'],
				'breakdown'     => [
					'item_total' => [
						'currency_code' => $data['currency_code'],
						'value'         => $data['amount'],
					],
				],
			],
			'payee'               => [
				'merchant_id' => $data['merchant_id'],
			],
			'payer'               => [
				'name'          => [
					'given_name' => $data['first_name'],
					'surname'    => $data['last_name'],
				],
				'email_address' => $data['email'],
			],
			'payment_instruction' => [
				'disbursement_mode' => 'INSTANT',
			],
			'items'               => $data['items'],
		];

		$body['purchase_units'][] = $purchase_units;

		$args = [
			'headers' => $this->get_request_headers( MS_Helper_Cast::to_string( $data['reference_id'] ) ),
			'body'    => $body,
		];

		$response = $this->client_post( 'v2/checkout/orders', [], $args, $is_sandbox );

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Fetches an order from the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string $order_id   The PayPal order ID.
	 * @param bool   $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function get_order( string $order_id, bool $is_sandbox = false ): array {
		$args = [
			'headers' => $this->get_request_headers( $order_id ),
			'body'    => [],
		];

		$response = $this->client_get(
			'v2/checkout/orders/' . rawurlencode( $order_id ),
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Captures an order in the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string $order_id   The PayPal order ID.
	 * @param string $payer_id   The PayPal payer ID.
	 * @param bool   $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function capture_order(
		string $order_id,
		string $payer_id = '',
		bool $is_sandbox = false
	): array {
		$body = [];

		if ( ! empty( $payer_id ) ) {
			$body['payer_id'] = $payer_id;
		}

		$args = [
			'headers' => $this->get_request_headers( $order_id . $payer_id ),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v2/checkout/orders/' . rawurlencode( $order_id ) . '/capture',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Creates a product in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/catalog-products/v1/#products_create
	 *
	 * @since 1.5.0
	 *
	 * @param array{name:string,description:string} $data The data to use.
	 * @param bool                                  $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function create_product( array $data, bool $is_sandbox = false ): array {
		$body = [
			'name'        => MS_Gateway_Paypalexpress_Helpers::trim_text( $data['name'] ),
			'description' => MS_Gateway_Paypalexpress_Helpers::trim_text( $data['description'], 256 ),
			'type'        => 'DIGITAL',
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					$this->request_data_to_string( $data )
				)
			),
			'body'    => array_filter( $body ),
		];

		$response = $this->client_post(
			'v1/catalogs/products',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Updates a product description in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/catalog-products/v1/#products_patch
	 *
	 * @since 1.5.0
	 *
	 * @param array{product_id:string,description:string} $data PayPal Product ID and membership description.
	 * @param bool                                        $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function update_product_description( array $data, bool $is_sandbox = false ): array {
		$body = [
			'op'    => 'replace',
			'path'  => '/description',
			'value' => $data['description'],
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					$this->request_data_to_string( $data )
				)
			),
			'body'    => $body,
		];

		$response = $this->client_patch(
			'v1/catalogs/products/' . $data['product_id'],
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Creates a plan in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#plans_create
	 *
	 * @since 1.5.0
	 *
	 * @phpstan-param array{
	 *   name: string,
	 *   description: string,
	 *   product_id: string,
	 *   billing_cycles: array<array<mixed>>,
	 *   setup_fee: string,
	 *   currency_code: string
	 * } $data The data to use.
	 *
	 * @param array<string,mixed> $data The data to use.
	 * @param bool                $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function create_plan( array $data, bool $is_sandbox = false ): array {
		$body = [
			'product_id'          => $data['product_id'],
			'name'                => MS_Gateway_Paypalexpress_Helpers::trim_text( $data['name'] ),
			'description'         => MS_Gateway_Paypalexpress_Helpers::trim_text( $data['description'] ),
			'billing_cycles'      => $data['billing_cycles'],
			'payment_preferences' => [
				'auto_bill_outstanding' => true,
			],
		];

		if ( ! empty( $data['setup_fee'] ) ) {
			$body['payment_preferences'] = [
				'setup_fee'                 => [
					'value'         => $data['setup_fee'],
					'currency_code' => $data['currency_code'],
				],
				'payment_failure_threshold' => 3,
			];
		}

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					$this->request_data_to_string( $data )
				)
			),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/billing/plans',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Updates a plan in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#plans_patch
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data The data to use.
	 * @param bool                $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function update_plan( array $data = [], bool $is_sandbox = false ): array {
		$body = [];

		foreach ( $data as $key => $value ) {
			if ( 'plan_id' === $key ) {
				continue;
			}

			$body[] = [
				'op'    => 'replace',
				'path'  => '/' . str_replace( '.', '/', $key ),
				'value' => $value,
			];
		}

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					$this->request_data_to_string( $data )
				)
			),
			'body'    => $body,
		];

		$response = $this->client_patch(
			'v1/billing/plans/' . $data['plan_id'],
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Updates a plan's pricing in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#plans_update-pricing-schemes
	 *
	 * @since 1.5.0
	 *
	 * @param string              $plan_id The PayPal plan ID.
	 * @param array<string,mixed> $pricing_data The pricing data to update.
	 * @param bool                $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function update_plan_pricing( string $plan_id, array $pricing_data, bool $is_sandbox = false ): array {
		$body = [
			'pricing_schemes' => [
				[
					'billing_cycle_sequence' => 1,
					'pricing_scheme'         => [
						'fixed_price' => [
							'value'         => $pricing_data['amount'],
							'currency_code' => $pricing_data['currency_code'],
						],
					],
				],
			],
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					$plan_id . $this->request_data_to_string( $pricing_data )
				)
			),
			'body'    => $body,
		];

		$response = $this->client_patch(
			'v1/billing/plans/' . $plan_id,
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Creates a subscription in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#subscriptions_create
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data The data to use.
	 * @param bool                $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function create_subscription( array $data, bool $is_sandbox = false ): array {
		$body = [
			'plan_id'             => $data['plan_id'],
			'quantity'            => 1,
			'custom_id'           => $data['custom_id'],
			'subscriber'          => [
				'name'          => [
					'given_name' => $data['first_name'],
					'surname'    => $data['last_name'],
				],
				'email_address' => $data['email'],
			],
			'application_context' => [
				'brand_name'          => MS_Gateway_Paypalexpress_Helpers::trim_text( get_bloginfo( 'name' ) ),
				'locale'              => str_replace( '_', '-', get_user_locale() ),
				'shipping_preference' => 'NO_SHIPPING',
				'user_action'         => 'SUBSCRIBE_NOW',
				'return_url'          => $data['return_url'],
				'cancel_url'          => $data['cancel_url'],
			],
			'payment_method'      => [
				'payee_preferred' => 'IMMEDIATE_PAYMENT_REQUIRED',
			],
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					$this->request_data_to_string( $data )
				)
			),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/billing/subscriptions',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Retrieves a subscription from the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#subscriptions_get
	 *
	 * @since 1.5.0
	 *
	 * @param string $subscription_id The PayPal subscription ID.
	 * @param bool   $is_sandbox      Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function get_subscription( string $subscription_id, bool $is_sandbox = false ): array {
		$args = [
			'headers' => $this->get_request_headers( $subscription_id ),
			'body'    => [],
		];

		$response = $this->client_get(
			'v1/billing/subscriptions/' . rawurlencode( $subscription_id ),
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Suspends a subscription in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#subscriptions_suspend
	 *
	 * @since 1.5.0
	 *
	 * @param string $subscription_id The PayPal subscription ID.
	 * @param bool   $is_sandbox      Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return bool
	 */
	public function suspend_subscription( string $subscription_id, bool $is_sandbox = false ): bool {
		$body = [
			'reason' => MS_Gateway_Paypalexpress_Helpers::trim_text( __( 'Customer Request', 'memberdash' ), 128 ),
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5( $subscription_id )
			),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/billing/subscriptions/' . $subscription_id . '/suspend',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Cancels a subscription in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#subscriptions_cancel
	 *
	 * @since 1.5.0
	 *
	 * @param string $subscription_id The PayPal subscription ID.
	 * @param bool   $is_sandbox      Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return bool
	 */
	public function cancel_subscription( string $subscription_id, bool $is_sandbox = false ): bool {
		$body = [
			'reason' => MS_Gateway_Paypalexpress_Helpers::trim_text( __( 'Customer Request', 'memberdash' ), 128 ),
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5( $subscription_id )
			),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/billing/subscriptions/' . $subscription_id . '/cancel',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Activates a subscription in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#subscriptions_activate
	 *
	 * @since 1.5.0
	 *
	 * @param string $subscription_id The PayPal subscription ID.
	 * @param bool   $is_sandbox      Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return bool
	 */
	public function activate_subscription( string $subscription_id, bool $is_sandbox = false ): bool {
		$body = [
			'reason' => MS_Gateway_Paypalexpress_Helpers::trim_text( __( 'Activating the subscription', 'memberdash' ), 128 ),
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5( $subscription_id )
			),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/billing/subscriptions/' . $subscription_id . '/activate',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Captures a subscription in the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/subscriptions/v1/#subscriptions_capture
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data       The data to use.
	 * @param bool                $is_sandbox Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return bool
	 */
	public function capture_subscription( array $data, bool $is_sandbox = false ): bool {
		$body = [
			'note'         => MS_Gateway_Paypalexpress_Helpers::trim_text( __( 'Capturing the subscription', 'memberdash' ), 128 ),
			'capture_type' => 'OUTSTANDING_BALANCE',
			'amount'       => [
				'currency_code' => $data['currency_code'],
				'value'         => $data['amount'],
			],
		];

		$args = [
			'headers' => $this->get_request_headers(
				md5(
					MS_Helper_Cast::to_string( $data['subscription_id'] )
				)
			),
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/billing/subscriptions/' . $data['subscription_id'] . '/capture',
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Deletes all membership meta data associated with the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function delete_all_membership_meta_data(): void {
		global $wpdb;

		$wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE 'ms_gateway_paypal_sandbox_%'" );
		$wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE 'ms_gateway_paypal_live_%'" );
	}

	/**
	 * Fetches the seller status from the PayPal API.
	 *
	 * @see https://developer.paypal.com/docs/api/partner-referrals/v1/#merchant-integration_status
	 *
	 * @since 1.5.0
	 *
	 * @param string $seller_merchant_id The seller merchant ID.
	 * @param bool   $is_sandbox         Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function get_seller_status( string $seller_merchant_id, bool $is_sandbox = false ): array {
		$args = [
			'headers' => $this->get_request_headers( $seller_merchant_id ),
			'body'    => [],
		];

		$partner_id = 'ZSSERF3Z4TG3L';
		if ( ! $is_sandbox ) {
			$partner_id = 'YAB5R8P3E4PYA';
		}

		$response = $this->client_get(
			'v1/customer/partners/' . rawurlencode( $partner_id ) . '/merchant-integrations/' . rawurlencode( $seller_merchant_id ),
			[],
			$args,
			$is_sandbox
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}
}
